<?php

namespace Tests\Feature;

use App\Models\User;
use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Auth;

class TemplateRenderingTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    /**
     * Test that app layout renders properly with authenticated user having zero values.
     */
    public function test_app_layout_handles_zero_values(): void
    {
        // Create a user with zero values that could potentially cause issues
        $user = User::factory()->create([
            'xp' => 0,
            'wallet_balance' => 0,
        ]);

        // Authenticate the user
        Auth::login($user);

        // Try to render a view that uses the app layout
        $response = $this->get('/');

        // Should not get a 500 error due to number_format type errors
        $response->assertStatus(200);
    }

    /**
     * Test that app layout renders properly with authenticated user having valid values.
     */
    public function test_app_layout_handles_valid_values(): void
    {
        // Create a user with valid values
        $user = User::factory()->create([
            'xp' => 1500,
            'wallet_balance' => 250.75,
        ]);

        // Authenticate the user
        Auth::login($user);

        // Try to render a view that uses the app layout
        $response = $this->get('/');

        // Should render successfully
        $response->assertStatus(200);
    }

    /**
     * Test that number_format functions work correctly with our fixes
     */
    public function test_number_format_with_null_coalescing(): void
    {
        $user = User::factory()->create([
            'xp' => 1500,
            'wallet_balance' => 250.75,
        ]);

        // Test our null coalescing operators work
        $xp = $user->xp ?? 0;
        $wallet = $user->wallet_balance ?? 0;
        
        $this->assertEquals(1500, $xp);
        $this->assertEquals(250.75, $wallet);
        
        // Test number_format doesn't throw errors
        $formattedXp = number_format($xp);
        $formattedWallet = number_format($wallet, 2);
        
        $this->assertEquals('1,500', $formattedXp);
        $this->assertEquals('250.75', $formattedWallet);
    }
}