<?php

namespace Tests\Feature;

use App\Models\Game;
use App\Models\Tournament;
use App\Models\TournamentInvite;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class TournamentInviteTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test games
        Game::factory()->create([
            'name' => 'FIFA 24',
            'slug' => 'fifa-24',
            'platform' => 'console'
        ]);
    }

    /** @test */
    public function test_tournament_can_be_created_with_knockout_type()
    {
        $host = User::factory()->create([
            'level' => 3, // Level 3 can host free tournaments
            'xp' => 500
        ]);
        
        $game = Game::first();

        $tournamentData = [
            'game_id' => $game->id,
            'type' => 'knockout',
            'title' => 'Test Tournament',
            'description' => 'A test tournament',
            'entry_fee' => 0,
            'starts_at' => now()->addHour()->format('Y-m-d\TH:i'),
        ];

        $response = $this->actingAs($host)
            ->post(route('tournaments.store'), $tournamentData);

        $response->assertRedirect();
        $this->assertDatabaseHas('tournaments', [
            'title' => 'Test Tournament',
            'type' => 'knockout',
            'host_id' => $host->id
        ]);
    }

    /** @test */
    public function test_tournament_can_invite_users()
    {
        $host = User::factory()->create(['level' => 3, 'xp' => 500]);
        $invitee = User::factory()->create(['level' => 2, 'xp' => 200]);
        $game = Game::first();

        $tournament = Tournament::create([
            'host_id' => $host->id,
            'game_id' => $game->id,
            'type' => 'knockout',
            'title' => 'Test Tournament',
            'entry_fee' => 0,
            'starts_at' => now()->addHour(),
        ]);

        $invite = $tournament->inviteUser($invitee, $host, 'Join my tournament!');

        $this->assertNotNull($invite);
        $this->assertEquals('pending', $invite->status);
        $this->assertEquals('Join my tournament!', $invite->message);
        $this->assertEquals($invitee->id, $invite->invitee_id);
        $this->assertEquals($host->id, $invite->inviter_id);
    }

    /** @test */
    public function test_user_cannot_be_invited_twice_to_same_tournament()
    {
        $host = User::factory()->create(['level' => 3, 'xp' => 500]);
        $invitee = User::factory()->create(['level' => 2, 'xp' => 200]);
        $game = Game::first();

        $tournament = Tournament::create([
            'host_id' => $host->id,
            'game_id' => $game->id,
            'type' => 'knockout',
            'title' => 'Test Tournament',
            'entry_fee' => 0,
            'starts_at' => now()->addHour(),
        ]);

        // First invite should succeed
        $invite1 = $tournament->inviteUser($invitee, $host);
        $this->assertNotNull($invite1);

        // Second invite should fail
        $invite2 = $tournament->inviteUser($invitee, $host);
        $this->assertNull($invite2);
    }

    /** @test */
    public function test_tournament_bracket_generation_requires_power_of_2()
    {
        $host = User::factory()->create(['level' => 3, 'xp' => 500]);
        $game = Game::first();

        $tournament = Tournament::create([
            'host_id' => $host->id,
            'game_id' => $game->id,
            'type' => 'knockout',
            'title' => 'Test Tournament',
            'entry_fee' => 0,
            'starts_at' => now()->addHour(),
        ]);

        // Add 4 participants (power of 2)
        $participants = User::factory()->count(4)->create();
        foreach ($participants as $participant) {
            $tournament->participants()->create([
                'user_id' => $participant->id,
            ]);
        }

        $this->assertTrue($tournament->canStart());

        // Generate bracket
        $bracket = $tournament->generateBracket();
        $this->assertIsArray($bracket);
        $this->assertEquals(2, $bracket['rounds']); // 4 participants = 2 rounds
        $this->assertEquals(4, $bracket['participant_count']);
    }

    /** @test */
    public function test_tournament_invite_can_be_accepted()
    {
        $host = User::factory()->create(['level' => 3, 'xp' => 500]);
        $invitee = User::factory()->create(['level' => 2, 'xp' => 200, 'wallet_balance' => 100]);
        $game = Game::first();

        $tournament = Tournament::create([
            'host_id' => $host->id,
            'game_id' => $game->id,
            'type' => 'knockout',
            'title' => 'Test Tournament',
            'entry_fee' => 0,
            'starts_at' => now()->addHour(),
        ]);

        $invite = $tournament->inviteUser($invitee, $host);

        $response = $this->actingAs($invitee)
            ->post(route('tournaments.invite.accept', $invite));

        $response->assertRedirect();
        
        $invite->refresh();
        $this->assertEquals('accepted', $invite->status);
        
        // Check that user is now a participant
        $this->assertTrue($tournament->users()->where('user_id', $invitee->id)->exists());
    }

    /** @test */
    public function test_tournament_invite_can_be_declined()
    {
        $host = User::factory()->create(['level' => 3, 'xp' => 500]);
        $invitee = User::factory()->create(['level' => 2, 'xp' => 200]);
        $game = Game::first();

        $tournament = Tournament::create([
            'host_id' => $host->id,
            'game_id' => $game->id,
            'type' => 'knockout',
            'title' => 'Test Tournament',
            'entry_fee' => 0,
            'starts_at' => now()->addHour(),
        ]);

        $invite = $tournament->inviteUser($invitee, $host);

        $response = $this->actingAs($invitee)
            ->post(route('tournaments.invite.decline', $invite));

        $response->assertRedirect();
        
        $invite->refresh();
        $this->assertEquals('declined', $invite->status);
        
        // Check that user is NOT a participant
        $this->assertFalse($tournament->users()->where('user_id', $invitee->id)->exists());
    }
}
