<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Services\LevelService;
use App\Models\Level;
use Illuminate\Foundation\Testing\RefreshDatabase;

class LevelServiceTest extends TestCase
{
    use RefreshDatabase;

    protected LevelService $levelService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->levelService = new LevelService();
    }

    public function test_get_level_title_returns_correct_titles()
    {
        $this->assertEquals('Beginner', $this->levelService->getLevelTitle(1));
        $this->assertEquals('Novice', $this->levelService->getLevelTitle(3));
        $this->assertEquals('Intermediate', $this->levelService->getLevelTitle(8));
        $this->assertEquals('Advanced', $this->levelService->getLevelTitle(12));
        $this->assertEquals('Expert', $this->levelService->getLevelTitle(18));
        $this->assertEquals('Master', $this->levelService->getLevelTitle(23));
        $this->assertEquals('Elite', $this->levelService->getLevelTitle(28));
        $this->assertEquals('Legend', $this->levelService->getLevelTitle(30));
    }

    public function test_get_level_display_name_for_creator()
    {
        $level = new Level(['name' => 'Creator']);
        $displayName = $this->levelService->getLevelDisplayName($level);
        
        $this->assertEquals('Creator - Content & Tournament Host', $displayName);
    }

    public function test_get_level_display_name_for_numeric_level()
    {
        $level = new Level(['name' => 'Level 5']);
        $displayName = $this->levelService->getLevelDisplayName($level);
        
        $this->assertEquals('Level 5 - Novice', $displayName);
    }

    public function test_get_level_display_name_with_limits()
    {
        $level = new Level([
            'name' => 'Level 10',
            'max_wager_amount' => 50000
        ]);
        
        $displayName = $this->levelService->getLevelDisplayNameWithLimits($level);
        
        $this->assertEquals('Level 10 - Intermediate (₦50,000 limit)', $displayName);
    }

    public function test_get_level_display_name_with_no_limits()
    {
        $level = new Level([
            'name' => 'Level 30',
            'max_wager_amount' => null
        ]);
        
        $displayName = $this->levelService->getLevelDisplayNameWithLimits($level);
        
        $this->assertEquals('Level 30 - Legend (No limit)', $displayName);
    }

    public function test_extract_level_number()
    {
        $this->assertEquals(5, $this->levelService->extractLevelNumber('Level 5'));
        $this->assertEquals(15, $this->levelService->extractLevelNumber('Level 15'));
        $this->assertEquals(30, $this->levelService->extractLevelNumber('Level 30'));
    }
}