<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\CronLog;
use Illuminate\Support\Facades\DB;

class CleanupNotificationsCommand extends Command
{
    protected $signature = 'fintech:cleanup-notifications';
    protected $description = 'Clean up old notifications and logs';

    public function handle()
    {
        $startTime = microtime(true);
        $startMemory = memory_get_usage(true);
        
        try {
            $this->info('Starting cleanup process...');
            
            $cleanedItems = 0;
            
            // Clean up old notifications (older than 90 days)
            $deletedNotifications = DB::table('notifications')
                ->where('created_at', '<', now()->subDays(90))
                ->delete();
            
            $cleanedItems += $deletedNotifications;
            $this->info("Cleaned up {$deletedNotifications} old notifications");
            
            // Clean up old cron logs (older than 30 days)
            $deletedCronLogs = CronLog::where('executed_at', '<', now()->subDays(30))->delete();
            
            $cleanedItems += $deletedCronLogs;
            $this->info("Cleaned up {$deletedCronLogs} old cron logs");
            
            // Clean up old sessions (older than 7 days)
            $deletedSessions = DB::table('sessions')
                ->where('last_activity', '<', now()->subDays(7)->timestamp)
                ->delete();
            
            $cleanedItems += $deletedSessions;
            $this->info("Cleaned up {$deletedSessions} old sessions");
            
            // Clean up old password reset tokens (older than 1 day)
            $deletedPasswordResets = DB::table('pwd_reset_tokens')
                ->where('created_at', '<', now()->subDay())
                ->delete();
            
            $cleanedItems += $deletedPasswordResets;
            $this->info("Cleaned up {$deletedPasswordResets} old password reset tokens");
            
            // Clean up old audit logs (older than 1 year)
            $deletedAuditLogs = DB::table('audit_logs')
                ->where('created_at', '<', now()->subYear())
                ->delete();
            
            $cleanedItems += $deletedAuditLogs;
            $this->info("Cleaned up {$deletedAuditLogs} old audit logs");
            
            // Optimize database tables
            $this->optimizeTables();
            
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $message = "Cleaned up {$cleanedItems} items";
            $this->info($message);
            
            CronLog::create([
                'task_name' => 'cleanup_notifications',
                'status' => 'success',
                'message' => $message,
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage
            ]);
            
        } catch (\Exception $e) {
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $this->error('Cleanup process failed: ' . $e->getMessage());
            
            CronLog::create([
                'task_name' => 'cleanup_notifications',
                'status' => 'failed',
                'message' => 'Cleanup process failed',
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage,
                'error_details' => [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]
            ]);
        }
    }
    
    private function optimizeTables()
    {
        $tables = [
            'users',
            'transactions',
            'wallets',
            'virtual_cards',
            'virtual_accounts',
            'notifications',
            'cron_logs',
            'audit_logs'
        ];
        
        foreach ($tables as $table) {
            try {
                DB::statement("OPTIMIZE TABLE {$table}");
                $this->info("Optimized table: {$table}");
            } catch (\Exception $e) {
                $this->warn("Failed to optimize table {$table}: " . $e->getMessage());
            }
        }
    }
}