<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\VirtualCard;
use App\Models\CronLog;
use App\Services\AnchorService;

class SyncVirtualCardsCommand extends Command
{
    protected $signature = 'fintech:sync-virtual-cards';
    protected $description = 'Update virtual card status and balances';

    protected $anchorService;

    public function __construct(AnchorService $anchorService)
    {
        parent::__construct();
        $this->anchorService = $anchorService;
    }

    public function handle()
    {
        $startTime = microtime(true);
        $startMemory = memory_get_usage(true);
        
        try {
            $this->info('Starting virtual card synchronization...');
            
            $syncedCount = 0;
            $errorCount = 0;
            
            // Get all active virtual cards
            $cards = VirtualCard::whereIn('status', ['active', 'frozen'])
                ->where('updated_at', '<', now()->subMinutes(10))
                ->get();
            
            foreach ($cards as $card) {
                try {
                    $this->syncCard($card);
                    $syncedCount++;
                    
                } catch (\Exception $e) {
                    $this->error("Error syncing card {$card->id}: " . $e->getMessage());
                    $errorCount++;
                }
            }
            
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $message = "Synced {$syncedCount} cards, {$errorCount} errors";
            $this->info($message);
            
            CronLog::create([
                'task_name' => 'sync_virtual_cards',
                'status' => $errorCount > 0 ? 'warning' : 'success',
                'message' => $message,
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage
            ]);
            
        } catch (\Exception $e) {
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $this->error('Virtual card sync failed: ' . $e->getMessage());
            
            CronLog::create([
                'task_name' => 'sync_virtual_cards',
                'status' => 'failed',
                'message' => 'Virtual card sync failed',
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage,
                'error_details' => [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]
            ]);
        }
    }
    
    private function syncCard(VirtualCard $card)
    {
        $response = $this->anchorService->getCardDetails($card->provider_reference);
        
        if ($response['success']) {
            $cardData = $response['data'];
            
            // Update card status if changed
            if ($cardData['status'] !== $card->status) {
                $oldStatus = $card->status;
                $card->update(['status' => $cardData['status']]);
                
                $this->info("Updated card {$card->id} status from {$oldStatus} to {$cardData['status']}");
            }
            
            // Update balance if changed
            if (isset($cardData['balance']) && $cardData['balance'] !== $card->balance) {
                $oldBalance = $card->balance;
                $card->update(['balance' => $cardData['balance']]);
                
                $this->info("Updated card {$card->id} balance from ₦{$oldBalance} to ₦{$cardData['balance']}");
            }
            
            // Sync recent transactions
            $this->syncCardTransactions($card, $cardData['transactions'] ?? []);
            
            // Update last sync time
            $card->touch();
        }
    }
    
    private function syncCardTransactions(VirtualCard $card, array $transactions)
    {
        foreach ($transactions as $txnData) {
            // Check if transaction already exists
            if (!$this->isCardTransactionProcessed($txnData['reference'])) {
                $this->processCardTransaction($card, $txnData);
            }
        }
    }
    
    private function isCardTransactionProcessed($reference)
    {
        return \App\Models\Transaction::where('external_reference', $reference)->exists();
    }
    
    private function processCardTransaction(VirtualCard $card, $transactionData)
    {
        // Create transaction record
        $transaction = \App\Models\Transaction::create([
            'user_id' => $card->user_id,
            'wallet_id' => $card->user->wallet->id,
            'type' => $transactionData['type'], // credit or debit
            'category' => 'virtual_card_transaction',
            'amount' => $transactionData['amount'],
            'fee' => $transactionData['fee'] ?? 0,
            'total_amount' => $transactionData['amount'] + ($transactionData['fee'] ?? 0),
            'currency' => $transactionData['currency'] ?? 'NGN',
            'status' => 'completed',
            'description' => $transactionData['description'] ?? 'Virtual card transaction',
            'external_reference' => $transactionData['reference'],
            'metadata' => [
                'virtual_card_id' => $card->id,
                'card_last_four' => substr($card->card_number, -4),
                'merchant' => $transactionData['merchant'] ?? 'Unknown',
                'location' => $transactionData['location'] ?? 'Unknown'
            ]
        ]);
        
        $this->info("Processed card transaction: ₦{$transactionData['amount']} for card {$card->id}");
    }
}