<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Models\VirtualCard;
use App\Services\AnchorService;

/**
 * Virtual Card API Controller
 * 
 * TODO: Add comprehensive input validation
 * TODO: Add card spending limits and controls
 * TODO: Add real-time transaction monitoring
 * TODO: Add card security features (freeze/unfreeze)
 * TODO: Add fraud detection for card transactions
 */
class VirtualCardController extends Controller
{
    protected $anchorService;

    public function __construct(AnchorService $anchorService)
    {
        $this->anchorService = $anchorService;
    }

    /**
     * Get user's virtual cards
     * 
     * TODO: Add pagination for large card lists
     * TODO: Add filtering by card status
     * TODO: Add card spending analytics
     */
    public function getUserCards(Request $request)
    {
        try {
            $user = $request->user();
            
            $cards = VirtualCard::where('user_id', $user->id)
                ->orderBy('created_at', 'desc')
                ->get();

            // TODO: Mask sensitive card details in response
            foreach ($cards as $card) {
                $card->makeHidden(['cvv', 'expiry_month', 'expiry_year']);
                $card->card_number = $this->maskCardNumber($card->card_number);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Virtual cards retrieved successfully',
                'data' => $cards
            ]);
        } catch (\Exception $e) {
            // TODO: Add proper error logging
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve virtual cards',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Create new virtual card
     * 
     * TODO: Add card type validation (debit, credit, prepaid)
     * TODO: Add spending limit configuration
     * TODO: Add card issuance fees calculation
     * TODO: Add compliance checks for card creation
     */
    public function create(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'card_type' => 'required|string|in:debit,prepaid',
                'currency' => 'required|string|in:NGN,USD',
                'spending_limit' => 'nullable|numeric|min:1000|max:1000000',
                'pin' => 'required|string|size:4|regex:/^[0-9]+$/',
                // TODO: Add more validation rules
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();

            // TODO: Check user's card creation limits
            // TODO: Verify user's wallet balance for card funding
            // TODO: Add fraud detection checks

            // Create virtual card through Anchor service
            $cardData = [
                'user_id' => $user->id,
                'card_type' => $request->card_type,
                'currency' => $request->currency,
                'spending_limit' => $request->spending_limit ?? 100000,
                'email' => $user->email,
                'phone' => $user->phone,
                'name' => $user->first_name . ' ' . $user->last_name
            ];

            $response = $this->anchorService->createVirtualCard($cardData);

            if (!$response['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to create virtual card',
                    'error' => $response['message']
                ], 500);
            }

            // Save card to database
            $virtualCard = VirtualCard::create([
                'user_id' => $user->id,
                'card_number' => $response['data']['card_number'],
                'card_name' => $response['data']['card_name'],
                'cvv' => $response['data']['cvv'],
                'expiry_month' => $response['data']['expiry_month'],
                'expiry_year' => $response['data']['expiry_year'],
                'card_type' => $request->card_type,
                'currency' => $request->currency,
                'spending_limit' => $request->spending_limit ?? 100000,
                'balance' => 0,
                'status' => 'active',
                'provider' => 'anchor',
                'provider_reference' => $response['data']['card_id']
            ]);

            // TODO: Hash and store PIN securely
            // TODO: Send card creation notification
            // TODO: Log card creation event

            // Mask sensitive data in response
            $virtualCard->makeHidden(['cvv', 'expiry_month', 'expiry_year']);
            $virtualCard->card_number = $this->maskCardNumber($virtualCard->card_number);

            return response()->json([
                'status' => 'success',
                'message' => 'Virtual card created successfully',
                'data' => $virtualCard
            ], 201);

        } catch (\Exception $e) {
            // TODO: Add proper error logging
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create virtual card',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get virtual card details
     * 
     * TODO: Add transaction history
     * TODO: Add spending analytics
     */
    public function show(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Virtual card not found'
                ], 404);
            }

            // Mask sensitive data
            $card->makeHidden(['cvv', 'expiry_month', 'expiry_year']);
            $card->card_number = $this->maskCardNumber($card->card_number);

            return response()->json([
                'status' => 'success',
                'message' => 'Virtual card details retrieved successfully',
                'data' => $card
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve card details',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Freeze virtual card
     * 
     * TODO: Add reason tracking for card freeze
     * TODO: Add automatic unfreeze scheduling
     */
    public function freeze(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Virtual card not found'
                ], 404);
            }

            if ($card->status === 'frozen') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Card is already frozen'
                ], 400);
            }

            // Freeze card through provider
            $response = $this->anchorService->freezeCard($card->provider_reference);

            if (!$response['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to freeze card',
                    'error' => $response['message']
                ], 500);
            }

            $card->update(['status' => 'frozen']);

            // TODO: Log card freeze event
            // TODO: Send freeze notification

            return response()->json([
                'status' => 'success',
                'message' => 'Virtual card frozen successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to freeze card',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Unfreeze virtual card
     */
    public function unfreeze(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Virtual card not found'
                ], 404);
            }

            if ($card->status !== 'frozen') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Card is not frozen'
                ], 400);
            }

            // Unfreeze card through provider
            $response = $this->anchorService->unfreezeCard($card->provider_reference);

            if (!$response['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to unfreeze card',
                    'error' => $response['message']
                ], 500);
            }

            $card->update(['status' => 'active']);

            // TODO: Log card unfreeze event
            // TODO: Send unfreeze notification

            return response()->json([
                'status' => 'success',
                'message' => 'Virtual card unfrozen successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to unfreeze card',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Terminate virtual card
     * 
     * TODO: Add confirmation requirement for termination
     * TODO: Add balance refund process
     */
    public function terminate(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Virtual card not found'
                ], 404);
            }

            if ($card->status === 'terminated') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Card is already terminated'
                ], 400);
            }

            // TODO: Check for pending transactions
            // TODO: Process balance refund

            // Terminate card through provider
            $response = $this->anchorService->terminateCard($card->provider_reference);

            if (!$response['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to terminate card',
                    'error' => $response['message']
                ], 500);
            }

            $card->update(['status' => 'terminated']);

            // TODO: Log card termination event
            // TODO: Send termination notification

            return response()->json([
                'status' => 'success',
                'message' => 'Virtual card terminated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to terminate card',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get card transactions
     * 
     * TODO: Add transaction filtering and search
     * TODO: Add transaction analytics
     */
    public function getTransactions(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Virtual card not found'
                ], 404);
            }

            // TODO: Implement transaction retrieval from provider
            // TODO: Add local transaction caching

            return response()->json([
                'status' => 'success',
                'message' => 'Card transactions retrieved successfully',
                'data' => []
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve card transactions',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mask card number for security
     */
    private function maskCardNumber($cardNumber)
    {
        $cardNumber = str_replace(' ', '', $cardNumber);
        return substr($cardNumber, 0, 4) . ' **** **** ' . substr($cardNumber, -4);
    }
}