<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AirtimeMargin extends Model
{
    use HasFactory;

    protected $fillable = [
        'network',
        'service_type',
        'margin_percentage',
        'fixed_fee',
        'min_margin_amount',
        'max_margin_amount',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'margin_percentage' => 'decimal:2',
            'fixed_fee' => 'decimal:2',
            'min_margin_amount' => 'decimal:2',
            'max_margin_amount' => 'decimal:2',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Scope to get active margins.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to get margins by network.
     */
    public function scopeByNetwork($query, $network)
    {
        return $query->where('network', $network);
    }

    /**
     * Scope to get margins by service type.
     */
    public function scopeByServiceType($query, $serviceType)
    {
        return $query->where('service_type', $serviceType);
    }

    /**
     * Calculate margin amount for a given transaction amount.
     */
    public function calculateMargin($transactionAmount): float
    {
        $percentageMargin = ($transactionAmount * $this->margin_percentage) / 100;
        $totalMargin = $percentageMargin + $this->fixed_fee;
        
        // Apply minimum margin
        if ($totalMargin < $this->min_margin_amount) {
            $totalMargin = $this->min_margin_amount;
        }
        
        // Apply maximum margin if set
        if ($this->max_margin_amount && $totalMargin > $this->max_margin_amount) {
            $totalMargin = $this->max_margin_amount;
        }
        
        return round($totalMargin, 2);
    }

    /**
     * Get margin for a specific network and service type.
     */
    public static function getMargin($network, $serviceType): ?self
    {
        return self::active()
            ->byNetwork($network)
            ->byServiceType($serviceType)
            ->first();
    }

    /**
     * Get all margins grouped by network.
     */
    public static function getAllMarginsGrouped(): array
    {
        $margins = self::active()->get();
        $grouped = [];
        
        foreach ($margins as $margin) {
            $grouped[$margin->network][$margin->service_type] = $margin;
        }
        
        return $grouped;
    }

    /**
     * Get margin for a specific transaction and calculate final price.
     */
    public static function calculateFinalPrice($network, $serviceType, $baseAmount): array
    {
        $margin = self::getMargin($network, $serviceType);
        
        if (!$margin) {
            return [
                'base_amount' => $baseAmount,
                'margin_amount' => 0,
                'final_amount' => $baseAmount,
                'margin_percentage' => 0,
                'has_margin' => false
            ];
        }

        $marginAmount = $margin->calculateMargin($baseAmount);
        $finalAmount = $baseAmount + $marginAmount;

        return [
            'base_amount' => $baseAmount,
            'margin_amount' => $marginAmount,
            'final_amount' => $finalAmount,
            'margin_percentage' => $margin->margin_percentage,
            'has_margin' => true,
            'margin_model' => $margin
        ];
    }

    /**
     * Get default margins for initialization.
     */
    public static function getDefaultMargins(): array
    {
        return [
            'airtime' => [
                'mtn' => 2.5,
                'glo' => 2.5,
                'airtel' => 2.5,
                '9mobile' => 2.5,
            ],
            'data' => [
                'mtn' => 3.0,
                'glo' => 3.5,
                'airtel' => 3.0,
                '9mobile' => 3.5,
            ]
        ];
    }
}
