<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Support\Facades\Crypt;

class ApiConfig extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'category',
        'provider_code',
        'description',
        'base_url',
        'test_url',
        'live_url',
        'api_key',
        'secret_key',
        'public_key',
        'private_key',
        'merchant_id',
        'client_id',
        'client_secret',
        'config',
        'endpoints',
        'webhook_config',
        'is_active',
        'is_live_mode',
        'last_tested_at',
        'last_test_status',
        'last_test_response',
        'webhook_url',
        'callback_url',
        'return_url',
    ];

    protected function casts(): array
    {
        return [
            'config' => 'array',
            'endpoints' => 'array',
            'webhook_config' => 'array',
            'is_active' => 'boolean',
            'is_live_mode' => 'boolean',
            'last_tested_at' => 'datetime',
        ];
    }

    /**
     * Encrypt sensitive fields when storing
     */
    protected function apiKey(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => $value ? Crypt::decryptString($value) : null,
            set: fn ($value) => $value ? Crypt::encryptString($value) : null,
        );
    }

    protected function secretKey(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => $value ? Crypt::decryptString($value) : null,
            set: fn ($value) => $value ? Crypt::encryptString($value) : null,
        );
    }

    protected function publicKey(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => $value ? Crypt::decryptString($value) : null,
            set: fn ($value) => $value ? Crypt::encryptString($value) : null,
        );
    }

    protected function privateKey(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => $value ? Crypt::decryptString($value) : null,
            set: fn ($value) => $value ? Crypt::encryptString($value) : null,
        );
    }

    protected function clientSecret(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => $value ? Crypt::decryptString($value) : null,
            set: fn ($value) => $value ? Crypt::encryptString($value) : null,
        );
    }

    /**
     * Scope a query to only include active configs.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to filter by category.
     */
    public function scopeCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Scope a query to filter by live mode.
     */
    public function scopeLiveMode($query, $live = true)
    {
        return $query->where('is_live_mode', $live);
    }

    /**
     * Get the effective base URL based on mode.
     */
    public function getEffectiveBaseUrlAttribute(): string
    {
        if ($this->is_live_mode && $this->live_url) {
            return $this->live_url;
        }
        
        if (!$this->is_live_mode && $this->test_url) {
            return $this->test_url;
        }
        
        return $this->base_url;
    }

    /**
     * Get masked API key for display.
     */
    public function getMaskedApiKeyAttribute(): string
    {
        if (!$this->api_key) {
            return 'Not configured';
        }
        
        $key = $this->api_key;
        if (strlen($key) <= 8) {
            return str_repeat('•', strlen($key));
        }
        
        return substr($key, 0, 4) . str_repeat('•', strlen($key) - 8) . substr($key, -4);
    }

    /**
     * Get the webhook URL for this provider.
     */
    public function getWebhookUrlAttribute(): string
    {
        return url("/api/webhooks/api/{$this->slug}");
    }

    /**
     * Get the callback URL for this provider.
     */
    public function getCallbackUrlAttribute(): string
    {
        return url("/api/callbacks/{$this->slug}");
    }

    /**
     * Check if this config is properly configured.
     */
    public function isConfigured(): bool
    {
        return !empty($this->api_key) || !empty($this->client_id);
    }

    /**
     * Get health status badge class.
     */
    public function getHealthStatusClass(): string
    {
        switch ($this->last_test_status) {
            case 'success':
                return 'bg-emerald-100 text-emerald-800';
            case 'failed':
                return 'bg-red-100 text-red-800';
            case 'pending':
                return 'bg-amber-100 text-amber-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    }

    /**
     * Get health status text.
     */
    public function getHealthStatusText(): string
    {
        switch ($this->last_test_status) {
            case 'success':
                return 'Healthy';
            case 'failed':
                return 'Unhealthy';
            case 'pending':
                return 'Testing...';
            default:
                return 'Unknown';
        }
    }
}
