<?php

namespace App\Services;

use Illuminate\Support\Facades\Config;

/**
 * Funding Source Management Service
 * 
 * Centralizes funding source constants and validation logic
 */
class FundingSourceService
{
    /**
     * Funding source constants
     */
    public const MANUAL = 'manual';
    public const MONNIFY = 'monnify';
    public const ANCHOR = 'anchor';

    /**
     * Get all available funding sources
     * 
     * @return array
     */
    public static function getAvailableSources(): array
    {
        return Config::get('fintech.funding_sources', []);
    }

    /**
     * Get enabled funding sources only
     * 
     * @return array
     */
    public static function getEnabledSources(): array
    {
        $sources = static::getAvailableSources();
        
        return array_filter($sources, function ($source) {
            return $source['enabled'] ?? false;
        });
    }

    /**
     * Get funding source keys (for validation)
     * 
     * @return array
     */
    public static function getSourceKeys(): array
    {
        return array_keys(static::getAvailableSources());
    }

    /**
     * Get enabled funding source keys (for validation)
     * 
     * @return array
     */
    public static function getEnabledSourceKeys(): array
    {
        return array_keys(static::getEnabledSources());
    }

    /**
     * Check if a funding source is valid
     * 
     * @param string $source
     * @return bool
     */
    public static function isValidSource(string $source): bool
    {
        return in_array($source, static::getSourceKeys());
    }

    /**
     * Check if a funding source is enabled
     * 
     * @param string $source
     * @return bool
     */
    public static function isEnabledSource(string $source): bool
    {
        return in_array($source, static::getEnabledSourceKeys());
    }

    /**
     * Get funding source configuration
     * 
     * @param string $source
     * @return array|null
     */
    public static function getSourceConfig(string $source): ?array
    {
        $sources = static::getAvailableSources();
        
        return $sources[$source] ?? null;
    }

    /**
     * Get funding source display name
     * 
     * @param string $source
     * @return string
     */
    public static function getSourceName(string $source): string
    {
        $config = static::getSourceConfig($source);
        
        return $config['name'] ?? ucfirst($source);
    }

    /**
     * Get funding source description
     * 
     * @param string $source
     * @return string
     */
    public static function getSourceDescription(string $source): string
    {
        $config = static::getSourceConfig($source);
        
        return $config['description'] ?? '';
    }

    /**
     * Check if source requires approval
     * 
     * @param string $source
     * @return bool
     */
    public static function requiresApproval(string $source): bool
    {
        $config = static::getSourceConfig($source);
        
        return $config['requires_approval'] ?? false;
    }

    /**
     * Get source timeout setting
     * 
     * @param string $source
     * @return int
     */
    public static function getSourceTimeout(string $source): int
    {
        $config = static::getSourceConfig($source);
        
        return $config['timeout'] ?? Config::get('fintech.api.timeout', 30);
    }

    /**
     * Validate funding source for transaction
     * 
     * @param string $source
     * @param array $additionalRules
     * @return array
     */
    public static function getValidationRules(string $source, array $additionalRules = []): array
    {
        $baseRules = [
            'funding_source' => [
                'nullable',
                'string',
                'in:' . implode(',', static::getEnabledSourceKeys())
            ]
        ];

        return array_merge($baseRules, $additionalRules);
    }

    /**
     * Get all funding sources formatted for select options
     * 
     * @param bool $enabledOnly
     * @return array
     */
    public static function getSelectOptions(bool $enabledOnly = true): array
    {
        $sources = $enabledOnly ? static::getEnabledSources() : static::getAvailableSources();
        $options = [];

        foreach ($sources as $key => $config) {
            $options[$key] = $config['name'];
        }

        return $options;
    }

    /**
     * Format funding source for API response
     * 
     * @param string $source
     * @return array
     */
    public static function formatForApi(string $source): array
    {
        $config = static::getSourceConfig($source);
        
        if (!$config) {
            return [
                'code' => $source,
                'name' => ucfirst($source),
                'enabled' => false,
                'available' => false
            ];
        }

        return [
            'code' => $source,
            'name' => $config['name'],
            'description' => $config['description'] ?? '',
            'enabled' => $config['enabled'] ?? false,
            'requires_approval' => $config['requires_approval'] ?? false,
            'available' => $config['enabled'] ?? false
        ];
    }
}