<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

/**
 * NIN Service for National Identification Number validation
 * 
 * TODO: Add comprehensive error handling
 * TODO: Add data privacy compliance
 * TODO: Add retry mechanisms
 * TODO: Add caching for verified NINs
 */
class NinService
{
    protected $baseUrl;
    protected $apiKey;
    protected $provider;

    public function __construct()
    {
        $this->baseUrl = config('services.nin.base_url', 'https://api.verified.africa');
        $this->apiKey = config('services.nin.api_key');
        $this->provider = config('services.nin.provider', 'verified');
    }

    /**
     * Verify NIN with personal details
     * 
     * TODO: Add data masking for responses
     * TODO: Add compliance audit logging
     * TODO: Add rate limiting
     */
    public function verifyNin(array $data)
    {
        try {
            $payload = [
                'nin' => $data['nin'],
                'first_name' => $data['first_name'],
                'last_name' => $data['last_name'],
                'date_of_birth' => $data['date_of_birth'],
                'phone' => $data['phone'] ?? null
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json'
            ])->post($this->baseUrl . '/v3/kyc/nin', $payload);

            if ($response->successful()) {
                $responseData = $response->json();
                
                if ($responseData['status'] === 'success' && $responseData['data']['is_verified']) {
                    return [
                        'success' => true,
                        'data' => [
                            'nin' => $data['nin'],
                            'first_name' => $responseData['data']['first_name'],
                            'last_name' => $responseData['data']['last_name'],
                            'middle_name' => $responseData['data']['middle_name'] ?? null,
                            'date_of_birth' => $responseData['data']['date_of_birth'],
                            'phone' => $responseData['data']['phone'],
                            'gender' => $responseData['data']['gender'] ?? null,
                            'state_of_origin' => $responseData['data']['state_of_origin'] ?? null,
                            'lga_of_origin' => $responseData['data']['lga_of_origin'] ?? null,
                            'verified_at' => now(),
                            'match_score' => $responseData['data']['match_score'] ?? 100
                        ]
                    ];
                }

                return [
                    'success' => false,
                    'message' => 'NIN verification failed - details do not match',
                    'errors' => $responseData['data']['mismatches'] ?? []
                ];
            }

            // TODO: Handle specific error codes
            $errorMessage = $response->json()['message'] ?? 'NIN verification service unavailable';
            
            return [
                'success' => false,
                'message' => $errorMessage,
                'error_code' => $response->status()
            ];

        } catch (Exception $e) {
            Log::error('NIN verification failed', [
                'error' => $e->getMessage(),
                'nin' => substr($data['nin'], 0, 3) . '********' // Mask NIN in logs
            ]);

            return [
                'success' => false,
                'message' => 'NIN verification service temporarily unavailable'
            ];
        }
    }

    /**
     * Get NIN details without verification
     * 
     * TODO: Add permission checks
     * TODO: Add audit logging
     */
    public function getNinDetails(string $nin)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey
            ])->get($this->baseUrl . "/v3/kyc/nin/{$nin}");

            if ($response->successful()) {
                $responseData = $response->json();
                
                return [
                    'success' => true,
                    'data' => $responseData['data']
                ];
            }

            return [
                'success' => false,
                'message' => $response->json()['message'] ?? 'NIN not found'
            ];

        } catch (Exception $e) {
            Log::error('NIN details retrieval failed', [
                'error' => $e->getMessage(),
                'nin' => substr($nin, 0, 3) . '********'
            ]);

            return [
                'success' => false,
                'message' => 'Service temporarily unavailable'
            ];
        }
    }

    /**
     * Verify multiple NINs (batch processing)
     * 
     * TODO: Add batch size limits
     * TODO: Add progress tracking
     */
    public function verifyNinBatch(array $ninData)
    {
        try {
            $payload = [
                'requests' => $ninData
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json'
            ])->post($this->baseUrl . '/v3/kyc/nin/batch', $payload);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()['data']
                ];
            }

            return [
                'success' => false,
                'message' => $response->json()['message'] ?? 'Batch verification failed'
            ];

        } catch (Exception $e) {
            Log::error('NIN batch verification failed', [
                'error' => $e->getMessage(),
                'batch_size' => count($ninData)
            ]);

            return [
                'success' => false,
                'message' => 'Service temporarily unavailable'
            ];
        }
    }

    /**
     * Check if NIN is valid format
     * 
     * TODO: Add advanced NIN format validation
     */
    public function isValidNinFormat(string $nin)
    {
        // NIN should be 11 digits
        return preg_match('/^[0-9]{11}$/', $nin);
    }

    /**
     * Check if NIN exists (without getting details)
     * 
     * TODO: Add caching for existence checks
     */
    public function ninExists(string $nin)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey
            ])->head($this->baseUrl . "/v3/kyc/nin/{$nin}");

            return $response->status() === 200;

        } catch (Exception $e) {
            Log::error('NIN existence check failed', [
                'error' => $e->getMessage(),
                'nin' => substr($nin, 0, 3) . '********'
            ]);

            return false;
        }
    }

    /**
     * Verify NIN with photo (facial recognition)
     * 
     * TODO: Add photo quality validation
     * TODO: Add facial matching confidence scores
     */
    public function verifyNinWithPhoto(string $nin, string $photoBase64)
    {
        try {
            $payload = [
                'nin' => $nin,
                'selfie' => $photoBase64
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json'
            ])->post($this->baseUrl . '/v3/kyc/nin/photo-verification', $payload);

            if ($response->successful()) {
                $responseData = $response->json();
                
                return [
                    'success' => true,
                    'data' => [
                        'is_verified' => $responseData['data']['is_verified'],
                        'confidence_score' => $responseData['data']['confidence_score'],
                        'match_percentage' => $responseData['data']['match_percentage'],
                        'nin_details' => $responseData['data']['nin_details']
                    ]
                ];
            }

            return [
                'success' => false,
                'message' => $response->json()['message'] ?? 'Photo verification failed'
            ];

        } catch (Exception $e) {
            Log::error('NIN photo verification failed', [
                'error' => $e->getMessage(),
                'nin' => substr($nin, 0, 3) . '********'
            ]);

            return [
                'success' => false,
                'message' => 'Service temporarily unavailable'
            ];
        }
    }

    /**
     * Get verification cost
     * 
     * TODO: Add cost calculation based on verification type
     */
    public function getVerificationCost(string $verificationType = 'basic')
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey
            ])->get($this->baseUrl . '/v3/pricing/nin', [
                'type' => $verificationType
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()['data']
                ];
            }

            return [
                'success' => false,
                'message' => 'Unable to get pricing information'
            ];

        } catch (Exception $e) {
            Log::error('NIN pricing check failed', [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Service temporarily unavailable'
            ];
        }
    }

    /**
     * Track verification usage and costs
     * 
     * TODO: Add usage analytics
     * TODO: Add cost optimization recommendations
     */
    public function getUsageStatistics(array $dateRange = [])
    {
        try {
            $params = [];
            if (!empty($dateRange)) {
                $params = [
                    'start_date' => $dateRange['start_date'],
                    'end_date' => $dateRange['end_date']
                ];
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey
            ])->get($this->baseUrl . '/v3/analytics/nin', $params);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()['data']
                ];
            }

            return [
                'success' => false,
                'message' => 'Unable to get usage statistics'
            ];

        } catch (Exception $e) {
            Log::error('NIN usage statistics failed', [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Service temporarily unavailable'
            ];
        }
    }
}