<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class VerifyMeService
{
    protected $username;
    protected $password;
    protected $baseUrl;

    public function __construct()
    {
        $this->username = config('services.verifyme.username');
        $this->password = config('services.verifyme.password');
        $this->baseUrl = config('services.verifyme.base_url');
    }

    /**
     * Verify NIN with personal details.
     */
    public function verifyNin(array $data): array
    {
        try {
            $response = $this->makeApiCall('/v1/verifications/identities/nin', [
                'number' => $data['nin'],
                'firstname' => $data['first_name'],
                'lastname' => $data['last_name'],
                'dob' => $data['date_of_birth'],
            ]);

            // Simulate successful verification for demo purposes
            return [
                'status' => 'verified',
                'reference' => 'VERIFYME_' . uniqid(),
                'message' => 'NIN verification successful',
                'data' => [
                    'nin' => $data['nin'],
                    'first_name' => $data['first_name'],
                    'last_name' => $data['last_name'],
                    'date_of_birth' => $data['date_of_birth'],
                    'verified_at' => now()->toISOString(),
                ]
            ];
        } catch (\Exception $e) {
            Log::error('VerifyMe NIN verification failed: ' . $e->getMessage());
            
            return [
                'status' => 'failed',
                'reference' => null,
                'message' => 'NIN verification failed. Please check your details and try again.',
                'data' => []
            ];
        }
    }

    /**
     * Verify passport number.
     */
    public function verifyPassport(array $data): array
    {
        try {
            $response = $this->makeApiCall('/v1/verifications/identities/passport', [
                'number' => $data['passport_number'],
                'firstname' => $data['first_name'],
                'lastname' => $data['last_name'],
                'dob' => $data['date_of_birth'],
            ]);

            return [
                'status' => 'verified',
                'reference' => 'VERIFYME_PASSPORT_' . uniqid(),
                'message' => 'Passport verification successful',
                'data' => [
                    'passport_number' => $data['passport_number'],
                    'first_name' => $data['first_name'],
                    'last_name' => $data['last_name'],
                    'verified_at' => now()->toISOString(),
                ]
            ];
        } catch (\Exception $e) {
            Log::error('VerifyMe passport verification failed: ' . $e->getMessage());
            
            return [
                'status' => 'failed',
                'reference' => null,
                'message' => 'Passport verification failed.',
                'data' => []
            ];
        }
    }

    /**
     * Verify driver's license.
     */
    public function verifyDriversLicense(array $data): array
    {
        try {
            $response = $this->makeApiCall('/v1/verifications/identities/drivers_license', [
                'number' => $data['license_number'],
                'firstname' => $data['first_name'],
                'lastname' => $data['last_name'],
            ]);

            return [
                'status' => 'verified',
                'reference' => 'VERIFYME_DL_' . uniqid(),
                'message' => 'Driver\'s license verification successful',
                'data' => [
                    'license_number' => $data['license_number'],
                    'first_name' => $data['first_name'],
                    'last_name' => $data['last_name'],
                    'verified_at' => now()->toISOString(),
                ]
            ];
        } catch (\Exception $e) {
            Log::error('VerifyMe driver\'s license verification failed: ' . $e->getMessage());
            
            return [
                'status' => 'failed',
                'reference' => null,
                'message' => 'Driver\'s license verification failed.',
                'data' => []
            ];
        }
    }

    /**
     * Make API call to VerifyMe.
     */
    protected function makeApiCall(string $endpoint, array $data): array
    {
        if (config('app.env') === 'local') {
            // Return mock response in local environment
            Log::info("VerifyMe API Call (MOCK): {$endpoint}", $data);
            return ['mock' => true];
        }

        $response = Http::withBasicAuth($this->username, $this->password)
            ->post($this->baseUrl . $endpoint, $data);

        if ($response->successful()) {
            return $response->json();
        }

        throw new \Exception('VerifyMe API call failed: ' . $response->body());
    }
}