<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;

class ApiConfigSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $apiConfigs = [
            // Payment Providers
            [
                'name' => 'Paystack',
                'slug' => 'paystack',
                'category' => 'payment',
                'provider_code' => 'paystack',
                'description' => 'Payment processing and card transactions',
                'base_url' => 'https://api.paystack.co',
                'test_url' => 'https://api.paystack.co',
                'live_url' => 'https://api.paystack.co',
                'config' => [
                    'timeout' => 30,
                    'retry_attempts' => 3,
                    'supported_currencies' => ['NGN', 'USD', 'GHS', 'ZAR'],
                    'supports_cards' => true,
                    'supports_bank_transfer' => true,
                ],
                'endpoints' => [
                    'initialize' => '/transaction/initialize',
                    'verify' => '/transaction/verify',
                    'banks' => '/bank',
                    'charge' => '/charge',
                ],
                'webhook_config' => [
                    'events' => ['charge.success', 'transfer.success', 'transfer.failed'],
                    'signature_header' => 'X-Paystack-Signature',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],
            [
                'name' => 'Flutterwave',
                'slug' => 'flutterwave',
                'category' => 'payment',
                'provider_code' => 'flutterwave',
                'description' => 'Payment gateway and financial services',
                'base_url' => 'https://api.flutterwave.com/v3',
                'test_url' => 'https://api.flutterwave.com/v3',
                'live_url' => 'https://api.flutterwave.com/v3',
                'config' => [
                    'timeout' => 30,
                    'retry_attempts' => 3,
                    'supported_currencies' => ['NGN', 'USD', 'EUR', 'GBP'],
                    'supports_cards' => true,
                    'supports_bank_transfer' => true,
                ],
                'endpoints' => [
                    'payments' => '/payments',
                    'verify' => '/transactions/{id}/verify',
                    'banks' => '/banks/NG',
                    'transfers' => '/transfers',
                ],
                'webhook_config' => [
                    'events' => ['charge.completed', 'transfer.completed'],
                    'signature_header' => 'verif-hash',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],
            [
                'name' => 'Monnify',
                'slug' => 'monnify',
                'category' => 'payment',
                'provider_code' => 'monnify',
                'description' => 'Payment processing and collection services',
                'base_url' => 'https://sandbox.monnify.com',
                'test_url' => 'https://sandbox.monnify.com',
                'live_url' => 'https://api.monnify.com',
                'config' => [
                    'timeout' => 30,
                    'retry_attempts' => 3,
                    'supported_currencies' => ['NGN'],
                    'supports_cards' => true,
                    'supports_bank_transfer' => true,
                ],
                'endpoints' => [
                    'auth' => '/api/v1/auth/login',
                    'initialize' => '/api/v1/merchant/transactions/init-transaction',
                    'verify' => '/api/v1/merchant/transactions/query',
                    'banks' => '/api/v1/banks',
                ],
                'webhook_config' => [
                    'events' => ['SUCCESSFUL_TRANSACTION', 'FAILED_TRANSACTION'],
                    'signature_header' => 'monnify-signature',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],

            // Utility Services
            [
                'name' => 'VTPass',
                'slug' => 'vtpass',
                'category' => 'utilities',
                'provider_code' => 'vtpass',
                'description' => 'Bill payment and utility services',
                'base_url' => 'https://vtpass.com/api',
                'test_url' => 'https://sandbox.vtpass.com/api',
                'live_url' => 'https://vtpass.com/api',
                'config' => [
                    'timeout' => 45,
                    'retry_attempts' => 3,
                    'supported_services' => ['airtime', 'data', 'electricity', 'cable_tv'],
                ],
                'endpoints' => [
                    'services' => '/services',
                    'products' => '/service-variations',
                    'verify' => '/merchant-verify',
                    'pay' => '/pay',
                ],
                'webhook_config' => [
                    'events' => ['successful', 'failed', 'pending'],
                    'signature_header' => 'X-VTPass-Signature',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],

            // Identity Verification
            [
                'name' => 'Dojah',
                'slug' => 'dojah',
                'category' => 'identity',
                'provider_code' => 'dojah',
                'description' => 'Identity verification and KYC services',
                'base_url' => 'https://api.dojah.io',
                'test_url' => 'https://sandbox.dojah.io',
                'live_url' => 'https://api.dojah.io',
                'config' => [
                    'timeout' => 60,
                    'retry_attempts' => 2,
                    'supported_verifications' => ['bvn', 'nin', 'phone', 'email', 'face'],
                ],
                'endpoints' => [
                    'bvn' => '/api/v1/kyc/bvn',
                    'nin' => '/api/v1/kyc/nin',
                    'phone' => '/api/v1/kyc/phone_number',
                    'face' => '/api/v1/ml/liveness',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],
            [
                'name' => 'VerifyMe',
                'slug' => 'verifyme',
                'category' => 'identity',
                'provider_code' => 'verifyme',
                'description' => 'Identity verification and compliance services',
                'base_url' => 'https://api.myidentitypay.com',
                'test_url' => 'https://sandbox-api.myidentitypay.com',
                'live_url' => 'https://api.myidentitypay.com',
                'config' => [
                    'timeout' => 60,
                    'retry_attempts' => 2,
                    'supported_verifications' => ['bvn', 'nin', 'cac', 'drivers_license'],
                ],
                'endpoints' => [
                    'bvn' => '/api/v2/biometrics/merchant/data/verification/bvn',
                    'nin' => '/api/v2/biometrics/merchant/data/verification/nin',
                    'phone' => '/api/v2/biometrics/merchant/data/verification/phone',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],

            // Messaging Services
            [
                'name' => 'Termii',
                'slug' => 'termii',
                'category' => 'messaging',
                'provider_code' => 'termii',
                'description' => 'SMS and messaging services',
                'base_url' => 'https://api.ng.termii.com',
                'test_url' => 'https://api.ng.termii.com',
                'live_url' => 'https://api.ng.termii.com',
                'config' => [
                    'timeout' => 30,
                    'retry_attempts' => 3,
                    'supported_channels' => ['sms', 'voice', 'whatsapp'],
                ],
                'endpoints' => [
                    'send' => '/api/sms/send',
                    'otp' => '/api/sms/otp/send',
                    'verify' => '/api/sms/otp/verify',
                    'balance' => '/api/get-balance',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],

            // Banking Services
            [
                'name' => 'Anchor',
                'slug' => 'anchor',
                'category' => 'banking',
                'provider_code' => 'anchor',
                'description' => 'Digital banking and account management',
                'base_url' => 'https://api.sandbox.getanchor.co',
                'test_url' => 'https://api.sandbox.getanchor.co',
                'live_url' => 'https://api.getanchor.co',
                'config' => [
                    'timeout' => 30,
                    'retry_attempts' => 3,
                    'supported_services' => ['accounts', 'transfers', 'collections'],
                ],
                'endpoints' => [
                    'accounts' => '/api/v1/accounts',
                    'transfers' => '/api/v1/transfers',
                    'transactions' => '/api/v1/transactions',
                    'balance' => '/api/v1/accounts/balance',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],
            [
                'name' => 'Bloc.io',
                'slug' => 'bloc-io',
                'category' => 'banking',
                'provider_code' => 'bloc',
                'description' => 'Cryptocurrency and digital wallet services with virtual cards',
                'base_url' => 'https://api.bloc.io',
                'test_url' => 'https://api.sandbox.bloc.io',
                'live_url' => 'https://api.bloc.io',
                'config' => [
                    'timeout' => 45,
                    'retry_attempts' => 3,
                    'supported_services' => ['crypto_wallets', 'exchange', 'transfers', 'virtual_cards'],
                    'supports_cards' => true,
                    'supported_card_types' => ['virtual', 'physical'],
                    'supported_card_brands' => ['visa', 'mastercard'],
                ],
                'endpoints' => [
                    'wallets' => '/api/v1/wallets',
                    'exchange' => '/api/v1/exchange/rates',
                    'transfers' => '/api/v1/transfers',
                    'balance' => '/api/v1/wallets/balance',
                    'cards' => '/api/v1/cards',
                    'card_fund' => '/api/v1/cards/{id}/fund',
                    'card_freeze' => '/api/v1/cards/{id}/freeze',
                    'card_unfreeze' => '/api/v1/cards/{id}/unfreeze',
                    'card_transactions' => '/api/v1/cards/{id}/transactions',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],
            [
                'name' => 'Wallets Africa',
                'slug' => 'wallets-africa',
                'category' => 'payment',
                'provider_code' => 'wallets_africa',
                'description' => 'Digital payment and wallet services for Africa with virtual cards',
                'base_url' => 'https://api.wallets.africa',
                'test_url' => 'https://sandbox.wallets.africa',
                'live_url' => 'https://api.wallets.africa',
                'config' => [
                    'timeout' => 30,
                    'retry_attempts' => 3,
                    'supported_services' => ['payments', 'transfers', 'bills', 'virtual_cards'],
                    'supported_currencies' => ['NGN', 'GHS', 'KES', 'UGX', 'USD'],
                    'supports_cards' => true,
                    'supported_card_types' => ['virtual'],
                    'supported_card_brands' => ['visa', 'mastercard'],
                ],
                'endpoints' => [
                    'payments' => '/api/v1/payments',
                    'transfers' => '/api/v1/transfers',
                    'bills' => '/api/v1/bills',
                    'balance' => '/api/v1/wallet/balance',
                    'cards' => '/api/v1/cards/create',
                    'card_fund' => '/api/v1/cards/fund',
                    'card_freeze' => '/api/v1/cards/freeze',
                    'card_unfreeze' => '/api/v1/cards/unfreeze',
                    'card_terminate' => '/api/v1/cards/terminate',
                    'card_transactions' => '/api/v1/cards/transactions',
                ],
                'webhook_config' => [
                    'events' => ['payment.success', 'payment.failed', 'transfer.success', 'card.transaction'],
                    'signature_header' => 'X-WA-Signature',
                ],
                'is_active' => false,
                'is_live_mode' => false,
            ],
        ];

        foreach ($apiConfigs as $config) {
            \App\Models\ApiConfig::updateOrCreate(
                ['slug' => $config['slug']],
                $config
            );
        }
    }
}
