<?php

namespace Tests\Unit\Services;

use Tests\TestCase;
use App\Services\CardProviderManager;
use App\Models\ApiConfig;
use Illuminate\Foundation\Testing\RefreshDatabase;

class CardProviderManagerTest extends TestCase
{
    use RefreshDatabase;

    public function test_card_provider_manager_returns_available_providers()
    {
        $cardProviderManager = new CardProviderManager();
        $providers = $cardProviderManager->getAvailableProviders();

        // Should return array of providers with expected structure
        $this->assertIsArray($providers);
        
        foreach ($providers as $provider) {
            $this->assertArrayHasKey('slug', $provider);
            $this->assertArrayHasKey('name', $provider);
            $this->assertArrayHasKey('configured', $provider);
            $this->assertArrayHasKey('is_active', $provider);
            $this->assertArrayHasKey('config_exists', $provider);
            $this->assertArrayHasKey('config_errors', $provider);
        }
    }

    public function test_card_provider_manager_detects_missing_config()
    {
        $cardProviderManager = new CardProviderManager();
        $errors = $cardProviderManager->getConfigurationErrors('bloc-io', null);

        $this->assertContains('API configuration not found', $errors);
    }

    public function test_card_provider_manager_validates_bloc_io_config()
    {
        // Create a minimal ApiConfig for bloc-io
        $config = ApiConfig::create([
            'name' => 'Bloc.io',
            'slug' => 'bloc-io',
            'category' => 'banking',
            'base_url' => 'https://api.bloc.io',
            'is_active' => true,
            'is_live_mode' => false,
        ]);

        $cardProviderManager = new CardProviderManager();
        $errors = $cardProviderManager->getConfigurationErrors('bloc-io', $config);

        // Should detect missing API key and secret key
        $this->assertContains('API key is required', $errors);
        $this->assertContains('Secret key is required', $errors);
    }

    public function test_card_provider_manager_validates_complete_config()
    {
        // Create a complete ApiConfig for bloc-io
        $config = ApiConfig::create([
            'name' => 'Bloc.io',
            'slug' => 'bloc-io',
            'category' => 'banking',
            'base_url' => 'https://api.bloc.io',
            'api_key' => 'test-api-key',
            'secret_key' => 'test-secret-key',
            'is_active' => true,
            'is_live_mode' => false,
        ]);

        $cardProviderManager = new CardProviderManager();
        $errors = $cardProviderManager->getConfigurationErrors('bloc-io', $config);

        // Should have no errors for a complete config
        $this->assertEmpty($errors);
    }

    public function test_card_provider_manager_checks_supported_providers()
    {
        $cardProviderManager = new CardProviderManager();

        $this->assertTrue($cardProviderManager->isProviderSupported('bloc-io'));
        $this->assertTrue($cardProviderManager->isProviderSupported('wallets-africa'));
        $this->assertFalse($cardProviderManager->isProviderSupported('nonexistent-provider'));
    }

    public function test_get_active_providers_filters_correctly()
    {
        // Create active and inactive configs
        ApiConfig::create([
            'name' => 'Active Provider',
            'slug' => 'bloc-io',
            'category' => 'banking',
            'base_url' => 'https://api.bloc.io',
            'api_key' => 'test-key',
            'secret_key' => 'test-secret',
            'is_active' => true,
        ]);

        ApiConfig::create([
            'name' => 'Inactive Provider',
            'slug' => 'wallets-africa',
            'category' => 'payment',
            'base_url' => 'https://api.wallets.africa',
            'api_key' => 'test-key',
            'secret_key' => 'test-secret',
            'is_active' => false, // Inactive
        ]);

        $cardProviderManager = new CardProviderManager();
        $activeProviders = $cardProviderManager->getActiveProviders();

        // Should only return active and configured providers
        $activeSlugs = collect($activeProviders)->pluck('slug')->toArray();
        $this->assertContains('bloc-io', $activeSlugs);
        $this->assertNotContains('wallets-africa', $activeSlugs);
    }
}