<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class VirtualCard extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'card_id',
        'masked_pan',
        'card_type',
        'brand',
        'currency',
        'balance',
        'daily_limit',
        'monthly_limit',
        'status',
        'provider',
        'provider_reference',
        'metadata',
        'expires_at',
    ];

    protected function casts(): array
    {
        return [
            'balance' => 'decimal:2',
            'daily_limit' => 'decimal:2',
            'monthly_limit' => 'decimal:2',
            'metadata' => 'array',
            'expires_at' => 'datetime',
        ];
    }

    /**
     * Get the user that owns the virtual card.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the transactions for this virtual card.
     */
    public function transactions(): HasMany
    {
        return $this->hasMany(Transaction::class);
    }

    /**
     * Check if card is active.
     */
    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    /**
     * Check if card is frozen.
     */
    public function isFrozen(): bool
    {
        return $this->status === 'frozen';
    }

    /**
     * Check if card is terminated.
     */
    public function isTerminated(): bool
    {
        return $this->status === 'terminated';
    }

    /**
     * Freeze the card.
     */
    public function freeze(): void
    {
        $this->update(['status' => 'frozen']);
    }

    /**
     * Unfreeze the card.
     */
    public function unfreeze(): void
    {
        $this->update(['status' => 'active']);
    }

    /**
     * Terminate the card.
     */
    public function terminate(): void
    {
        $this->update(['status' => 'terminated']);
    }

    /**
     * Get formatted card details.
     */
    public function getFormattedDetailsAttribute(): string
    {
        return "{$this->brand} **** {$this->getLast4Digits()}";
    }

    /**
     * Get last 4 digits of card.
     */
    public function getLast4Digits(): string
    {
        return substr($this->masked_pan, -4);
    }

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return \Database\Factories\VirtualCardFactory::new();
    }
}