<?php

namespace App\Services\Airtime\Providers;

use App\Services\Airtime\BaseAirtimeProvider;

class VTPassProvider extends BaseAirtimeProvider
{
    protected string $apiKey;
    protected string $secretKey;
    protected string $username;
    protected string $password;

    /**
     * Configure VTPass provider.
     */
    public function configure(array $credentials): void
    {
        parent::configure($credentials);
        
        $this->apiKey = $credentials['api_key'] ?? '';
        $this->secretKey = $credentials['secret_key'] ?? '';
        $this->username = $credentials['username'] ?? '';
        $this->password = $credentials['password'] ?? '';
        
        if (empty($this->apiKey) || empty($this->secretKey)) {
            throw new \InvalidArgumentException('VTPass API key and secret key are required');
        }
    }

    /**
     * Purchase airtime via VTPass.
     */
    public function purchaseAirtime(array $params): array
    {
        $this->validateParameters($params, ['phone_number', 'amount', 'network']);

        $serviceId = $this->getServiceId('airtime', $params['network']);
        $requestId = $this->generateReference('VTPASS_AIR');

        $data = [
            'request_id' => $requestId,
            'serviceID' => $serviceId,
            'amount' => $params['amount'],
            'phone' => $this->formatPhoneNumber($params['phone_number']),
        ];

        $response = $this->makeRequest('POST', '/pay', $data, $this->getAuthHeaders());

        if (!$response['success']) {
            return [
                'success' => false,
                'message' => 'Network error: ' . ($response['error'] ?? 'Unknown error'),
                'reference' => $requestId,
                'provider_response' => $response,
            ];
        }

        $responseData = $response['data'];
        $success = isset($responseData['code']) && in_array($responseData['code'], ['000', '200']);

        return [
            'success' => $success,
            'message' => $responseData['response_description'] ?? ($success ? 'Transaction successful' : 'Transaction failed'),
            'reference' => $requestId,
            'provider_reference' => $responseData['requestId'] ?? $requestId,
            'amount' => $params['amount'],
            'phone_number' => $params['phone_number'],
            'network' => $params['network'],
            'provider_response' => $response,
        ];
    }

    /**
     * Purchase data bundle via VTPass.
     */
    public function purchaseData(array $params): array
    {
        $this->validateParameters($params, ['phone_number', 'data_plan', 'network']);

        $serviceId = $this->getServiceId('data', $params['network']);
        $requestId = $this->generateReference('VTPASS_DATA');

        $data = [
            'request_id' => $requestId,
            'serviceID' => $serviceId,
            'billersCode' => $this->formatPhoneNumber($params['phone_number']),
            'variation_code' => $params['data_plan'],
            'amount' => $params['amount'] ?? null,
            'phone' => $this->formatPhoneNumber($params['phone_number']),
        ];

        $response = $this->makeRequest('POST', '/pay', $data, $this->getAuthHeaders());

        if (!$response['success']) {
            return [
                'success' => false,
                'message' => 'Network error: ' . ($response['error'] ?? 'Unknown error'),
                'reference' => $requestId,
                'provider_response' => $response,
            ];
        }

        $responseData = $response['data'];
        $success = isset($responseData['code']) && in_array($responseData['code'], ['000', '200']);

        return [
            'success' => $success,
            'message' => $responseData['response_description'] ?? ($success ? 'Transaction successful' : 'Transaction failed'),
            'reference' => $requestId,
            'provider_reference' => $responseData['requestId'] ?? $requestId,
            'amount' => $responseData['amount'] ?? $params['amount'],
            'phone_number' => $params['phone_number'],
            'network' => $params['network'],
            'data_plan' => $params['data_plan'],
            'provider_response' => $response,
        ];
    }

    /**
     * Get available data plans for a network.
     */
    public function getDataPlans(string $network): array
    {
        $serviceId = $this->getServiceId('data', $network);
        
        $response = $this->makeRequest('GET', "/service-variations", ['serviceID' => $serviceId], $this->getAuthHeaders());

        if (!$response['success']) {
            return [];
        }

        $responseData = $response['data'];
        if (!isset($responseData['content']['variations'])) {
            return [];
        }

        $plans = [];
        foreach ($responseData['content']['variations'] as $variation) {
            $plans[] = [
                'id' => $variation['variation_code'],
                'name' => $variation['name'],
                'amount' => $variation['variation_amount'],
                'validity' => $variation['fixedPrice'] ?? null,
            ];
        }

        return $plans;
    }

    /**
     * Check transaction status.
     */
    public function checkTransactionStatus(string $reference): array
    {
        $data = ['request_id' => $reference];
        $response = $this->makeRequest('POST', '/requery', $data, $this->getAuthHeaders());

        if (!$response['success']) {
            return [
                'success' => false,
                'status' => 'unknown',
                'message' => 'Failed to query status',
            ];
        }

        $responseData = $response['data'];
        $status = 'pending';

        if (isset($responseData['content']['transactions']['status'])) {
            $vtpassStatus = strtolower($responseData['content']['transactions']['status']);
            $status = match($vtpassStatus) {
                'delivered', 'successful' => 'completed',
                'failed' => 'failed',
                default => 'pending',
            };
        }

        return [
            'success' => true,
            'status' => $status,
            'message' => $responseData['response_description'] ?? 'Status checked',
            'provider_response' => $response,
        ];
    }

    /**
     * Get account balance.
     */
    public function getBalance(): array
    {
        $response = $this->makeRequest('GET', '/balance', [], $this->getAuthHeaders());

        if (!$response['success']) {
            return [
                'success' => false,
                'balance' => 0,
                'message' => 'Failed to get balance',
            ];
        }

        $responseData = $response['data'];
        
        return [
            'success' => true,
            'balance' => $responseData['contents']['balance'] ?? 0,
            'currency' => 'NGN',
            'message' => 'Balance retrieved successfully',
        ];
    }

    /**
     * Test connection to VTPass.
     */
    public function testConnection(): array
    {
        try {
            $balanceResponse = $this->getBalance();
            
            return [
                'success' => $balanceResponse['success'],
                'message' => $balanceResponse['success'] ? 'Connection successful' : 'Connection failed',
                'response_time' => 0, // Would need to measure this properly
                'provider_response' => $balanceResponse,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage(),
                'response_time' => 0,
            ];
        }
    }

    /**
     * Verify phone number and detect network.
     */
    public function verifyPhoneNumber(string $phoneNumber): array
    {
        $network = $this->detectNetwork($phoneNumber);
        
        return [
            'success' => !is_null($network),
            'phone_number' => $this->formatPhoneNumber($phoneNumber),
            'network' => $network,
            'message' => $network ? 'Phone number verified' : 'Network not detected',
        ];
    }

    /**
     * Get VTPass service ID for network and service type.
     */
    protected function getServiceId(string $serviceType, string $network): string
    {
        $serviceIds = [
            'airtime' => [
                'mtn' => 'mtn',
                'glo' => 'glo',
                'airtel' => 'airtel',
                '9mobile' => 'etisalat',
            ],
            'data' => [
                'mtn' => 'mtn-data',
                'glo' => 'glo-data',
                'airtel' => 'airtel-data',
                '9mobile' => 'etisalat-data',
            ],
        ];

        return $serviceIds[$serviceType][$network] ?? '';
    }

    /**
     * Get authentication headers for VTPass API.
     */
    protected function getAuthHeaders(): array
    {
        return [
            'api-key' => $this->apiKey,
            'secret-key' => $this->secretKey,
        ];
    }
}