<?php

namespace App\Services;

use App\Services\CardProviders\CardProviderInterface;
use App\Services\CardProviders\BlocService;
use App\Services\CardProviders\WalletsAfricaService;
use App\Models\ApiConfig;
use Illuminate\Support\Facades\Log;

class CardProviderManager
{
    protected array $providers = [
        'bloc-io' => BlocService::class,
        'wallets-africa' => WalletsAfricaService::class,
    ];

    /**
     * Get a card provider instance by slug.
     */
    public function getProvider(string $slug): CardProviderInterface
    {
        if (!isset($this->providers[$slug])) {
            throw new \InvalidArgumentException("Card provider '{$slug}' is not supported");
        }

        $providerClass = $this->providers[$slug];
        
        return new $providerClass();
    }

    /**
     * Get available card providers.
     */
    public function getAvailableProviders(): array
    {
        $providers = [];

        foreach ($this->providers as $slug => $class) {
            try {
                $provider = new $class();
                $config = ApiConfig::where('slug', $slug)->first();
                
                $providers[] = [
                    'slug' => $slug,
                    'name' => $provider->getProviderName(),
                    'configured' => $config ? $provider->isConfigured() : false,
                    'is_active' => $config ? $config->is_active : false,
                    'is_live_mode' => $config ? $config->is_live_mode : false,
                    'config_exists' => $config !== null,
                    'last_test_status' => $config ? $config->last_test_status : null,
                    'config_errors' => $this->getConfigurationErrors($slug, $config),
                ];
            } catch (\Exception $e) {
                Log::warning("Failed to initialize card provider '{$slug}': " . $e->getMessage(), ['exception' => $e]);
                if (App::environment(['local', 'development', 'testing'])) {
                    // In development, rethrow the exception to surface the real issue
                    throw $e;
                }
                $providers[] = [
                    'slug' => $slug,
                    'name' => ucfirst(str_replace('-', ' ', $slug)),
                    'configured' => false,
                    'is_active' => false,
                    'is_live_mode' => false,
                    'config_exists' => false,
                    'last_test_status' => 'error',
                    'config_errors' => ['Provider class not found or failed to initialize'],
                ];
            }
        }

        return $providers;
    }

    /**
     * Get detailed configuration errors for a provider.
     */
    public function getConfigurationErrors(string $slug, ?ApiConfig $config = null): array
    {
        $errors = [];
        
        if (!$config) {
            $errors[] = 'API configuration not found';
            return $errors;
        }
        
        // Check required credentials based on provider
        switch ($slug) {
            case 'bloc-io':
                if (empty($config->api_key)) {
                    $errors[] = 'API key is required';
                }
                if (empty($config->secret_key)) {
                    $errors[] = 'Secret key is required';
                }
                break;
                
            case 'wallets-africa':
                if (empty($config->api_key)) {
                    $errors[] = 'API key is required';
                }
                if (empty($config->secret_key)) {
                    $errors[] = 'Secret key is required';
                }
                break;
                
            default:
                // Generic checks
                if (empty($config->api_key) && empty($config->client_id)) {
                    $errors[] = 'Either API key or Client ID is required';
                }
        }
        
        // Check URLs
        if (empty($config->base_url)) {
            $errors[] = 'Base URL is required';
        }
        
        if ($config->is_live_mode && empty($config->live_url)) {
            $errors[] = 'Live URL is required when in live mode';
        }
        
        return $errors;
    }

    /**
     * Get configured and active card providers.
     */
    public function getActiveProviders(): array
    {
        return array_filter($this->getAvailableProviders(), function ($provider) {
            return $provider['configured'] && $provider['is_active'];
        });
    }

    /**
     * Check if a provider is supported.
     */
    public function isProviderSupported(string $slug): bool
    {
        return isset($this->providers[$slug]);
    }

    /**
     * Create a virtual card using the specified provider.
     */
    public function createCard(string $providerSlug, array $data): array
    {
        $provider = $this->getProvider($providerSlug);
        
        return $provider->createCard($data);
    }

    /**
     * Fund a virtual card using the specified provider.
     */
    public function fundCard(string $providerSlug, string $cardId, float $amount): array
    {
        $provider = $this->getProvider($providerSlug);
        
        return $provider->fundCard($cardId, $amount);
    }

    /**
     * Freeze a virtual card using the specified provider.
     */
    public function freezeCard(string $providerSlug, string $cardId): array
    {
        $provider = $this->getProvider($providerSlug);
        
        return $provider->freezeCard($cardId);
    }

    /**
     * Unfreeze a virtual card using the specified provider.
     */
    public function unfreezeCard(string $providerSlug, string $cardId): array
    {
        $provider = $this->getProvider($providerSlug);
        
        return $provider->unfreezeCard($cardId);
    }

    /**
     * Terminate a virtual card using the specified provider.
     */
    public function terminateCard(string $providerSlug, string $cardId): array
    {
        $provider = $this->getProvider($providerSlug);
        
        return $provider->terminateCard($cardId);
    }
}