@extends('layouts.admin-sidebar')

@section('title', 'Add New API Configuration')
@section('page-title', 'Add New API Configuration')
@section('page-description', 'Configure a new API service or card provider')

@section('content')
<div class="space-y-6">
    <!-- Header with Back Button -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
        <div class="flex items-center space-x-4">
            <a href="{{ route('admin.api-config.index') }}" class="text-gray-600 hover:text-gray-800">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                </svg>
            </a>
            <div>
                <h1 class="text-2xl font-bold text-gray-900">Add New API Configuration</h1>
                <p class="text-gray-600">Configure a new service provider or card provider</p>
            </div>
        </div>
    </div>

    <form action="{{ route('admin.api-config.store') }}" method="POST" class="space-y-6">
        @csrf
        
        <!-- Provider Template Selection -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <h3 class="text-lg font-semibold text-gray-900 mb-4">Provider Template</h3>
            <p class="text-sm text-gray-600 mb-4">Choose a template to pre-fill configuration details, or select "Custom" to configure manually.</p>
            
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                @foreach($cardProviderTemplates as $templateKey => $template)
                <div class="border border-gray-200 rounded-lg p-4 cursor-pointer hover:border-blue-500 transition-colors template-option" 
                     data-template="{{ $templateKey }}" 
                     data-config="{{ json_encode($template) }}">
                    <div class="flex items-center mb-2">
                        <input type="radio" name="template" value="{{ $templateKey }}" id="template_{{ $templateKey }}" 
                               class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300">
                        <label for="template_{{ $templateKey }}" class="ml-2 text-sm font-medium text-gray-900">
                            {{ $template['name'] ?: ucfirst(str_replace('-', ' ', $templateKey)) }}
                        </label>
                    </div>
                    @if($template['description'])
                    <p class="text-xs text-gray-600">{{ $template['description'] }}</p>
                    @endif
                    @if(isset($template['config']['supports_cards']) && $template['config']['supports_cards'])
                    <div class="mt-2">
                        <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                            Card Provider
                        </span>
                    </div>
                    @endif
                </div>
                @endforeach
            </div>
        </div>

        <!-- Basic Information -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <h3 class="text-lg font-semibold text-gray-900 mb-4">Basic Information</h3>
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label for="name" class="block text-sm font-medium text-gray-700 mb-1">Service Name *</label>
                    <input type="text" name="name" id="name" value="{{ old('name') }}" 
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                    @error('name')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
                
                <div>
                    <label for="slug" class="block text-sm font-medium text-gray-700 mb-1">Slug *</label>
                    <input type="text" name="slug" id="slug" value="{{ old('slug') }}" 
                           placeholder="e.g., bloc-io, wallets-africa"
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                    <p class="mt-1 text-xs text-gray-500">Lowercase letters, numbers, and hyphens only. This will be used in URLs and code.</p>
                    @error('slug')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
            </div>
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                <div>
                    <label for="category" class="block text-sm font-medium text-gray-700 mb-1">Category *</label>
                    <select name="category" id="category" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                        @foreach($categories as $key => $label)
                            <option value="{{ $key }}" {{ old('category') === $key ? 'selected' : '' }}>{{ $label }}</option>
                        @endforeach
                    </select>
                    @error('category')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
                
                <div class="flex items-center space-x-4 pt-6">
                    <label class="flex items-center">
                        <input type="hidden" name="is_active" value="0">
                        <input type="checkbox" name="is_active" value="1" {{ old('is_active') ? 'checked' : '' }}
                               class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                        <span class="ml-2 text-sm text-gray-700">Active</span>
                    </label>
                    
                    <label class="flex items-center">
                        <input type="hidden" name="is_live_mode" value="0">
                        <input type="checkbox" name="is_live_mode" value="1" {{ old('is_live_mode') ? 'checked' : '' }}
                               class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                        <span class="ml-2 text-sm text-gray-700">Live Mode</span>
                    </label>
                </div>
            </div>
            
            <div class="mt-4">
                <label for="description" class="block text-sm font-medium text-gray-700 mb-1">Description</label>
                <textarea name="description" id="description" rows="3" 
                          class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">{{ old('description') }}</textarea>
                @error('description')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>
        </div>
        
        <!-- API URLs -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <h3 class="text-lg font-semibold text-gray-900 mb-4">API Endpoints</h3>
            
            <div class="space-y-4">
                <div>
                    <label for="base_url" class="block text-sm font-medium text-gray-700 mb-1">Base URL *</label>
                    <input type="url" name="base_url" id="base_url" value="{{ old('base_url') }}" 
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                    @error('base_url')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label for="test_url" class="block text-sm font-medium text-gray-700 mb-1">Test/Sandbox URL</label>
                        <input type="url" name="test_url" id="test_url" value="{{ old('test_url') }}" 
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                        @error('test_url')
                            <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                    
                    <div>
                        <label for="live_url" class="block text-sm font-medium text-gray-700 mb-1">Live/Production URL</label>
                        <input type="url" name="live_url" id="live_url" value="{{ old('live_url') }}" 
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                        @error('live_url')
                            <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Authentication Credentials -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <h3 class="text-lg font-semibold text-gray-900 mb-4">Authentication Credentials</h3>
            <p class="text-sm text-gray-600 mb-4">These credentials will be encrypted when stored.</p>
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label for="api_key" class="block text-sm font-medium text-gray-700 mb-1">API Key</label>
                    <input type="text" name="api_key" id="api_key" value="{{ old('api_key') }}" 
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    @error('api_key')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
                
                <div>
                    <label for="secret_key" class="block text-sm font-medium text-gray-700 mb-1">Secret Key</label>
                    <input type="text" name="secret_key" id="secret_key" value="{{ old('secret_key') }}" 
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    @error('secret_key')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
                
                <div>
                    <label for="client_id" class="block text-sm font-medium text-gray-700 mb-1">Client ID</label>
                    <input type="text" name="client_id" id="client_id" value="{{ old('client_id') }}" 
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    @error('client_id')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
                
                <div>
                    <label for="client_secret" class="block text-sm font-medium text-gray-700 mb-1">Client Secret</label>
                    <input type="text" name="client_secret" id="client_secret" value="{{ old('client_secret') }}" 
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    @error('client_secret')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
                
                <div>
                    <label for="merchant_id" class="block text-sm font-medium text-gray-700 mb-1">Merchant ID</label>
                    <input type="text" name="merchant_id" id="merchant_id" value="{{ old('merchant_id') }}" 
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    @error('merchant_id')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
            </div>
        </div>
        
        <!-- Configuration Settings -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <h3 class="text-lg font-semibold text-gray-900 mb-4">Configuration Settings</h3>
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label for="config_timeout" class="block text-sm font-medium text-gray-700 mb-1">Timeout (seconds)</label>
                    <input type="number" name="config[timeout]" id="config_timeout" value="{{ old('config.timeout', 30) }}" 
                           min="5" max="120"
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    @error('config.timeout')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
                
                <div>
                    <label for="config_retry_attempts" class="block text-sm font-medium text-gray-700 mb-1">Retry Attempts</label>
                    <input type="number" name="config[retry_attempts]" id="config_retry_attempts" value="{{ old('config.retry_attempts', 3) }}" 
                           min="0" max="5"
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    @error('config.retry_attempts')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
            </div>
            
            <!-- Card Provider Specific Settings -->
            <div id="card-provider-settings" class="mt-6 hidden">
                <h4 class="text-md font-semibold text-gray-900 mb-3">Card Provider Settings</h4>
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="flex items-center">
                            <input type="hidden" name="config[supports_cards]" value="0">
                            <input type="checkbox" name="config[supports_cards]" value="1" id="supports_cards"
                                   class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                            <span class="ml-2 text-sm text-gray-700">Supports Virtual Cards</span>
                        </label>
                    </div>
                </div>
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                    <div>
                        <label for="supported_card_types" class="block text-sm font-medium text-gray-700 mb-1">Supported Card Types</label>
                        <input type="text" name="config[supported_card_types]" id="supported_card_types" 
                               placeholder="virtual,physical" value="{{ old('config.supported_card_types') }}"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                        <p class="mt-1 text-xs text-gray-500">Comma-separated list: virtual, physical</p>
                    </div>
                    
                    <div>
                        <label for="supported_card_brands" class="block text-sm font-medium text-gray-700 mb-1">Supported Card Brands</label>
                        <input type="text" name="config[supported_card_brands]" id="supported_card_brands" 
                               placeholder="visa,mastercard" value="{{ old('config.supported_card_brands') }}"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                        <p class="mt-1 text-xs text-gray-500">Comma-separated list: visa, mastercard</p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Submit Buttons -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div class="flex items-center justify-between">
                <a href="{{ route('admin.api-config.index') }}" 
                   class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors">
                    Cancel
                </a>
                <div class="flex items-center space-x-3">
                    <button type="submit" name="action" value="save_inactive"
                            class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors">
                        Save as Inactive
                    </button>
                    <button type="submit" name="action" value="save_active"
                            class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                        Save & Activate
                    </button>
                </div>
            </div>
        </div>
    </form>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const templateOptions = document.querySelectorAll('.template-option');
    const categorySelect = document.getElementById('category');
    const cardProviderSettings = document.getElementById('card-provider-settings');
    
    // Template selection functionality
    templateOptions.forEach(option => {
        option.addEventListener('click', function() {
            const radio = this.querySelector('input[type="radio"]');
            radio.checked = true;
            
            // Remove active state from all options
            templateOptions.forEach(opt => opt.classList.remove('border-blue-500', 'bg-blue-50'));
            
            // Add active state to selected option
            this.classList.add('border-blue-500', 'bg-blue-50');
            
            // Load template data
            const templateData = JSON.parse(this.dataset.config);
            loadTemplate(templateData);
        });
    });
    
    // Category change handler
    categorySelect.addEventListener('change', function() {
        toggleCardProviderSettings();
    });
    
    // Auto-generate slug from name
    const nameInput = document.getElementById('name');
    const slugInput = document.getElementById('slug');
    
    nameInput.addEventListener('input', function() {
        if (!slugInput.value) {
            const slug = this.value.toLowerCase()
                .replace(/[^a-z0-9\s-]/g, '')
                .replace(/\s+/g, '-')
                .replace(/-+/g, '-')
                .trim('-');
            slugInput.value = slug;
        }
    });
    
    function loadTemplate(templateData) {
        // Fill form fields with template data
        document.getElementById('name').value = templateData.name || '';
        document.getElementById('slug').value = templateData.name ? 
            templateData.name.toLowerCase().replace(/[^a-z0-9\s-]/g, '').replace(/\s+/g, '-') : '';
        document.getElementById('category').value = templateData.category || 'payment';
        document.getElementById('description').value = templateData.description || '';
        document.getElementById('base_url').value = templateData.base_url || '';
        document.getElementById('test_url').value = templateData.test_url || '';
        document.getElementById('live_url').value = templateData.live_url || '';
        
        // Configuration settings
        document.getElementById('config_timeout').value = templateData.config?.timeout || 30;
        document.getElementById('config_retry_attempts').value = templateData.config?.retry_attempts || 3;
        
        // Card provider settings
        if (templateData.config?.supports_cards) {
            document.getElementById('supports_cards').checked = true;
            document.getElementById('supported_card_types').value = 
                templateData.config?.supported_card_types?.join(',') || '';
            document.getElementById('supported_card_brands').value = 
                templateData.config?.supported_card_brands?.join(',') || '';
        }
        
        toggleCardProviderSettings();
    }
    
    function toggleCardProviderSettings() {
        const category = categorySelect.value;
        const isCardProvider = category === 'banking' || category === 'payment';
        
        if (isCardProvider) {
            cardProviderSettings.classList.remove('hidden');
        } else {
            cardProviderSettings.classList.add('hidden');
        }
    }
    
    // Initial toggle
    toggleCardProviderSettings();
});
</script>
@endsection