<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\WalletController;
use App\Http\Controllers\TransactionController;
use App\Http\Controllers\KycController;
use App\Http\Controllers\VirtualAccountController;
use App\Http\Controllers\BillPaymentController;
use App\Http\Controllers\VirtualCardController;
use App\Http\Controllers\SettingsController;
use App\Http\Controllers\Admin\AuthController as AdminAuthController;
use App\Http\Controllers\Admin\DashboardController as AdminDashboardController;
use App\Http\Controllers\Admin\UserController as AdminUserController;
use App\Http\Controllers\Admin\KycController as AdminKycController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

Route::get('/', function () {
    return view('welcome');
});

// Authentication Routes
Route::middleware('guest')->group(function () {
    Route::get('/login', [AuthController::class, 'showLoginForm'])->name('login');
    Route::post('/login', [AuthController::class, 'login']);
    Route::get('/register', [AuthController::class, 'showRegistrationForm'])->name('register');
    Route::post('/register', [AuthController::class, 'register']);
});

// Phone verification (accessible to authenticated users who haven't verified phone)
Route::middleware('auth')->group(function () {
    Route::get('/verify-phone', [AuthController::class, 'showVerifyPhone'])->name('verify.phone');
    Route::post('/verify-phone', [AuthController::class, 'verifyPhone']);
});

Route::post('/logout', [AuthController::class, 'logout'])->name('logout');

// Admin Authentication Routes
Route::prefix('admin')->name('admin.')->group(function () {
    Route::middleware('guest:admin')->group(function () {
        Route::get('/login', [AdminAuthController::class, 'showLoginForm'])->name('login');
        Route::post('/login', [AdminAuthController::class, 'login']);
    });
    
    Route::middleware(['auth:admin', 'admin'])->group(function () {
        // Dashboard
        Route::get('/dashboard', [AdminDashboardController::class, 'index'])->name('dashboard');
        Route::get('/alerts', [AdminDashboardController::class, 'getAlerts'])->name('alerts');
        
        // User Management
        Route::prefix('users')->name('users.')->group(function () {
            Route::get('/', [AdminUserController::class, 'index'])->name('index');
            Route::get('/{user}', [AdminUserController::class, 'show'])->name('show');
            Route::post('/{user}/toggle-block', [AdminUserController::class, 'toggleBlock'])->name('toggle-block');
            Route::post('/{user}/toggle-active', [AdminUserController::class, 'toggleActive'])->name('toggle-active');
            Route::post('/{user}/verify', [AdminUserController::class, 'verify'])->name('verify');
            Route::post('/{user}/reset-password', [AdminUserController::class, 'resetPassword'])->name('reset-password');
            Route::post('/{user}/reset-2fa', [AdminUserController::class, 'reset2FA'])->name('reset-2fa');
            Route::delete('/{user}', [AdminUserController::class, 'destroy'])->name('destroy');
            
            // Wallet Management
            Route::get('/{user}/wallet', [AdminUserController::class, 'wallet'])->name('wallet');
            Route::post('/{user}/adjust-balance', [AdminUserController::class, 'adjustBalance'])->name('adjust-balance');
            Route::post('/{user}/toggle-wallet-freeze', [AdminUserController::class, 'toggleWalletFreeze'])->name('toggle-wallet-freeze');
        });
        
        // KYC Management
        Route::prefix('kyc')->name('kyc.')->group(function () {
            Route::get('/', [AdminKycController::class, 'index'])->name('index');
            Route::get('/analytics', [AdminKycController::class, 'analytics'])->name('analytics');
            Route::get('/{kycVerification}', [AdminKycController::class, 'show'])->name('show');
            Route::post('/{kycVerification}/approve', [AdminKycController::class, 'approve'])->name('approve');
            Route::post('/{kycVerification}/reject', [AdminKycController::class, 'reject'])->name('reject');
            Route::post('/{kycVerification}/reset', [AdminKycController::class, 'reset'])->name('reset');
            Route::get('/{kycVerification}/download', [AdminKycController::class, 'downloadDocument'])->name('download');
            Route::post('/bulk-approve', [AdminKycController::class, 'bulkApprove'])->name('bulk-approve');
            Route::post('/bulk-reject', [AdminKycController::class, 'bulkReject'])->name('bulk-reject');
        });
        
        // Virtual Cards Management
        Route::prefix('virtual-cards')->name('virtual-cards.')->group(function () {
            Route::get('/', [App\Http\Controllers\Admin\VirtualCardController::class, 'index'])->name('index');
            Route::get('/analytics', [App\Http\Controllers\Admin\VirtualCardController::class, 'analytics'])->name('analytics');
            Route::post('/assign', [App\Http\Controllers\Admin\VirtualCardController::class, 'assign'])->name('assign');
            Route::get('/{virtualCard}', [App\Http\Controllers\Admin\VirtualCardController::class, 'show'])->name('show');
            Route::post('/{virtualCard}/activate', [App\Http\Controllers\Admin\VirtualCardController::class, 'activate'])->name('activate');
            Route::post('/{virtualCard}/deactivate', [App\Http\Controllers\Admin\VirtualCardController::class, 'deactivate'])->name('deactivate');
            Route::post('/{virtualCard}/freeze', [App\Http\Controllers\Admin\VirtualCardController::class, 'freeze'])->name('freeze');
            Route::post('/{virtualCard}/unfreeze', [App\Http\Controllers\Admin\VirtualCardController::class, 'unfreeze'])->name('unfreeze');
            Route::post('/{virtualCard}/terminate', [App\Http\Controllers\Admin\VirtualCardController::class, 'terminate'])->name('terminate');
            Route::post('/{virtualCard}/reissue', [App\Http\Controllers\Admin\VirtualCardController::class, 'reissue'])->name('reissue');
            Route::post('/{virtualCard}/credit', [App\Http\Controllers\Admin\VirtualCardController::class, 'credit'])->name('credit');
            Route::post('/{virtualCard}/debit', [App\Http\Controllers\Admin\VirtualCardController::class, 'debit'])->name('debit');
            Route::put('/{virtualCard}/limits', [App\Http\Controllers\Admin\VirtualCardController::class, 'updateLimits'])->name('update-limits');
            Route::put('/{virtualCard}/fees', [App\Http\Controllers\Admin\VirtualCardController::class, 'updateFees'])->name('update-fees');
        });
        
        // Wallet Management
        Route::prefix('wallets')->name('wallets.')->group(function () {
            Route::get('/', [App\Http\Controllers\Admin\WalletController::class, 'index'])->name('index');
            Route::get('/transactions', [App\Http\Controllers\Admin\WalletController::class, 'transactions'])->name('transactions');
            Route::get('/pending-withdrawals', [App\Http\Controllers\Admin\WalletController::class, 'pendingWithdrawals'])->name('pending-withdrawals');
            Route::post('/{wallet}/adjust-balance', [App\Http\Controllers\Admin\WalletController::class, 'adjustBalance'])->name('adjust-balance');
            Route::post('/{wallet}/toggle-freeze', [App\Http\Controllers\Admin\WalletController::class, 'toggleFreeze'])->name('toggle-freeze');
            Route::post('/withdrawals/{transaction}/approve', [App\Http\Controllers\Admin\WalletController::class, 'approveWithdrawal'])->name('approve-withdrawal');
            Route::post('/withdrawals/{transaction}/decline', [App\Http\Controllers\Admin\WalletController::class, 'declineWithdrawal'])->name('decline-withdrawal');
        });
        
        // API Configuration
        Route::prefix('api-config')->name('api-config.')->group(function () {
            Route::get('/', [App\Http\Controllers\Admin\ApiConfigController::class, 'index'])->name('index');
            Route::get('/logs', [App\Http\Controllers\Admin\ApiConfigController::class, 'logs'])->name('logs');
            Route::get('/create', [App\Http\Controllers\Admin\ApiConfigController::class, 'create'])->name('create');
            Route::post('/', [App\Http\Controllers\Admin\ApiConfigController::class, 'store'])->name('store');
            Route::get('/{apiConfig}', [App\Http\Controllers\Admin\ApiConfigController::class, 'show'])->name('show');
            Route::put('/{apiConfig}', [App\Http\Controllers\Admin\ApiConfigController::class, 'update'])->name('update');
            Route::post('/{apiConfig}/test', [App\Http\Controllers\Admin\ApiConfigController::class, 'testConnection'])->name('test');
            Route::post('/{apiConfig}/toggle-status', [App\Http\Controllers\Admin\ApiConfigController::class, 'toggleStatus'])->name('toggle-status');
            Route::post('/{apiConfig}/toggle-mode', [App\Http\Controllers\Admin\ApiConfigController::class, 'toggleMode'])->name('toggle-mode');
        });
        
        // Airtime & Data Management
        Route::prefix('airtime')->name('airtime.')->group(function () {
            Route::get('/orders', [App\Http\Controllers\Admin\AirtimeController::class, 'orders'])->name('orders');
            Route::get('/failed', [App\Http\Controllers\Admin\AirtimeController::class, 'failed'])->name('failed');
            Route::get('/providers', [App\Http\Controllers\Admin\AirtimeController::class, 'providers'])->name('providers');
            Route::get('/providers/create', [App\Http\Controllers\Admin\AirtimeController::class, 'createProvider'])->name('providers.create');
            Route::post('/providers', [App\Http\Controllers\Admin\AirtimeController::class, 'storeProvider'])->name('providers.store');
            Route::get('/providers/{provider}/edit', [App\Http\Controllers\Admin\AirtimeController::class, 'editProvider'])->name('providers.edit');
            Route::put('/providers/{provider}', [App\Http\Controllers\Admin\AirtimeController::class, 'updateProvider'])->name('providers.update');
            Route::delete('/providers/{provider}', [App\Http\Controllers\Admin\AirtimeController::class, 'deleteProvider'])->name('providers.delete');
            Route::post('/providers/{provider}/test', [App\Http\Controllers\Admin\AirtimeController::class, 'testProvider'])->name('providers.test');
            Route::get('/margins', [App\Http\Controllers\Admin\AirtimeController::class, 'margins'])->name('margins');
            Route::post('/margins', [App\Http\Controllers\Admin\AirtimeController::class, 'updateMargins'])->name('margins.update');
            Route::post('/{transaction}/resend', [App\Http\Controllers\Admin\AirtimeController::class, 'resendTransaction'])->name('resend');
        });
        
        // Settings Management
        Route::prefix('settings')->name('settings.')->group(function () {
            Route::get('/fees', [App\Http\Controllers\Admin\SettingsController::class, 'fees'])->name('fees');
            Route::post('/fees', [App\Http\Controllers\Admin\SettingsController::class, 'updateFees'])->name('fees.update');
            Route::get('/site', [App\Http\Controllers\Admin\SettingsController::class, 'site'])->name('site');
            Route::post('/site', [App\Http\Controllers\Admin\SettingsController::class, 'updateSiteSettings'])->name('site.update');
            Route::get('/notifications', [App\Http\Controllers\Admin\SettingsController::class, 'notifications'])->name('notifications');
            Route::post('/notifications/setting', [App\Http\Controllers\Admin\SettingsController::class, 'updateNotificationSettings'])->name('notifications.setting');
            Route::post('/notifications/broadcast', [App\Http\Controllers\Admin\SettingsController::class, 'sendBroadcast'])->name('notifications.broadcast');
            Route::post('/notifications/test', [App\Http\Controllers\Admin\SettingsController::class, 'testNotifications'])->name('notifications.test');
            Route::post('/notifications/smtp', [App\Http\Controllers\Admin\SettingsController::class, 'updateSmtpConfig'])->name('notifications.smtp');
            Route::post('/notifications/smtp/test', [App\Http\Controllers\Admin\SettingsController::class, 'testSmtpConnection'])->name('notifications.smtp.test');
            Route::post('/notifications/smtp/send-test', [App\Http\Controllers\Admin\SettingsController::class, 'sendTestEmail'])->name('notifications.smtp.send-test');
            Route::get('/security-logs', [App\Http\Controllers\Admin\SettingsController::class, 'securityLogs'])->name('security-logs');
            Route::post('/security-logs/export', [App\Http\Controllers\Admin\SettingsController::class, 'exportLogs'])->name('security-logs.export');
            Route::delete('/security-logs/clear', [App\Http\Controllers\Admin\SettingsController::class, 'clearOldLogs'])->name('security-logs.clear');
            Route::get('/security-logs/{log}/details', [App\Http\Controllers\Admin\SettingsController::class, 'getLogDetails'])->name('security-logs.details');
            Route::post('/security-logs/block-ip', [App\Http\Controllers\Admin\SettingsController::class, 'blockIP'])->name('security-logs.block-ip');
            Route::get('/pages', [App\Http\Controllers\Admin\SettingsController::class, 'pages'])->name('pages');
            Route::post('/pages', [App\Http\Controllers\Admin\SettingsController::class, 'storePage'])->name('pages.store');
            Route::get('/pages/export', [App\Http\Controllers\Admin\SettingsController::class, 'exportPages'])->name('pages.export');
            Route::get('/pages/{page}', [App\Http\Controllers\Admin\SettingsController::class, 'showPage'])->name('pages.show');
            Route::get('/pages/{page}/edit', [App\Http\Controllers\Admin\SettingsController::class, 'editPage'])->name('pages.edit');
            Route::put('/pages/{page}', [App\Http\Controllers\Admin\SettingsController::class, 'updatePage'])->name('pages.update');
            Route::patch('/pages/{page}/status', [App\Http\Controllers\Admin\SettingsController::class, 'togglePageStatus'])->name('pages.status');
            Route::post('/pages/{page}/duplicate', [App\Http\Controllers\Admin\SettingsController::class, 'duplicatePage'])->name('pages.duplicate');
            Route::delete('/pages/{page}', [App\Http\Controllers\Admin\SettingsController::class, 'destroyPage'])->name('pages.destroy');
            Route::get('/modules', [App\Http\Controllers\Admin\SettingsController::class, 'modules'])->name('modules');
            Route::post('/modules', [App\Http\Controllers\Admin\SettingsController::class, 'updateModules'])->name('modules.update');
            Route::get('/providers/{category}', [App\Http\Controllers\Admin\SettingsController::class, 'getAvailableProviders'])->name('providers.category');
        });
        
        // Authentication
        Route::post('/logout', [AdminAuthController::class, 'logout'])->name('logout');
    });
});

// Authenticated User Routes
Route::middleware(['auth', 'kyc.verified'])->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    
    // Wallet Routes
    Route::prefix('wallet')->group(function () {
        Route::get('/', [WalletController::class, 'index'])->name('wallet.index');
        Route::post('/fund', [WalletController::class, 'fund'])->name('wallet.fund');
        Route::get('/transactions', [TransactionController::class, 'index'])->name('transactions.index');
    });
    
    // Virtual Account Routes
    Route::prefix('virtual-account')->group(function () {
        Route::get('/', [VirtualAccountController::class, 'index'])->name('virtual-account.index');
        Route::post('/create', [VirtualAccountController::class, 'create'])->name('virtual-account.create');
    });
    
    // Transfer Routes
    Route::prefix('transfer')->group(function () {
        Route::get('/p2p', [TransactionController::class, 'showP2PForm'])->name('transfer.p2p');
        Route::post('/p2p', [TransactionController::class, 'p2pTransfer']);
        Route::get('/bank', [TransactionController::class, 'showBankTransferForm'])->name('transfer.bank');
        Route::post('/bank', [TransactionController::class, 'bankTransfer']);
    });
    
    // Bill Payment Routes
    Route::prefix('bills')->group(function () {
        Route::get('/', [BillPaymentController::class, 'index'])->name('bills.index');
        Route::get('/airtime', [BillPaymentController::class, 'showAirtimeForm'])->name('bills.airtime');
        Route::post('/airtime', [BillPaymentController::class, 'purchaseAirtime']);
        Route::get('/data', [BillPaymentController::class, 'showDataForm'])->name('bills.data');
        Route::post('/data', [BillPaymentController::class, 'purchaseData']);
        Route::get('/electricity', [BillPaymentController::class, 'showElectricityForm'])->name('bills.electricity');
        Route::post('/electricity', [BillPaymentController::class, 'payElectricity']);
    });
    
    // Virtual Card Routes
    Route::prefix('cards')->group(function () {
        Route::get('/', [VirtualCardController::class, 'index'])->name('cards.index');
        Route::post('/create', [VirtualCardController::class, 'create'])->name('cards.create');
        Route::post('/{card}/freeze', [VirtualCardController::class, 'freeze'])->name('cards.freeze');
        Route::post('/{card}/unfreeze', [VirtualCardController::class, 'unfreeze'])->name('cards.unfreeze');
    });
    
    // Settings Routes
    Route::prefix('settings')->name('settings.')->group(function () {
        Route::get('/', [SettingsController::class, 'index'])->name('index');
        Route::get('/profile', [SettingsController::class, 'profile'])->name('profile');
        Route::post('/profile', [SettingsController::class, 'updateProfile'])->name('profile.update');
        Route::get('/security', [SettingsController::class, 'security'])->name('security');
        Route::post('/password', [SettingsController::class, 'updatePassword'])->name('password.update');
        Route::post('/transaction-pin', [SettingsController::class, 'updateTransactionPin'])->name('transaction-pin.update');
        Route::get('/notifications', [SettingsController::class, 'notifications'])->name('notifications');
        Route::post('/notifications', [SettingsController::class, 'updateNotifications'])->name('notifications.update');
        Route::get('/api', [SettingsController::class, 'api'])->name('api');
        Route::post('/api/generate', [SettingsController::class, 'generateApiKey'])->name('api.generate');
        Route::delete('/api/{token}', [SettingsController::class, 'revokeApiKey'])->name('api.revoke');
        Route::post('/logout-all', [SettingsController::class, 'logoutAllDevices'])->name('logout-all');
    });
});

// KYC Routes (accessible to authenticated users without KYC verification)
Route::middleware('auth')->group(function () {
    Route::prefix('kyc')->group(function () {
        Route::get('/', [KycController::class, 'index'])->name('kyc.index');
        Route::get('/bvn', [KycController::class, 'showBvnForm'])->name('kyc.bvn');
        Route::post('/bvn', [KycController::class, 'verifyBvn']);
        Route::get('/nin', [KycController::class, 'showNinForm'])->name('kyc.nin');
        Route::post('/nin', [KycController::class, 'verifyNin']);
        Route::get('/document', [KycController::class, 'showDocumentForm'])->name('kyc.document');
        Route::post('/document', [KycController::class, 'uploadDocument']);
    });
});

// Public pages - this should be last to avoid conflicts with other routes
Route::get('/{slug}', [App\Http\Controllers\PageController::class, 'show'])
    ->where('slug', '[a-z0-9\-]+')
    ->name('page.show');