<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\VirtualCard;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

class VirtualCardAnalyticsRouteTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Migrate the database
        $this->artisan('migrate');
    }

    /** @test */
    public function test_analytics_route_without_authentication()
    {
        // Test that the route exists and requires authentication
        $response = $this->get('/admin/virtual-cards/analytics');
        
        // Should redirect to login (302) or return unauthorized (401)
        $this->assertTrue(in_array($response->getStatusCode(), [302, 401]));
    }

    /** @test */
    public function test_direct_analytics_method_renders_view_content()
    {
        // Create some test data
        $user = User::factory()->create();
        
        VirtualCard::factory()->create([
            'user_id' => $user->id,
            'status' => 'active',
            'card_type' => 'virtual',
        ]);
        
        VirtualCard::factory()->create([
            'user_id' => $user->id,
            'status' => 'frozen',
            'card_type' => 'physical',
        ]);

        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        $response = $controller->analytics();
        
        // Verify it returns the correct view
        $this->assertInstanceOf(\Illuminate\View\View::class, $response);
        $this->assertEquals('admin.virtual-cards.analytics', $response->getName());
        
        // Verify the data structure and content
        $data = $response->getData();
        $this->assertArrayHasKey('analytics', $data);
        $analytics = $data['analytics'];
        
        // Should contain the data we created
        $this->assertEquals(1, $analytics['cards_by_status']['active']);
        $this->assertEquals(1, $analytics['cards_by_status']['frozen']);
        $this->assertEquals(1, $analytics['cards_by_type']['virtual']);
        $this->assertEquals(1, $analytics['cards_by_type']['physical']);
    }

    /** @test */
    public function test_analytics_view_handles_empty_state()
    {
        // No data created - test empty state
        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        $response = $controller->analytics();
        
        // Verify it returns the correct view
        $this->assertInstanceOf(\Illuminate\View\View::class, $response);
        $this->assertEquals('admin.virtual-cards.analytics', $response->getName());
        
        // Verify empty data structure
        $data = $response->getData();
        $analytics = $data['analytics'];
        
        // All arrays should be empty but still arrays
        $this->assertIsArray($analytics['cards_by_status']);
        $this->assertIsArray($analytics['cards_by_type']);
        $this->assertIsArray($analytics['monthly_creations']);
        $this->assertIsArray($analytics['transaction_volume']);
        
        $this->assertEmpty($analytics['cards_by_status']);
        $this->assertEmpty($analytics['cards_by_type']);
        $this->assertEmpty($analytics['monthly_creations']);
        $this->assertEmpty($analytics['transaction_volume']);
    }
}