<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Models\Wallet;
use App\Models\Transaction;
use App\Services\PaystackService;
use App\Services\MonnifyService;

/**
 * Wallet API Controller
 * 
 * TODO: Add comprehensive input validation
 * TODO: Add wallet security features (PIN verification)
 * TODO: Add transaction limits and controls
 * TODO: Add fraud detection for wallet operations
 * TODO: Add multi-currency wallet support
 */
class WalletController extends Controller
{
    protected $paystackService;
    protected $monnifyService;

    public function __construct(PaystackService $paystackService, MonnifyService $monnifyService)
    {
        $this->paystackService = $paystackService;
        $this->monnifyService = $monnifyService;
    }

    /**
     * Get wallet balance and information
     * Enhanced to match frontend expectations
     */
    public function getBalance(Request $request)
    {
        try {
            $user = $request->user();
            $wallet = $user->wallet;

            if (!$wallet) {
                // Create wallet if doesn't exist
                $wallet = Wallet::create([
                    'user_id' => $user->id,
                    'balance' => 0,
                    'available_balance' => 0,
                    'locked_balance' => 0,
                    'currency' => 'NGN',
                    'status' => 'active',
                    'daily_limit' => 100000,
                    'remaining_daily_limit' => 100000,
                ]);
            }

            return response()->json([
                'success' => true,
                'status' => 'success',
                'message' => 'Wallet balance retrieved successfully',
                'data' => [
                    'balance' => $wallet->balance,
                    'available_balance' => $wallet->available_balance ?? $wallet->balance,
                    'locked_balance' => $wallet->locked_balance ?? 0,
                    'currency' => $wallet->currency,
                    'status' => $wallet->status,
                    'daily_limit' => $wallet->daily_limit ?? 100000,
                    'remaining_daily_limit' => $wallet->remaining_daily_limit ?? $wallet->daily_limit ?? 100000,
                    'is_frozen' => $wallet->is_frozen ?? false,
                    'last_updated' => $wallet->updated_at?->toISOString(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'status' => 'error',
                'message' => 'Failed to retrieve wallet balance',
                'error' => config('app.debug') ? $e->getMessage() : 'An error occurred'
            ], 500);
        }
    }

    /**
     * Fund wallet
     * 
     * TODO: Add funding source validation
     * TODO: Add funding limits per user
     * TODO: Add KYC level-based funding limits
     * TODO: Add automatic fraud detection
     */
    public function fund(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'amount' => 'required|numeric|min:100|max:1000000',
                'funding_source' => 'required|string|in:card,bank_transfer,virtual_account',
                'card_details' => 'required_if:funding_source,card|array',
                'card_details.number' => 'required_if:funding_source,card|string',
                'card_details.expiry_month' => 'required_if:funding_source,card|string|size:2',
                'card_details.expiry_year' => 'required_if:funding_source,card|string|size:4',
                'card_details.cvv' => 'required_if:funding_source,card|string|size:3',
                // TODO: Add more validation rules
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();
            $wallet = $user->wallet;

            if (!$wallet) {
                $wallet = Wallet::create([
                    'user_id' => $user->id,
                    'balance' => 0,
                    'currency' => 'NGN',
                    'status' => 'active'
                ]);
            }

            // TODO: Check user's daily/monthly funding limits
            // TODO: Add fraud detection checks

            $fundingData = [
                'amount' => $request->amount,
                'email' => $user->email,
                'reference' => 'FUND_' . time() . '_' . $user->id,
                'callback_url' => config('app.url') . '/api/webhooks/paystack'
            ];

            if ($request->funding_source === 'card') {
                $fundingData['card'] = $request->card_details;
                $response = $this->paystackService->chargeCard($fundingData);
            } else {
                $response = $this->paystackService->initializeTransaction($fundingData);
            }

            if (!$response['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to initialize funding',
                    'error' => $response['message']
                ], 500);
            }

            // Create pending transaction record
            $transaction = Transaction::create([
                'user_id' => $user->id,
                'type' => 'wallet_funding',
                'category' => 'credit',
                'amount' => $request->amount,
                'description' => 'Wallet funding',
                'reference' => $fundingData['reference'],
                'provider_reference' => $response['data']['reference'],
                'status' => 'pending',
                'metadata' => json_encode([
                    'funding_source' => $request->funding_source,
                    'provider' => 'paystack'
                ])
            ]);

            // TODO: Send funding notification
            // TODO: Log funding attempt

            return response()->json([
                'status' => 'success',
                'message' => 'Wallet funding initialized successfully',
                'data' => [
                    'transaction' => $transaction,
                    'payment_url' => $response['data']['authorization_url'] ?? null,
                    'provider_response' => $response['data']
                ]
            ], 201);

        } catch (\Exception $e) {
            // TODO: Add proper error logging
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fund wallet',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get wallet transaction history
     * 
     * TODO: Add transaction filtering and search
     * TODO: Add pagination
     * TODO: Add transaction analytics
     */
    public function getTransactions(Request $request)
    {
        try {
            $user = $request->user();
            
            $query = Transaction::where('user_id', $user->id);

            // Add filters if provided
            if ($request->has('type')) {
                $query->where('type', $request->type);
            }

            if ($request->has('category')) {
                $query->where('category', $request->category);
            }

            if ($request->has('status')) {
                $query->where('status', $request->status);
            }

            if ($request->has('date_from')) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }

            if ($request->has('date_to')) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            $transactions = $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 20));

            return response()->json([
                'status' => 'success',
                'message' => 'Wallet transactions retrieved successfully',
                'data' => $transactions
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve wallet transactions',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Transfer between wallets (P2P)
     * 
     * TODO: Add recipient validation
     * TODO: Add transfer limits
     * TODO: Add transaction PIN verification
     * TODO: Add fraud detection
     */
    public function transfer(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'recipient_phone' => 'required|string|regex:/^(\+234|234|0)[789][01]\d{8}$/',
                'amount' => 'required|numeric|min:100|max:1000000',
                'description' => 'nullable|string|max:255',
                'transaction_pin' => 'required|string|size:4',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();
            $wallet = $user->wallet;

            // TODO: Verify transaction PIN

            // Find recipient
            $recipient = User::where('phone', $request->recipient_phone)
                ->where('kyc_verified', true)
                ->first();

            if (!$recipient) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Recipient not found or not verified'
                ], 404);
            }

            if ($recipient->id === $user->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Cannot transfer to yourself'
                ], 400);
            }

            // Check balance
            if ($wallet->balance < $request->amount) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Insufficient wallet balance'
                ], 400);
            }

            // TODO: Add fraud detection checks
            // TODO: Check transfer limits

            // Process transfer
            $reference = 'TRF_' . time() . '_' . $user->id;

            // Debit sender
            $wallet->decrement('balance', $request->amount);

            // Credit recipient
            $recipient->wallet->increment('balance', $request->amount);

            // Create sender transaction
            $senderTransaction = Transaction::create([
                'user_id' => $user->id,
                'type' => 'p2p_transfer',
                'category' => 'debit',
                'amount' => $request->amount,
                'description' => $request->description ?? "Transfer to {$recipient->first_name}",
                'reference' => $reference,
                'status' => 'completed',
                'metadata' => json_encode([
                    'recipient_user_id' => $recipient->id,
                    'recipient_phone' => $recipient->phone
                ])
            ]);

            // Create recipient transaction
            $recipientTransaction = Transaction::create([
                'user_id' => $recipient->id,
                'type' => 'p2p_transfer',
                'category' => 'credit',
                'amount' => $request->amount,
                'description' => "Transfer from {$user->first_name}",
                'reference' => $reference,
                'status' => 'completed',
                'metadata' => json_encode([
                    'sender_id' => $user->id,
                    'sender_phone' => $user->phone
                ])
            ]);

            // TODO: Send transfer notifications to both parties
            // TODO: Log transfer event

            return response()->json([
                'status' => 'success',
                'message' => 'Transfer completed successfully',
                'data' => [
                    'transaction' => $senderTransaction,
                    'new_balance' => $wallet->fresh()->balance
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Transfer failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Set wallet PIN
     * 
     * TODO: Add PIN strength validation
     * TODO: Add PIN change history
     */
    public function setPin(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'pin' => 'required|string|size:4|regex:/^[0-9]+$/',
                'confirm_pin' => 'required|same:pin',
                'current_password' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();

            // Verify current password
            if (!Hash::check($request->current_password, $user->password)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Current password is incorrect'
                ], 400);
            }

            // TODO: Check PIN strength and patterns
            // TODO: Prevent common PINs (1234, 0000, etc.)

            $user->update([
                'transaction_pin' => Hash::make($request->pin)
            ]);

            // TODO: Log PIN change event
            // TODO: Send PIN change notification

            return response()->json([
                'status' => 'success',
                'message' => 'Transaction PIN set successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to set transaction PIN',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}