<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

/**
 * BVN Service for Bank Verification Number validation
 * 
 * TODO: Add comprehensive error handling
 * TODO: Add data privacy compliance
 * TODO: Add retry mechanisms
 * TODO: Add caching for verified BVNs
 */
class BvnService
{
    protected $baseUrl;
    protected $apiKey;
    protected $provider;

    public function __construct()
    {
        $this->baseUrl = config('services.bvn.base_url', 'https://api.verified.africa');
        $this->apiKey = config('services.bvn.api_key');
        $this->provider = config('services.bvn.provider', 'verified');
    }

    /**
     * Verify BVN with personal details
     * 
     * TODO: Add data masking for responses
     * TODO: Add compliance audit logging
     * TODO: Add rate limiting
     */
    public function verifyBvn(array $data)
    {
        try {
            $payload = [
                'bvn' => $data['bvn'],
                'first_name' => $data['first_name'],
                'last_name' => $data['last_name'],
                'date_of_birth' => $data['date_of_birth'],
                'phone' => $data['phone'] ?? null
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json'
            ])->post($this->baseUrl . '/v3/kyc/bvn', $payload);

            if ($response->successful()) {
                $responseData = $response->json();
                
                if ($responseData['status'] === 'success' && $responseData['data']['is_verified']) {
                    return [
                        'success' => true,
                        'data' => [
                            'bvn' => $data['bvn'],
                            'first_name' => $responseData['data']['first_name'],
                            'last_name' => $responseData['data']['last_name'],
                            'middle_name' => $responseData['data']['middle_name'] ?? null,
                            'date_of_birth' => $responseData['data']['date_of_birth'],
                            'phone' => $responseData['data']['phone'],
                            'gender' => $responseData['data']['gender'] ?? null,
                            'verified_at' => now(),
                            'match_score' => $responseData['data']['match_score'] ?? 100
                        ]
                    ];
                }

                return [
                    'success' => false,
                    'message' => 'BVN verification failed - details do not match',
                    'errors' => $responseData['data']['mismatches'] ?? []
                ];
            }

            // TODO: Handle specific error codes
            $errorMessage = $response->json()['message'] ?? 'BVN verification service unavailable';
            
            return [
                'success' => false,
                'message' => $errorMessage,
                'error_code' => $response->status()
            ];

        } catch (Exception $e) {
            Log::error('BVN verification failed', [
                'error' => $e->getMessage(),
                'bvn' => substr($data['bvn'], 0, 3) . '********' // Mask BVN in logs
            ]);

            return [
                'success' => false,
                'message' => 'BVN verification service temporarily unavailable'
            ];
        }
    }

    /**
     * Get BVN details without verification
     * 
     * TODO: Add permission checks
     * TODO: Add audit logging
     */
    public function getBvnDetails(string $bvn)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey
            ])->get($this->baseUrl . "/v3/kyc/bvn/{$bvn}");

            if ($response->successful()) {
                $responseData = $response->json();
                
                return [
                    'success' => true,
                    'data' => $responseData['data']
                ];
            }

            return [
                'success' => false,
                'message' => $response->json()['message'] ?? 'BVN not found'
            ];

        } catch (Exception $e) {
            Log::error('BVN details retrieval failed', [
                'error' => $e->getMessage(),
                'bvn' => substr($bvn, 0, 3) . '********'
            ]);

            return [
                'success' => false,
                'message' => 'Service temporarily unavailable'
            ];
        }
    }

    /**
     * Verify multiple BVNs (batch processing)
     * 
     * TODO: Add batch size limits
     * TODO: Add progress tracking
     */
    public function verifyBvnBatch(array $bvnData)
    {
        try {
            $payload = [
                'requests' => $bvnData
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json'
            ])->post($this->baseUrl . '/v3/kyc/bvn/batch', $payload);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()['data']
                ];
            }

            return [
                'success' => false,
                'message' => $response->json()['message'] ?? 'Batch verification failed'
            ];

        } catch (Exception $e) {
            Log::error('BVN batch verification failed', [
                'error' => $e->getMessage(),
                'batch_size' => count($bvnData)
            ]);

            return [
                'success' => false,
                'message' => 'Service temporarily unavailable'
            ];
        }
    }

    /**
     * Check if BVN is valid format
     * 
     * TODO: Add advanced BVN format validation
     */
    public function isValidBvnFormat(string $bvn)
    {
        // BVN should be 11 digits
        return preg_match('/^[0-9]{11}$/', $bvn);
    }

    /**
     * Check if BVN exists (without getting details)
     * 
     * TODO: Add caching for existence checks
     */
    public function bvnExists(string $bvn)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey
            ])->head($this->baseUrl . "/v3/kyc/bvn/{$bvn}");

            return $response->status() === 200;

        } catch (Exception $e) {
            Log::error('BVN existence check failed', [
                'error' => $e->getMessage(),
                'bvn' => substr($bvn, 0, 3) . '********'
            ]);

            return false;
        }
    }

    /**
     * Get verification cost
     * 
     * TODO: Add cost calculation based on verification type
     */
    public function getVerificationCost(string $verificationType = 'basic')
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey
            ])->get($this->baseUrl . '/v3/pricing/bvn', [
                'type' => $verificationType
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()['data']
                ];
            }

            return [
                'success' => false,
                'message' => 'Unable to get pricing information'
            ];

        } catch (Exception $e) {
            Log::error('BVN pricing check failed', [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Service temporarily unavailable'
            ];
        }
    }
}