<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\WalletController;
use App\Http\Controllers\TransactionController;
use App\Http\Controllers\KycController;
use App\Http\Controllers\VirtualAccountController;
use App\Http\Controllers\BillPaymentController;
use App\Http\Controllers\VirtualCardController;
use App\Http\Controllers\SettingsController;
use App\Http\Controllers\Admin\AuthController as AdminAuthController;
use App\Http\Controllers\Admin\DashboardController as AdminDashboardController;
use App\Http\Controllers\Admin\UserController as AdminUserController;
use App\Http\Controllers\Admin\KycController as AdminKycController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

Route::get('/', function () {
    return view('welcome');
});

// Authentication Routes
Route::middleware('guest')->group(function () {
    Route::get('/login', [AuthController::class, 'showLoginForm'])->name('login');
    Route::post('/login', [AuthController::class, 'login']);
    Route::get('/register', [AuthController::class, 'showRegistrationForm'])->name('register');
    Route::post('/register', [AuthController::class, 'register']);
});

// Phone verification (accessible to authenticated users who haven't verified phone)
Route::middleware('auth')->group(function () {
    Route::get('/verify-phone', [AuthController::class, 'showVerifyPhone'])->name('verify.phone');
    Route::post('/verify-phone', [AuthController::class, 'verifyPhone']);
});

Route::post('/logout', [AuthController::class, 'logout'])->name('logout');

// Admin Authentication Routes
Route::prefix('admin')->name('admin.')->group(function () {
    Route::middleware('guest:admin')->group(function () {
        Route::get('/login', [AdminAuthController::class, 'showLoginForm'])->name('login');
        Route::post('/login', [AdminAuthController::class, 'login']);
    });
    
    Route::middleware(['auth:admin', 'admin'])->group(function () {
        // Dashboard
        Route::get('/dashboard', [AdminDashboardController::class, 'index'])->name('dashboard');
        Route::get('/alerts', [AdminDashboardController::class, 'getAlerts'])->name('alerts');
        
        // User Management
        Route::prefix('users')->name('users.')->group(function () {
            Route::get('/', [AdminUserController::class, 'index'])->name('index');
            Route::get('/{user}', [AdminUserController::class, 'show'])->name('show');
            Route::post('/{user}/toggle-block', [AdminUserController::class, 'toggleBlock'])->name('toggle-block');
            Route::post('/{user}/toggle-active', [AdminUserController::class, 'toggleActive'])->name('toggle-active');
            Route::post('/{user}/verify', [AdminUserController::class, 'verify'])->name('verify');
            Route::post('/{user}/reset-password', [AdminUserController::class, 'resetPassword'])->name('reset-password');
            Route::post('/{user}/reset-2fa', [AdminUserController::class, 'reset2FA'])->name('reset-2fa');
            Route::delete('/{user}', [AdminUserController::class, 'destroy'])->name('destroy');
            
            // Wallet Management
            Route::get('/{user}/wallet', [AdminUserController::class, 'wallet'])->name('wallet');
            Route::post('/{user}/adjust-balance', [AdminUserController::class, 'adjustBalance'])->name('adjust-balance');
            Route::post('/{user}/toggle-wallet-freeze', [AdminUserController::class, 'toggleWalletFreeze'])->name('toggle-wallet-freeze');
        });
        
        // KYC Management
        Route::prefix('kyc')->name('kyc.')->group(function () {
            Route::get('/', [AdminKycController::class, 'index'])->name('index');
            Route::get('/analytics', [AdminKycController::class, 'analytics'])->name('analytics');
            Route::get('/{kycVerification}', [AdminKycController::class, 'show'])->name('show');
            Route::post('/{kycVerification}/approve', [AdminKycController::class, 'approve'])->name('approve');
            Route::post('/{kycVerification}/reject', [AdminKycController::class, 'reject'])->name('reject');
            Route::post('/{kycVerification}/reset', [AdminKycController::class, 'reset'])->name('reset');
            Route::get('/{kycVerification}/download', [AdminKycController::class, 'downloadDocument'])->name('download');
            Route::post('/bulk-approve', [AdminKycController::class, 'bulkApprove'])->name('bulk-approve');
            Route::post('/bulk-reject', [AdminKycController::class, 'bulkReject'])->name('bulk-reject');
        });
        
        // Authentication
        Route::post('/logout', [AdminAuthController::class, 'logout'])->name('logout');
    });
});

// Authenticated User Routes
Route::middleware(['auth', 'kyc.verified'])->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    
    // Wallet Routes
    Route::prefix('wallet')->group(function () {
        Route::get('/', [WalletController::class, 'index'])->name('wallet.index');
        Route::post('/fund', [WalletController::class, 'fund'])->name('wallet.fund');
        Route::get('/transactions', [TransactionController::class, 'index'])->name('transactions.index');
    });
    
    // Virtual Account Routes
    Route::prefix('virtual-account')->group(function () {
        Route::get('/', [VirtualAccountController::class, 'index'])->name('virtual-account.index');
        Route::post('/create', [VirtualAccountController::class, 'create'])->name('virtual-account.create');
    });
    
    // Transfer Routes
    Route::prefix('transfer')->group(function () {
        Route::get('/p2p', [TransactionController::class, 'showP2PForm'])->name('transfer.p2p');
        Route::post('/p2p', [TransactionController::class, 'p2pTransfer']);
        Route::get('/bank', [TransactionController::class, 'showBankTransferForm'])->name('transfer.bank');
        Route::post('/bank', [TransactionController::class, 'bankTransfer']);
    });
    
    // Bill Payment Routes
    Route::prefix('bills')->group(function () {
        Route::get('/', [BillPaymentController::class, 'index'])->name('bills.index');
        Route::get('/airtime', [BillPaymentController::class, 'showAirtimeForm'])->name('bills.airtime');
        Route::post('/airtime', [BillPaymentController::class, 'purchaseAirtime']);
        Route::get('/data', [BillPaymentController::class, 'showDataForm'])->name('bills.data');
        Route::post('/data', [BillPaymentController::class, 'purchaseData']);
        Route::get('/electricity', [BillPaymentController::class, 'showElectricityForm'])->name('bills.electricity');
        Route::post('/electricity', [BillPaymentController::class, 'payElectricity']);
    });
    
    // Virtual Card Routes
    Route::prefix('cards')->group(function () {
        Route::get('/', [VirtualCardController::class, 'index'])->name('cards.index');
        Route::post('/create', [VirtualCardController::class, 'create'])->name('cards.create');
        Route::post('/{card}/freeze', [VirtualCardController::class, 'freeze'])->name('cards.freeze');
        Route::post('/{card}/unfreeze', [VirtualCardController::class, 'unfreeze'])->name('cards.unfreeze');
    });
    
    // Settings Routes
    Route::prefix('settings')->name('settings.')->group(function () {
        Route::get('/', [SettingsController::class, 'index'])->name('index');
        Route::get('/profile', [SettingsController::class, 'profile'])->name('profile');
        Route::post('/profile', [SettingsController::class, 'updateProfile'])->name('profile.update');
        Route::get('/security', [SettingsController::class, 'security'])->name('security');
        Route::post('/password', [SettingsController::class, 'updatePassword'])->name('password.update');
        Route::post('/transaction-pin', [SettingsController::class, 'updateTransactionPin'])->name('transaction-pin.update');
        Route::get('/notifications', [SettingsController::class, 'notifications'])->name('notifications');
        Route::post('/notifications', [SettingsController::class, 'updateNotifications'])->name('notifications.update');
        Route::get('/api', [SettingsController::class, 'api'])->name('api');
        Route::post('/api/generate', [SettingsController::class, 'generateApiKey'])->name('api.generate');
        Route::delete('/api/{token}', [SettingsController::class, 'revokeApiKey'])->name('api.revoke');
        Route::post('/logout-all', [SettingsController::class, 'logoutAllDevices'])->name('logout-all');
    });
});

// KYC Routes (accessible to authenticated users without KYC verification)
Route::middleware('auth')->group(function () {
    Route::prefix('kyc')->group(function () {
        Route::get('/', [KycController::class, 'index'])->name('kyc.index');
        Route::get('/bvn', [KycController::class, 'showBvnForm'])->name('kyc.bvn');
        Route::post('/bvn', [KycController::class, 'verifyBvn']);
        Route::get('/nin', [KycController::class, 'showNinForm'])->name('kyc.nin');
        Route::post('/nin', [KycController::class, 'verifyNin']);
        Route::get('/document', [KycController::class, 'showDocumentForm'])->name('kyc.document');
        Route::post('/document', [KycController::class, 'uploadDocument']);
    });
});