# Nigerian Fintech Platform

A comprehensive Laravel-based fintech platform specifically designed for the Nigerian market, providing digital wallet services, peer-to-peer transfers, bill payments, virtual bank accounts, and virtual card issuing with full admin management capabilities.

## 🚀 Features

### 🔐 Authentication & Security
- Multi-factor authentication (email + phone verification)
- Laravel Sanctum API authentication
- Transaction PIN protection
- Secure password hashing
- Session management
- Device tracking and security scoring

### 🆔 KYC Verification
- BVN (Bank Verification Number) integration via Dojah API
- NIN (National Identification Number) verification via VerifyMe API
- Document upload and verification
- Compliance with Nigerian financial regulations
- Real-time verification status tracking

### 💰 Digital Wallet System
- Multi-currency wallet support (primarily NGN)
- Real-time balance tracking
- Transaction limits and controls
- Automated transaction logging
- Wallet freeze/unfreeze functionality
- Daily spending limits with progress tracking

### 🏦 Virtual Bank Accounts
- Monnify integration for virtual account creation
- Anchor API support as backup provider
- Automatic wallet funding via bank transfers
- Multiple bank account options per user
- NUBAN account number generation

### 💸 Payment Processing
- Paystack integration for card payments
- P2P transfers between platform users
- Bank transfers to external accounts
- Transaction fee calculation and management
- Automated retry for failed transactions

### 📱 Bill Payment Services
- Airtime purchases (MTN, Glo, Airtel, 9mobile)
- Data bundle purchases
- Electricity bill payments
- Cable TV subscriptions
- VTPass API integration
- Transaction status monitoring and auto-sync

### 💳 Virtual Card Management
- Virtual card issuing via Bloc.io and Wallets Africa
- Card freeze/unfreeze functionality
- Transaction monitoring
- Spending limits and controls
- International and domestic card options
- Real-time card status synchronization

### 👨‍💼 Admin Panel
- Comprehensive admin dashboard
- User management with blocking/activation
- KYC approval/rejection workflow
- Transaction monitoring
- Audit logging with IP tracking
- Role-based access control
- Bulk operations for efficiency

### 🔔 Notifications System
- In-app notifications
- Email notifications
- SMS notifications (via Termii)
- Real-time updates for transactions
- Automated alerts for security events
- Customizable notification preferences

### ⚙️ Settings & Configuration
- Profile management
- Security settings (password, 2FA, transaction PIN)
- Notification preferences
- API key management for developers
- Session management across devices

### 🤖 Automated Operations
- Wallet balance synchronization (every 5 minutes)
- Virtual card status updates (every 10 minutes)
- Failed transaction retry logic (hourly)
- Bill payment status updates (every 3 minutes)
- Inactive account notifications (daily)
- System cleanup and optimization (weekly)
- Comprehensive business reporting (daily)

## 📋 Requirements

- PHP 8.1 or higher
- MySQL 8.0 or higher
- Redis 6.0 or higher
- Node.js 16.0 or higher
- Composer 2.0 or higher

## 🛠 Installation

### 1. Clone the Repository
```bash
git clone https://github.com/your-repo/fintech-platform.git
cd fintech-platform
```

### 2. Install Dependencies
```bash
# Install PHP dependencies
composer install

# Install Node.js dependencies
npm install
```

### 3. Environment Configuration
```bash
# Copy environment file
cp .env.example .env

# Generate application key
php artisan key:generate
```

### 4. Configure Environment Variables

Edit your `.env` file with the following configurations:

#### Database Configuration
```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=fintech_platform
DB_USERNAME=your_username
DB_PASSWORD=your_password
```

#### Redis Configuration
```env
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
```

#### Mail Configuration
```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your_email@gmail.com
MAIL_PASSWORD=your_app_password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=noreply@yourfintech.com
MAIL_FROM_NAME="Your Fintech"
```

#### Queue Configuration
```env
QUEUE_CONNECTION=redis
```

#### Session Configuration
```env
SESSION_DRIVER=redis
SESSION_LIFETIME=120
```

#### Cache Configuration
```env
CACHE_DRIVER=redis
```

### 5. API Keys Configuration

#### Paystack (Payment Processing)
```env
PAYSTACK_PUBLIC_KEY=pk_test_your_public_key
PAYSTACK_SECRET_KEY=sk_test_your_secret_key
PAYSTACK_PAYMENT_URL=https://api.paystack.co
```

#### Monnify (Virtual Accounts)
```env
MONNIFY_API_KEY=your_api_key
MONNIFY_SECRET_KEY=your_secret_key
MONNIFY_CONTRACT_CODE=your_contract_code
MONNIFY_BASE_URL=https://sandbox.monnify.com
```

#### VTPass (Bill Payments)
```env
VTPASS_API_KEY=your_api_key
VTPASS_SECRET_KEY=your_secret_key
VTPASS_BASE_URL=https://sandbox.vtpass.com
```

#### Dojah (BVN Verification)
```env
DOJAH_API_KEY=your_api_key
DOJAH_APP_ID=your_app_id
DOJAH_BASE_URL=https://sandbox.dojah.io
```

#### VerifyMe (NIN Verification)
```env
VERIFYME_API_KEY=your_api_key
VERIFYME_BASE_URL=https://api.verifyme.ng
```

#### Anchor (Virtual Cards)
```env
ANCHOR_API_KEY=your_api_key
ANCHOR_SECRET_KEY=your_secret_key
ANCHOR_BASE_URL=https://api.sandbox.anchor.dev
```

#### Termii (SMS)
```env
TERMII_API_KEY=your_api_key
TERMII_SENDER_ID=your_sender_id
TERMII_BASE_URL=https://termii.com
```

### 6. Database Setup
```bash
# Run migrations
php artisan migrate

# Seed database with initial data
php artisan db:seed
```

### 7. Build Assets
```bash
# Build frontend assets
npm run build

# For development
npm run dev
```

### 8. Setup Cron Jobs

Add the following to your crontab (`crontab -e`):

```bash
# Laravel Scheduler (runs every minute)
* * * * * cd /path/to/your/project && php artisan schedule:run >> /dev/null 2>&1

# Specific cron jobs for redundancy (optional)
*/5 * * * * cd /path/to/your/project && php artisan fintech:sync-wallets
*/10 * * * * cd /path/to/your/project && php artisan fintech:sync-virtual-cards
0 * * * * cd /path/to/your/project && php artisan fintech:retry-failed-transactions
*/3 * * * * cd /path/to/your/project && php artisan fintech:sync-bill-payments
0 9 * * * cd /path/to/your/project && php artisan fintech:notify-inactive-accounts
0 0 * * 0 cd /path/to/your/project && php artisan fintech:cleanup-notifications
0 1 * * * cd /path/to/your/project && php artisan fintech:generate-daily-reports
```

### 9. Setup Queue Workers

For production, use Supervisor to manage queue workers:

Create `/etc/supervisor/conf.d/fintech-worker.conf`:
```ini
[program:fintech-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /path/to/your/project/artisan queue:work redis --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=8
redirect_stderr=true
stdout_logfile=/path/to/your/project/storage/logs/worker.log
stopwaitsecs=3600
```

Then run:
```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start fintech-worker:*
```

## 🏗 Production Deployment

### 1. Server Requirements
- Ubuntu 20.04 LTS or CentOS 8
- Nginx 1.18+ or Apache 2.4+
- PHP 8.1+ with required extensions
- MySQL 8.0+ or PostgreSQL 13+
- Redis 6.0+
- SSL certificate

### 2. Web Server Configuration

#### Nginx Configuration
```nginx
server {
    listen 80;
    listen [::]:80;
    server_name yourfintech.com www.yourfintech.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    listen [::]:443 ssl http2;
    server_name yourfintech.com www.yourfintech.com;
    root /var/www/fintech-platform/public;

    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-Content-Type-Options "nosniff";

    index index.php;

    charset utf-8;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location = /favicon.ico { access_log off; log_not_found off; }
    location = /robots.txt  { access_log off; log_not_found off; }

    error_page 404 /index.php;

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }

    ssl_certificate /etc/letsencrypt/live/yourfintech.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/yourfintech.com/privkey.pem;
}
```

### 3. Production Environment Variables
```env
APP_ENV=production
APP_DEBUG=false
APP_URL=https://yourfintech.com

# Use production API endpoints
PAYSTACK_PUBLIC_KEY=pk_live_your_live_key
PAYSTACK_SECRET_KEY=sk_live_your_live_key
MONNIFY_BASE_URL=https://api.monnify.com
VTPASS_BASE_URL=https://vtpass.com
DOJAH_BASE_URL=https://api.dojah.io

# Security
SESSION_SECURE_COOKIE=true
SESSION_SAME_SITE=strict

# Logging
LOG_CHANNEL=daily
LOG_LEVEL=warning

# Admin notifications
ADMIN_EMAIL=admin@yourfintech.com
```

### 4. Performance Optimization
```bash
# Optimize Laravel
php artisan config:cache
php artisan route:cache
php artisan view:cache
php artisan event:cache

# Optimize Composer autoloader
composer install --optimize-autoloader --no-dev

# Set proper permissions
sudo chown -R www-data:www-data /var/www/fintech-platform
sudo chmod -R 755 /var/www/fintech-platform
sudo chmod -R 775 /var/www/fintech-platform/storage
sudo chmod -R 775 /var/www/fintech-platform/bootstrap/cache
```

### 5. Database Optimization
```sql
-- MySQL configuration optimizations
SET GLOBAL innodb_buffer_pool_size = 1073741824;  -- 1GB
SET GLOBAL query_cache_size = 268435456;          -- 256MB
SET GLOBAL max_connections = 200;
```

### 6. Monitoring and Logging

#### Setup Log Rotation
Create `/etc/logrotate.d/fintech-platform`:
```
/var/www/fintech-platform/storage/logs/*.log {
    daily
    missingok
    rotate 52
    compress
    delaycompress
    notifempty
    create 644 www-data www-data
}
```

#### Monitor Cron Jobs
Add monitoring for cron job logs:
```bash
# Monitor cron execution
tail -f /var/www/fintech-platform/storage/logs/cron-*.log

# Check for failed cron jobs
grep -i "error\|failed" /var/www/fintech-platform/storage/logs/cron-*.log
```

## 🔧 Module Configuration

### API Configuration Management

The platform includes a comprehensive API configuration system that allows administrators to manage all external service integrations through a modern web interface.

#### Supported API Providers

##### Payment Services
- **Paystack**: Payment processing and card transactions
  - Required fields: API Key, Secret Key
  - Features: Card payments, bank transfers, transaction verification
  - Test URL: `https://api.paystack.co`
  - Live URL: `https://api.paystack.co`

- **Flutterwave**: Payment gateway and financial services
  - Required fields: Public Key, Secret Key
  - Features: Multi-currency payments, card issuing
  - Test URL: `https://api.flutterwave.com/v3`
  - Live URL: `https://api.flutterwave.com/v3`

- **Monnify**: Payment processing and collection services
  - Required fields: API Key, Secret Key, Client ID
  - Features: Virtual accounts, collections, transfers
  - Test URL: `https://sandbox.monnify.com`
  - Live URL: `https://api.monnify.com`

##### Identity Verification Services
- **Dojah**: Identity verification and KYC services
  - Required fields: API Key, Private Key
  - Features: BVN verification, NIN verification, address verification
  - Test URL: `https://sandbox.dojah.io`
  - Live URL: `https://api.dojah.io`

- **VerifyMe**: Identity verification and compliance services
  - Required fields: API Key, Client ID
  - Features: BVN verification, NIN verification, CAC lookup
  - Test URL: `https://sandbox-api.myidentitypay.com`
  - Live URL: `https://api.myidentitypay.com`

##### Messaging Services
- **Termii**: SMS and messaging services
  - Required fields: API Key
  - Features: SMS delivery, OTP services, voice calls
  - URL: `https://api.ng.termii.com`

##### Utility Services
- **VTPass**: Bill payment and utility services
  - Required fields: API Key, Secret Key
  - Features: Airtime, data, electricity, cable TV payments
  - Test URL: `https://sandbox.vtpass.com/api`
  - Live URL: `https://vtpass.com/api`

##### Banking Services
- **Anchor**: Digital banking and account management
  - Required fields: API Key, Secret Key
  - Features: Virtual accounts, transfers, collections
  - Test URL: `https://api.sandbox.getanchor.co`
  - Live URL: `https://api.getanchor.co`

#### Configuration Process

1. **Access Admin Panel**: Navigate to `/admin/api-config` in the admin dashboard
2. **Select Provider**: Choose the API provider you want to configure
3. **Fill Credentials**: Enter the required API keys and credentials
4. **Configure URLs**: Set test and live environment URLs
5. **Set Webhooks**: Configure callback and webhook URLs
6. **Test Connection**: Use the built-in connection testing feature
7. **Activate Service**: Enable the service once configuration is complete

#### API Configuration Interface

The admin interface provides:
- **Real-time Status Monitoring**: Check API health and response times
- **Secure Credential Storage**: All sensitive data is encrypted at rest
- **Environment Management**: Easy switching between test and live modes
- **Webhook URL Generation**: Automatic generation of callback URLs
- **Configuration Validation**: Built-in validation and testing tools
- **Audit Logging**: Complete tracking of all configuration changes

#### Setting Up API Credentials

##### Paystack Configuration
```bash
# Example configuration
API Key: pk_test_xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
Secret Key: sk_test_xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
Webhook URL: https://yourdomain.com/webhooks/api/paystack
```

##### Flutterwave Configuration
```bash
# Example configuration
Public Key: FLWPUBK_TEST-xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx-X
Secret Key: FLWSECK_TEST-xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx-X
Webhook URL: https://yourdomain.com/webhooks/api/flutterwave
```

##### Dojah Configuration
```bash
# Example configuration
API Key: test_xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
Private Key: xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
Webhook URL: https://yourdomain.com/webhooks/api/dojah
```

#### Webhook Configuration

The system automatically generates webhook URLs for each configured provider:
- **Format**: `https://yourdomain.com/webhooks/api/{provider_slug}`
- **Authentication**: Each webhook includes signature verification
- **Retry Logic**: Failed webhooks are automatically retried
- **Logging**: All webhook events are logged for debugging

#### Environment Variables

Add these to your `.env` file to override default settings:
```bash
# API Configuration Defaults
API_CONFIG_DEFAULT_TIMEOUT=30
API_CONFIG_DEFAULT_RETRY_ATTEMPTS=3
API_CONFIG_ENCRYPTION_KEY=your-32-character-key

# Webhook Security
WEBHOOK_SIGNATURE_SECRET=your-webhook-secret
WEBHOOK_TIMEOUT=30
```

#### Testing API Connections

The platform includes built-in connection testing:
1. Navigate to the API configuration page
2. Click "Test Connection" for any configured provider
3. View real-time results including response times
4. Check logs for detailed error information if tests fail

#### Production Deployment

Before going live:
1. ✅ Configure all required API providers
2. ✅ Test all connections in sandbox mode
3. ✅ Verify webhook URLs are accessible
4. ✅ Switch to live mode for each provider
5. ✅ Monitor API health and response times
6. ✅ Set up alert notifications for failures

#### API Configuration Logs

Access comprehensive logging at `/admin/api-config/logs`:
- **Configuration Changes**: Track all credential updates
- **Connection Tests**: Monitor API health checks
- **Mode Switches**: Log test/live mode changes
- **Status Changes**: Track service activation/deactivation
- **Error Tracking**: Detailed error logs for troubleshooting

### User Dashboard Modules

#### Wallet Module
- **Funding**: Integrated with Paystack for card payments
- **Transfers**: P2P and bank transfers with real-time processing
- **Limits**: Configurable daily/monthly limits based on KYC level
- **History**: Comprehensive transaction filtering and search

#### Virtual Cards Module
- **Creation**: KYC-verified users can request cards
- **Management**: Freeze/unfreeze, spending limits, top-up
- **Monitoring**: Real-time transaction tracking and alerts
- **Security**: Masked card details with reveal on authentication

#### Bill Payments Module
- **Airtime**: All major Nigerian networks with instant delivery
- **Data**: Network-specific data plans with auto-activation
- **Electricity**: All DisCos with meter validation
- **Status Tracking**: Real-time status updates and failure handling

#### KYC Module
- **BVN Verification**: Real-time BVN validation via Dojah
- **NIN Verification**: National ID verification via VerifyMe
- **Document Upload**: Secure document storage and review
- **Status Tracking**: Real-time verification progress

### Admin Panel Modules

#### User Management
- **User Listing**: Advanced search and filtering
- **Account Controls**: Block/unblock, verify, reset passwords
- **Wallet Management**: Balance adjustments, freeze/unfreeze
- **Audit Trail**: Complete action logging with IP tracking

#### KYC Management
- **Review Queue**: Pending verifications with detailed review
- **Bulk Operations**: Approve/reject multiple requests
- **Document Viewer**: Secure document download and review
- **Compliance Reports**: KYC statistics and compliance tracking

#### Transaction Monitoring
- **Real-time Dashboard**: Live transaction monitoring
- **Fraud Detection**: Suspicious activity alerts
- **Manual Review**: Flag and review suspicious transactions
- **Reporting**: Comprehensive transaction analytics

## 🔐 Security Features

### Data Protection
- All sensitive data encrypted at rest
- PCI DSS compliant payment processing
- GDPR compliant data handling
- Regular security audits and penetration testing

### API Security
- Rate limiting on all endpoints
- JWT token authentication
- IP whitelisting for admin access
- Request/response logging for audit

### Transaction Security
- Transaction PIN requirement
- Biometric authentication support
- Device fingerprinting
- Suspicious activity detection

## 📊 Analytics and Reporting

### Business Intelligence
- Daily/weekly/monthly business reports
- Revenue and transaction analytics
- User behavior tracking
- KYC completion rates
- Service performance metrics

### Operational Monitoring
- System health monitoring
- API performance tracking
- Queue job monitoring
- Error rate tracking
- Database performance metrics

## 🆘 Troubleshooting

### Common Issues

#### Cron Jobs Not Running
```bash
# Check crontab
crontab -l

# Check cron service
sudo systemctl status cron

# Check Laravel scheduler
php artisan schedule:list
```

#### Queue Jobs Failing
```bash
# Check queue status
php artisan queue:work --once

# Monitor failed jobs
php artisan queue:failed

# Retry failed jobs
php artisan queue:retry all
```

#### API Integration Issues
```bash
# Test API connections
php artisan tinker
>>> \App\Services\PaystackService::class
>>> app(\App\Services\PaystackService::class)->getBanks()
```

#### Database Connection Issues
```bash
# Test database connection
php artisan tinker
>>> \DB::connection()->getPdo()
```

### Performance Issues

#### Slow Database Queries
```sql
-- Enable slow query log
SET GLOBAL slow_query_log = 'ON';
SET GLOBAL long_query_time = 2;

-- Check slow queries
SELECT * FROM mysql.slow_log;
```

#### High Memory Usage
```bash
# Monitor PHP memory usage
php -i | grep memory_limit

# Monitor system memory
free -h
top -p $(pgrep php)
```

### Log Analysis
```bash
# Application logs
tail -f storage/logs/laravel.log

# Cron job logs
tail -f storage/logs/cron-*.log

# Queue logs
tail -f storage/logs/queue.log

# Nginx/Apache logs
tail -f /var/log/nginx/access.log
tail -f /var/log/nginx/error.log
```

## 📞 Support

For technical support or questions:
- Email: support@yourfintech.com
- Documentation: https://docs.yourfintech.com
- GitHub Issues: https://github.com/your-repo/fintech-platform/issues

## 📄 License

This project is proprietary software. All rights reserved.

## 🤝 Contributing

Please read CONTRIBUTING.md for details on our code of conduct and the process for submitting pull requests.

## 📈 Roadmap

### Upcoming Features
- Mobile applications (iOS/Android)
- Cryptocurrency support
- Investment products
- Insurance services
- Business accounts
- API marketplace
- White-label solutions

### Performance Improvements
- Database optimization
- Caching improvements
- CDN integration
- Microservices architecture
- Auto-scaling capabilities

### 👨‍💼 Admin Panel
- **Dedicated Admin Authentication**: Separate login system with guard isolation
- **Multi-role Support**: Super Admin, Admin, Support, and Viewer roles
- **KYC Management**: Review and approve/reject KYC submissions  
- **User Management**: View and manage user accounts
- **Transaction Monitoring**: Real-time transaction oversight
- **System Dashboard**: Overview statistics and system status
- **Permission-based Access**: Role-based permission system
- **Secure Admin Routes**: Protected with admin guard middleware

### 🔗 Webhook Integration
- Real-time payment notifications
- Automatic transaction updates
- Webhook signature verification
- Retry mechanisms for failed webhooks

## Tech Stack

- **Framework**: Laravel 11
- **Database**: MySQL/PostgreSQL
- **Authentication**: Laravel Sanctum
- **Queue**: Database/Redis
- **Storage**: Local/S3
- **Frontend**: Blade Templates (Mobile-first)

## Installation

### System Requirements

**Minimum Requirements:**
- PHP 8.2 or higher with required extensions
- Composer 2.0+
- MySQL 8.0+ or PostgreSQL 13+
- Node.js 18+ and NPM/Yarn
- Redis 6.0+ (for caching and queues)
- Minimum 2GB RAM, 10GB storage
- SSL certificate for production (required for webhooks)

**PHP Extensions Required:**
- BCMath, Ctype, cURL, DOM, Fileinfo, Filter, Hash, Mbstring, OpenSSL, PCRE, PDO, Session, Tokenizer, XML

### Local Development Setup

1. **Clone the repository**
   ```bash
   git clone https://github.com/codemoll/Fintech.git
   cd Fintech
   ```

2. **Install dependencies**
   ```bash
   composer install
   npm install
   ```

3. **Environment configuration**
   ```bash
   cp .env.example .env
   php artisan key:generate
   ```

4. **Configure database**
   Update your `.env` file with database credentials:
   ```env
   DB_CONNECTION=mysql
   DB_HOST=127.0.0.1
   DB_PORT=3306
   DB_DATABASE=fintech_nigeria
   DB_USERNAME=your_username
   DB_PASSWORD=your_password
   ```

5. **Run migrations and setup**
   ```bash
   php artisan migrate
   php artisan db:seed
   ```

6. **Compile assets**
   ```bash
   npm run build
   ```

7. **Start the development server**
   ```bash
   php artisan serve
   
   # In a separate terminal, start queue workers
   php artisan queue:work
   ```

## cPanel Hosting Deployment Guide

### Prerequisites for cPanel Hosting

- **Shared hosting with:**
  - PHP 8.2+ support
  - MySQL database access
  - SSL certificate installed
  - SSH access (preferred) or File Manager
  - At least 2GB storage space
  - Cron job access

### Step-by-Step cPanel Deployment

#### 1. Prepare Your Local Project

```bash
# Clone and prepare the project
git clone https://github.com/codemoll/Fintech.git
cd Fintech

# Install dependencies
composer install --optimize-autoloader --no-dev
npm install
npm run build

# Create deployment archive (exclude unnecessary files)
zip -r fintech-deploy.zip . -x "node_modules/*" ".git/*" "tests/*" "storage/logs/*" "*.md"
```

#### 2. Upload Files via cPanel

**Option A: SSH Upload (Preferred)**
```bash
# Upload to your hosting account
scp fintech-deploy.zip username@yourhost.com:~/
ssh username@yourhost.com
cd ~/
unzip fintech-deploy.zip
mv Fintech/* public_html/
```

**Option B: File Manager Upload**
1. Login to cPanel → File Manager
2. Navigate to `public_html` directory
3. Upload `fintech-deploy.zip`
4. Extract the archive
5. Move contents from `Fintech/` folder to `public_html/`

#### 3. Database Setup

1. **Create Database:**
   - cPanel → MySQL Databases
   - Create database: `username_fintech`
   - Create user: `username_fintech_user`
   - Add user to database with ALL PRIVILEGES

2. **Import Database Schema:**
   ```bash
   # Via SSH
   php artisan migrate --force
   
   # Or via phpMyAdmin
   # Import the SQL schema manually if SSH not available
   ```

#### 4. Environment Configuration

1. **Copy and configure `.env` file:**
   ```bash
   cp .env.example .env
   ```

2. **Edit `.env` with cPanel details:**
   ```env
   APP_NAME="Your Fintech Platform"
   APP_ENV=production
   APP_KEY=base64:YOUR_GENERATED_KEY_HERE
   APP_DEBUG=false
   APP_URL=https://yourdomain.com

   # Database Configuration
   DB_CONNECTION=mysql
   DB_HOST=localhost
   DB_PORT=3306
   DB_DATABASE=username_fintech
   DB_USERNAME=username_fintech_user
   DB_PASSWORD=your_database_password

   # Production Settings
   SESSION_DRIVER=database
   QUEUE_CONNECTION=database
   CACHE_STORE=database
   
   # Mail Configuration (using cPanel email)
   MAIL_MAILER=smtp
   MAIL_HOST=mail.yourdomain.com
   MAIL_PORT=465
   MAIL_USERNAME=noreply@yourdomain.com
   MAIL_PASSWORD=your_email_password
   MAIL_ENCRYPTION=ssl
   MAIL_FROM_ADDRESS=noreply@yourdomain.com
   MAIL_FROM_NAME="${APP_NAME}"

   # Payment Provider Keys (PRODUCTION)
   PAYSTACK_PUBLIC_KEY=pk_live_xxxxxxxxxxxxx
   PAYSTACK_SECRET_KEY=sk_live_xxxxxxxxxxxxx
   PAYSTACK_PAYMENT_URL=https://api.paystack.co
   PAYSTACK_MERCHANT_EMAIL=your@email.com

   # Add other production service keys here
   ```

#### 5. File Permissions Setup

```bash
# Set correct permissions
chmod -R 755 public_html/
chmod -R 775 public_html/storage/
chmod -R 775 public_html/bootstrap/cache/
chmod 644 public_html/.env

# Secure sensitive files
chmod 600 public_html/.env
chmod -R 750 public_html/storage/logs/
```

#### 6. Laravel Optimization for Production

```bash
# Clear and cache configurations
php artisan config:cache
php artisan route:cache
php artisan view:cache

# Generate application key if not set
php artisan key:generate --force

# Run migrations
php artisan migrate --force
```

#### 7. Subdomain Setup (Optional but Recommended)

1. **Create Subdomain:**
   - cPanel → Subdomains
   - Create: `api.yourdomain.com` → points to `public_html/public`

2. **SSL Certificate:**
   - cPanel → SSL/TLS → Let's Encrypt
   - Enable for main domain and subdomain

#### 8. Cron Jobs Setup

1. **Access Cron Jobs:**
   - cPanel → Cron Jobs

2. **Add Laravel Scheduler:**
   ```bash
   # Run every minute
   * * * * * cd /home/username/public_html && php artisan schedule:run >> /dev/null 2>&1
   ```

3. **Add Queue Worker (if supported):**
   ```bash
   # Every 5 minutes, restart queue workers
   */5 * * * * cd /home/username/public_html && php artisan queue:restart
   ```

#### 9. Webhook URLs Configuration

Update your payment provider webhooks to point to:
```
https://yourdomain.com/api/webhooks/paystack
https://yourdomain.com/api/webhooks/monnify
https://yourdomain.com/api/webhooks/anchor
https://yourdomain.com/api/webhooks/vtpass
```

#### 10. Security Headers (.htaccess)

Create/update `.htaccess` in `public_html/public/`:
```apache
<IfModule mod_rewrite.c>
    <IfModule mod_negotiation.c>
        Options -MultiViews -Indexes
    </IfModule>

    RewriteEngine On

    # Handle Angular and Vue History API fallback requests
    RewriteCond %{REQUEST_FILENAME} -d [OR]
    RewriteCond %{REQUEST_FILENAME} -f
    RewriteRule ^ ^$1 [N]

    RewriteCond %{REQUEST_URI} (\.\w+$) [NC]
    RewriteRule ^(.*)$ public/$1

    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteRule ^ server.php

    # Security Headers
    Header always set X-Frame-Options DENY
    Header always set X-Content-Type-Options nosniff
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Strict-Transport-Security "max-age=31536000; includeSubDomains"
    Header always set Content-Security-Policy "default-src 'self'"
</IfModule>

# Block access to sensitive files
<Files .env>
    Order allow,deny
    Deny from all
</Files>

<Files composer.lock>
    Order allow,deny
    Deny from all
</Files>
```

### Post-Deployment Verification

1. **Test Application:**
   ```bash
   # Check if app loads
   curl -I https://yourdomain.com
   
   # Test API endpoints
   curl https://yourdomain.com/api/auth/register
   ```

2. **Check Logs:**
   ```bash
   tail -f storage/logs/laravel.log
   ```

3. **Verify Database Connection:**
   ```bash
   php artisan tinker
   # Test: DB::connection()->getPdo();
   ```

### Troubleshooting Common cPanel Issues

#### Issue: 500 Internal Server Error
**Solutions:**
1. Check `.env` file exists and has correct values
2. Verify file permissions (755 for directories, 644 for files)
3. Run `php artisan config:clear`
4. Check error logs in cPanel → Error Logs

#### Issue: Database Connection Failed
**Solutions:**
1. Verify database credentials in `.env`
2. Check if database user has proper privileges
3. Confirm database exists and is accessible

#### Issue: Queue Jobs Not Processing
**Solutions:**
1. Verify cron job is set up correctly
2. Use `database` queue driver instead of `redis`
3. Check if hosting supports background processes

#### Issue: Assets Not Loading
**Solutions:**
1. Ensure `npm run build` was executed
2. Check if `public/` directory exists
3. Verify asset URLs in configuration

### Maintenance Mode

```bash
# Enable maintenance mode
php artisan down --message="Upgrading system" --retry=60

# Disable maintenance mode
php artisan up
```

### Backup Strategy

1. **Database Backup:**
   ```bash
   # Daily database backup
   mysqldump -u username -p database_name > backup_$(date +%Y%m%d).sql
   ```

2. **File Backup:**
   ```bash
   # Weekly file backup
   tar -czf backup_files_$(date +%Y%m%d).tar.gz public_html/
   ```

### Production Monitoring

1. **Enable Error Logging:**
   ```env
   LOG_CHANNEL=daily
   LOG_LEVEL=error
   ```

2. **Monitor Key Metrics:**
   - Application response time
   - Database query performance
   - Failed queue jobs
   - Error rates

3. **Set Up Alerts:**
   - Email alerts for application errors
   - Transaction failure notifications
   - System resource monitoring

## Configuration

### Payment Providers

#### Paystack
```env
PAYSTACK_PUBLIC_KEY=pk_test_xxxxxxxxxxxxxxxxxxxxxxxxxx
PAYSTACK_SECRET_KEY=sk_test_xxxxxxxxxxxxxxxxxxxxxxxxxx
PAYSTACK_PAYMENT_URL=https://api.paystack.co
PAYSTACK_MERCHANT_EMAIL=merchant@example.com
```

#### Monnify
```env
MONNIFY_API_KEY=MK_TEST_xxxxxxxxxxxxxxxxxxxxxxxxxx
MONNIFY_SECRET_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
MONNIFY_CONTRACT_CODE=xxxxxxxxxxxxxxxxxxxxxxxxxx
MONNIFY_BASE_URL=https://sandbox.monnify.com
```

### KYC Providers

#### Dojah
```env
DOJAH_APP_ID=xxxxxxxxxxxxxxxxxxxxxxxxxx
DOJAH_SECRET_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
DOJAH_BASE_URL=https://sandbox.dojah.io
```

#### VerifyMe
```env
VERIFYME_USERNAME=xxxxxxxxxxxxxxxxxxxxxxxxxx
VERIFYME_PASSWORD=xxxxxxxxxxxxxxxxxxxxxxxxxx
VERIFYME_BASE_URL=https://developer.verifymy.co.uk
```

### Bill Payment

#### VTPass
```env
VTPASS_USERNAME=xxxxxxxxxxxxxxxxxxxxxxxxxx
VTPASS_PASSWORD=xxxxxxxxxxxxxxxxxxxxxxxxxx
VTPASS_BASE_URL=https://sandbox.vtpass.com
```

### Virtual Cards

#### Bloc.io
```env
BLOC_SECRET_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
BLOC_PUBLIC_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
BLOC_BASE_URL=https://blochq.io
```

#### Wallets Africa
```env
WALLETS_AFRICA_PUBLIC_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
WALLETS_AFRICA_SECRET_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
WALLETS_AFRICA_BASE_URL=https://api.wallets.africa
```

## API Documentation

### Authentication Endpoints

```http
POST /api/auth/register
POST /api/auth/login
POST /api/auth/verify-phone
POST /api/auth/logout
```

### Wallet Endpoints

```http
GET /api/wallet
POST /api/wallet/fund
GET /api/wallet/transactions
```

### Transaction Endpoints

```http
GET /api/transactions
POST /api/transactions/p2p
POST /api/transactions/bank-transfer
```

### KYC Endpoints

```http
GET /api/kyc/status
POST /api/kyc/bvn
POST /api/kyc/nin
POST /api/kyc/upload-document
```

### Bill Payment Endpoints

```http
GET /api/bills/providers
POST /api/bills/airtime
POST /api/bills/data
POST /api/bills/electricity
```

## 📸 Admin Interface Screenshots

### API Configuration Dashboard
The main API configuration dashboard provides an overview of all integrated services:

![API Configuration Dashboard](https://github.com/codemoll/Fintech/blob/main/docs/screenshots/api-config-dashboard.png)

*Features shown:*
- Service status overview with health monitoring
- Quick actions for testing all connections
- Categorized service groups (Payment, Identity, Messaging, Utilities, Banking)
- Real-time status indicators and response times

### Individual API Configuration
Detailed configuration interface for each API provider:

![Paystack Configuration](https://github.com/codemoll/Fintech/blob/main/docs/screenshots/api-config-paystack.png)

*Features shown:*
- Comprehensive credential management with secure storage
- Environment-specific URL configuration (test/live)
- Auto-generated webhook and callback URLs
- Real-time connection testing
- Quick action buttons for activation and mode switching

### API Configuration Logs
Comprehensive audit trail of all API configuration changes:

![API Configuration Logs](https://github.com/codemoll/Fintech/blob/main/docs/screenshots/api-config-logs.png)

*Features shown:*
- Detailed activity logging with timestamps
- Advanced filtering by service, action, and date range
- User attribution for all changes
- Change tracking with before/after values

## Security Best Practices

### Transaction Security
- All financial transactions require PIN verification
- Transaction limits enforced at wallet level
- Real-time fraud detection (to be implemented)
- Secure API communication with HTTPS

### Data Protection
- PII encryption for sensitive data
- Secure storage of KYC documents
- Regular security audits recommended
- GDPR/Nigerian Data Protection compliance

### API Security
- Rate limiting on all endpoints
- Request signature verification for webhooks
- Token-based authentication with expiration
- Input validation and sanitization

## Deployment

### Production Environment

1. **Server Requirements**
   - PHP 8.2+ with required extensions
   - MySQL/PostgreSQL database
   - Redis for caching and queues
   - Nginx/Apache web server
   - SSL certificate (required)

2. **Environment Variables**
   ```bash
   APP_ENV=production
   APP_DEBUG=false
   APP_URL=https://your-domain.com
   ```

3. **Optimize for Production**
   ```bash
   php artisan config:cache
   php artisan route:cache
   php artisan view:cache
   composer install --optimize-autoloader --no-dev
   ```

4. **Queue Workers**
   ```bash
   php artisan queue:work --daemon
   ```

5. **Scheduled Tasks**
   Add to crontab:
   ```bash
   * * * * * cd /path-to-your-project && php artisan schedule:run >> /dev/null 2>&1
   ```

### Docker Deployment (Optional)

```dockerfile
# Use the official PHP image
FROM php:8.2-fpm

# Install dependencies
RUN apt-get update && apt-get install -y \
    git \
    curl \
    libpng-dev \
    libonig-dev \
    libxml2-dev \
    zip \
    unzip

# Install PHP extensions
RUN docker-php-ext-install pdo_mysql mbstring exif pcntl bcmath gd

# Get Composer
COPY --from=composer:latest /usr/bin/composer /usr/bin/composer

# Set working directory
WORKDIR /var/www

# Copy project files
COPY . /var/www

# Install dependencies
RUN composer install --optimize-autoloader --no-dev

# Set permissions
RUN chown -R www-data:www-data /var/www
```

## Testing

<!-- TODO: CRITICAL - No testing infrastructure exists -->
<!-- TODO: Create tests directory and phpunit.xml configuration -->
<!-- TODO: Implement comprehensive test suite for production readiness -->

### Running Tests
```bash
php artisan test  # TODO: Set up testing infrastructure first
```

### Feature Testing
```bash
php artisan test --filter=FeatureTest  # TODO: Create feature tests for all endpoints
```

### Unit Testing
```bash
php artisan test --filter=UnitTest  # TODO: Create unit tests for models and services
```

<!-- TODO: Add testing guidelines and examples -->
<!-- TODO: Set up test database configuration -->
<!-- TODO: Create factories for all models -->
<!-- TODO: Add integration tests for external APIs -->

## Admin Access & Management

### Admin Authentication

The platform includes a separate admin authentication system with dedicated routes and guards:

#### Accessing the Admin Panel

1. **Admin Login URL**: `/admin/login`
2. **Default Admin Dashboard**: `/admin/dashboard`

#### Default Admin Credentials (Pre-seeded)

The application comes with pre-configured admin accounts that are automatically created during setup:

**Super Admin Account:**
- **Email**: `admin@fintechplatform.com`
- **Password**: `SecurePassword123!`
- **Role**: `super_admin`
- **Permissions**: Full system access

**Support Admin Account:**
- **Email**: `support@fintechplatform.com`
- **Password**: `SupportPass123!`
- **Role**: `support`
- **Permissions**: User support and limited access

#### Admin Setup Instructions

1. **Run Database Migrations and Seeders:**
   ```bash
   php artisan migrate
   php artisan db:seed
   ```

2. **Access Admin Panel:**
   - Navigate to: `http://your-domain.com/admin/login`
   - Use the credentials above to log in

3. **First Login Steps:**
   - Change default passwords immediately
   - Configure additional admin users as needed
   - Review and update admin permissions
   - Test KYC approval workflow

#### Creating Additional Admin Users

To create additional admin users, use Laravel Tinker or the admin interface:

```bash
php artisan tinker

# Create a new admin
App\Models\Admin::create([
    'name' => 'KYC Manager',
    'email' => 'kyc@yourfintech.com',
    'password' => Hash::make('secure_password_here'),
    'role' => 'admin',
    'permissions' => ['users.view', 'kyc.view', 'kyc.approve'],
    'is_active' => true
]);
```

#### Admin Roles & Permissions

| Role | Permissions | Description |
|------|-------------|-------------|
| `super_admin` | All permissions (`*`) | Full system access |
| `admin` | User management, KYC, transactions, wallets | Standard admin access |
| `support` | User view, transaction view, support tickets | Customer support role |
| `viewer` | Dashboard view, reports view | Read-only access |

#### Admin Security Best Practices

1. **Change Default Passwords**: Immediately update the default admin passwords
2. **Enable 2FA**: Configure two-factor authentication for admin accounts
3. **Regular Password Updates**: Change passwords every 90 days
4. **Access Monitoring**: Monitor admin login attempts and activities
5. **IP Whitelisting**: Restrict admin access to specific IP ranges if possible

#### Troubleshooting Admin Access

**Common Issues and Solutions:**

1. **Admin Login Not Working:**
   ```bash
   # Verify admin exists and is active
   php artisan tinker
   App\Models\Admin::where('email', 'admin@fintechplatform.com')->first();
   
   # Reset admin password if needed
   $admin = App\Models\Admin::where('email', 'admin@fintechplatform.com')->first();
   $admin->password = Hash::make('NewSecurePassword123!');
   $admin->save();
   ```

2. **Admin Dashboard Not Loading:**
   ```bash
   # Clear caches
   php artisan cache:clear
   php artisan config:clear
   php artisan view:clear
   ```

3. **Middleware Errors:**
   ```bash
   # Check if middleware is properly registered
   php artisan route:list | grep admin
   ```

4. **Permission Errors:**
   ```bash
   # Verify admin permissions
   php artisan tinker
   $admin = App\Models\Admin::find(1);
   dd($admin->getRolePermissions());
   ```

### Admin Dashboard Features

- **User Statistics**: Total users, new registrations
- **KYC Management**: Pending verifications, approval queue
- **Transaction Overview**: Transaction counts and monitoring
- **System Status**: Real-time system health indicators
- **Quick Actions**: Direct access to common admin tasks

## User Authentication & Registration

### User Registration Flow

The platform provides a comprehensive user registration and verification system:

#### 1. Registration Process

**Registration URL**: `/register`

**Required Information:**
- First Name and Last Name
- Email Address (must be unique)
- Phone Number (Nigerian format: 08xxxxxxxxx)
- Date of Birth (must be 18+ years old)
- Gender (Male/Female/Other)
- Password (minimum 8 characters)
- Terms of Service acceptance

**Registration Steps:**
1. Navigate to registration page
2. Fill out the registration form
3. Submit form (creates user account and wallet)
4. Automatic login and redirect to phone verification

#### 2. Phone Verification

**Phone Verification URL**: `/verify-phone`

After successful registration, users must verify their phone number:

1. **OTP Generation**: 6-digit code sent via SMS
2. **Verification Form**: Enter the OTP code
3. **Code Expiry**: OTP valid for 10 minutes
4. **Resend Option**: Users can request new OTP after 60 seconds
5. **Success**: Redirect to KYC verification page

**Development Note**: In local environment, OTP codes are logged to `storage/logs/laravel.log`

#### 3. User Login Flow

**Login URL**: `/login`

**Login Process:**
1. Enter email/phone and password
2. Successful authentication triggers redirect logic:
   - If phone not verified → redirect to `/verify-phone`
   - If KYC not completed → redirect to `/kyc`
   - If all complete → redirect to `/dashboard`

#### 4. User Dashboard Access

**Dashboard URL**: `/dashboard`

**Access Requirements:**
- ✅ Valid user authentication
- ✅ Phone number verified
- ✅ KYC verification completed

**Middleware Protection:**
- `auth`: Ensures user is logged in
- `kyc.verified`: Ensures KYC is completed

### User Authentication Testing

#### Test User Registration

You can test the complete user flow:

1. **Register New User:**
   - Go to `/register`
   - Fill out the registration form
   - Submit registration

2. **Verify Phone Number:**
   - Check Laravel logs for OTP: `tail -f storage/logs/laravel.log`
   - Enter the 6-digit OTP code
   - Confirm verification success

3. **Complete KYC:**
   - Choose a KYC verification method
   - Complete the verification process
   - Access full dashboard features

#### Development Testing

For development and testing purposes:

```bash
# Create a test user with phone verified and KYC completed
php artisan tinker

$user = App\Models\User::create([
    'first_name' => 'Test',
    'last_name' => 'User',
    'email' => 'testuser@example.com',
    'phone' => '08123456789',
    'password' => Hash::make('password123'),
    'date_of_birth' => '1990-01-01',
    'gender' => 'male',
    'country' => 'Nigeria',
    'phone_verified_at' => now(),
]);

# Create default wallet
App\Models\Wallet::create([
    'user_id' => $user->id,
    'currency' => 'NGN',
]);

# Create KYC verification (to bypass KYC requirement)
App\Models\KycVerification::create([
    'user_id' => $user->id,
    'type' => 'bvn',
    'data' => ['bvn' => '12345678901'],
    'status' => 'verified',
    'verified_at' => now(),
]);
```

### User Authentication Troubleshooting

**Common Issues and Solutions:**

1. **Registration Validation Errors:**
   ```bash
   # Check validation rules in AuthController
   # Ensure phone number is in correct format (08xxxxxxxxx)
   # Verify age requirement (18+ years)
   ```

2. **Phone Verification Not Working:**
   ```bash
   # Check OTP in Laravel logs
   tail -f storage/logs/laravel.log | grep "OTP"
   
   # Manually set phone as verified
   php artisan tinker
   $user = App\Models\User::where('email', 'user@example.com')->first();
   $user->phone_verified_at = now();
   $user->save();
   ```

3. **Cannot Access Dashboard:**
   ```bash
   # Check KYC status
   php artisan tinker
   $user = App\Models\User::find(1);
   echo $user->hasCompletedKyc() ? 'KYC Complete' : 'KYC Pending';
   ```

4. **Session/Login Issues:**
   ```bash
   # Clear sessions and cache
   php artisan session:table
   php artisan migrate
   php artisan cache:clear
   ```

## KYC Verification Flow

### For Users

After registration, users are redirected to complete KYC verification:

1. **KYC Dashboard** (`/kyc`): Overview of verification options
2. **BVN Verification** (`/kyc/bvn`): Bank Verification Number verification
3. **NIN Verification** (`/kyc/nin`): National Identification Number verification  
4. **Document Upload** (`/kyc/document`): Government-issued ID upload

### KYC Forms & Requirements

#### BVN Verification
- 11-digit Bank Verification Number
- Full name (as registered with bank)
- Date of birth
- Real-time verification via Dojah API

#### NIN Verification  
- 11-digit National Identification Number
- Full name (as on NIN slip)
- Date of birth
- Real-time verification via VerifyMe API

#### Document Upload
- Accepted documents: International Passport, Driver's License, Voter's Card
- File formats: PDF, JPG, PNG (max 2MB)
- Manual review by admin team
- Processing time: 24-48 hours

### For Admins

Admins can review and manage KYC submissions through the admin panel:

1. **KYC Queue**: View pending verifications
2. **Document Review**: Approve/reject uploaded documents
3. **Verification History**: Track KYC completion status
4. **Compliance Reports**: Generate KYC compliance reports

## New Routes Added

### Admin Routes
```
GET|POST /admin/login          # Admin authentication
GET     /admin/dashboard       # Admin dashboard  
POST    /admin/logout          # Admin logout
```

### KYC Routes
```
GET     /kyc                   # KYC overview dashboard
GET     /kyc/bvn              # BVN verification form
POST    /kyc/bvn              # Submit BVN verification
GET     /kyc/nin              # NIN verification form  
POST    /kyc/nin              # Submit NIN verification
GET     /kyc/document         # Document upload form
POST    /kyc/document         # Submit document upload
```

## Contributing

1. Fork the repository
2. Create a feature branch (`git checkout -b feature/amazing-feature`)
3. Commit your changes (`git commit -m 'Add some amazing feature'`)
4. Push to the branch (`git push origin feature/amazing-feature`)
5. Open a Pull Request

## License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## Support

For support and questions:
- Email: support@yourfintech.com
- Documentation: https://docs.yourfintech.com
- Issues: https://github.com/codemoll/Fintech/issues

## Roadmap

- [ ] Advanced fraud detection
- [ ] Crypto wallet integration
- [ ] International transfers
- [ ] Merchant payment gateway
- [ ] Mobile applications (iOS/Android)
- [ ] Investment platform integration
- [ ] Loan/credit services
- [ ] Insurance products integration

## Compliance

This platform is designed to comply with:
- Central Bank of Nigeria (CBN) regulations
- Nigerian Data Protection Regulation (NDPR)
- Anti-Money Laundering (AML) requirements
- Know Your Customer (KYC) compliance

**Note**: Always consult with legal and compliance experts before deploying financial services in production.