<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\VirtualCard;
use App\Models\Transaction;

class DashboardController extends Controller
{
    /**
     * Show the application dashboard.
     */
    public function index()
    {
        $user = Auth::user();
        $wallet = $user->wallet;
        
        // Get recent transactions
        $recentTransactions = $user->transactions()
            ->latest()
            ->limit(8)
            ->get();

        // Get virtual cards
        $virtualCards = VirtualCard::where('user_id', $user->id)
            ->whereIn('status', ['active', 'frozen'])
            ->latest()
            ->get();

        // Calculate monthly statistics
        $monthlyStats = $this->calculateMonthlyStats($user);

        // Get alerts for the user
        $alerts = $this->getUserAlerts($user, $wallet);

        return view('dashboard', compact(
            'user', 
            'wallet', 
            'recentTransactions', 
            'virtualCards',
            'monthlyStats',
            'alerts'
        ));
    }

    /**
     * Calculate monthly statistics for the user
     */
    private function calculateMonthlyStats($user)
    {
        $startOfMonth = now()->startOfMonth();
        
        $monthlyTransactions = $user->transactions()
            ->where('created_at', '>=', $startOfMonth)
            ->where('status', 'completed');

        $totalSpent = $monthlyTransactions
            ->where('type', 'debit')
            ->sum('amount');

        $totalReceived = $monthlyTransactions
            ->where('type', 'credit')
            ->sum('amount');

        $transactionCount = $monthlyTransactions->count();

        // Calculate savings (simplified - money received minus money spent)
        $savings = max(0, $totalReceived - $totalSpent);

        return [
            'total_spent' => $totalSpent,
            'total_received' => $totalReceived,
            'transaction_count' => $transactionCount,
            'savings' => $savings
        ];
    }

    /**
     * Get user alerts and notifications
     */
    private function getUserAlerts($user, $wallet)
    {
        $alerts = [];

        // KYC incomplete alert
        if (!$user->hasCompletedKyc()) {
            $alerts[] = [
                'type' => 'warning',
                'icon' => '⚠️',
                'title' => 'KYC Verification Pending',
                'message' => 'Complete your identity verification to unlock all features and increase limits.'
            ];
        }

        // Low balance alert
        if ($wallet && $wallet->available_balance < 1000 && $wallet->available_balance > 0) {
            $alerts[] = [
                'type' => 'info',
                'icon' => '💰',
                'title' => 'Low Balance',
                'message' => 'Your wallet balance is running low. Consider funding your wallet.'
            ];
        }

        // Frozen wallet alert
        if ($wallet && $wallet->is_frozen) {
            $alerts[] = [
                'type' => 'error',
                'icon' => '🔒',
                'title' => 'Wallet Frozen',
                'message' => 'Your wallet is frozen. Contact support for assistance.'
            ];
        }

        // Failed transactions alert
        $failedTransactions = $user->transactions()
            ->where('status', 'failed')
            ->where('created_at', '>=', now()->subDays(7))
            ->count();

        if ($failedTransactions > 0) {
            $alerts[] = [
                'type' => 'warning',
                'icon' => '❌',
                'title' => 'Failed Transactions',
                'message' => "You have {$failedTransactions} failed transaction(s) in the last 7 days."
            ];
        }

        // Inactive virtual cards
        $inactiveCards = VirtualCard::where('user_id', $user->id)
            ->where('status', 'active')
            ->where('updated_at', '<', now()->subDays(30))
            ->count();

        if ($inactiveCards > 0) {
            $alerts[] = [
                'type' => 'info',
                'icon' => '💳',
                'title' => 'Inactive Cards',
                'message' => "You have {$inactiveCards} virtual card(s) that haven't been used in 30 days."
            ];
        }

        return $alerts;
    }
}