<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Transaction;
use App\Models\CronLog;

class RetryFailedTransactionsCommand extends Command
{
    protected $signature = 'fintech:retry-failed-transactions';
    protected $description = 'Check for failed transactions and retry them';

    public function handle()
    {
        $startTime = microtime(true);
        $startMemory = memory_get_usage(true);
        
        try {
            $this->info('Starting failed transaction retry process...');
            
            $retriedCount = 0;
            $errorCount = 0;
            
            // Get failed transactions that are retryable (failed within last 24 hours)
            $failedTransactions = Transaction::where('status', 'failed')
                ->where('created_at', '>=', now()->subHours(24))
                ->whereJsonDoesntContain('metadata->retry_count', 3) // Max 3 retries
                ->where('category', '!=', 'p2p_transfer') // Don't retry P2P transfers
                ->get();
            
            foreach ($failedTransactions as $transaction) {
                try {
                    if ($this->retryTransaction($transaction)) {
                        $retriedCount++;
                        $this->info("Retried transaction {$transaction->id}");
                    }
                } catch (\Exception $e) {
                    $this->error("Error retrying transaction {$transaction->id}: " . $e->getMessage());
                    $errorCount++;
                }
            }
            
            // Check for stuck processing transactions (older than 30 minutes)
            $stuckTransactions = Transaction::where('status', 'processing')
                ->where('updated_at', '<', now()->subMinutes(30))
                ->get();
            
            foreach ($stuckTransactions as $transaction) {
                $this->checkStuckTransaction($transaction);
            }
            
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $message = "Retried {$retriedCount} transactions, {$errorCount} errors, checked " . count($stuckTransactions) . " stuck transactions";
            $this->info($message);
            
            CronLog::create([
                'task_name' => 'retry_failed_transactions',
                'status' => $errorCount > 0 ? 'warning' : 'success',
                'message' => $message,
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage
            ]);
            
        } catch (\Exception $e) {
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $this->error('Failed transaction retry process failed: ' . $e->getMessage());
            
            CronLog::create([
                'task_name' => 'retry_failed_transactions',
                'status' => 'failed',
                'message' => 'Failed transaction retry process failed',
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage,
                'error_details' => [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]
            ]);
        }
    }
    
    private function retryTransaction(Transaction $transaction)
    {
        $metadata = $transaction->metadata ?? [];
        $retryCount = $metadata['retry_count'] ?? 0;
        
        // Increment retry count
        $metadata['retry_count'] = $retryCount + 1;
        $metadata['last_retry_at'] = now()->toISOString();
        
        $transaction->update([
            'status' => 'processing',
            'metadata' => $metadata
        ]);
        
        // Retry based on transaction category
        switch ($transaction->category) {
            case 'wallet_funding':
                return $this->retryWalletFunding($transaction);
            case 'bank_transfer':
                return $this->retryBankTransfer($transaction);
            case 'airtime_purchase':
            case 'data_purchase':
            case 'bill_payment':
                return $this->retryBillPayment($transaction);
            default:
                $this->warn("Unknown transaction category: {$transaction->category}");
                return false;
        }
    }
    
    private function retryWalletFunding(Transaction $transaction)
    {
        // Check payment status with Paystack
        if (isset($transaction->external_reference)) {
            $paystackService = app(\App\Services\PaystackService::class);
            $response = $paystackService->verifyPayment($transaction->external_reference);
            
            if ($response['success'] && $response['status'] === 'success') {
                // Payment was successful, credit wallet
                $wallet = $transaction->wallet;
                $wallet->credit($transaction->amount);
                
                $transaction->markAsCompleted();
                return true;
            } elseif ($response['status'] === 'failed') {
                $transaction->markAsFailed('Payment verification failed');
                return false;
            }
        }
        
        return false;
    }
    
    private function retryBankTransfer(Transaction $transaction)
    {
        // Implement bank transfer retry logic
        // This would typically involve checking with the bank transfer provider
        
        $this->info("Bank transfer retry not implemented for transaction {$transaction->id}");
        return false;
    }
    
    private function retryBillPayment(Transaction $transaction)
    {
        // Check bill payment status with VTPass
        if (isset($transaction->external_reference)) {
            $vtpassService = app(\App\Services\VTPassService::class);
            $response = $vtpassService->checkTransactionStatus($transaction->external_reference);
            
            if ($response['success']) {
                if ($response['status'] === 'delivered' || $response['status'] === 'successful') {
                    $transaction->markAsCompleted();
                    return true;
                } elseif ($response['status'] === 'failed') {
                    $transaction->markAsFailed('Bill payment failed');
                    return false;
                }
            }
        }
        
        return false;
    }
    
    private function checkStuckTransaction(Transaction $transaction)
    {
        $this->info("Checking stuck transaction {$transaction->id}");
        
        // Try to get status from external provider
        $statusChecked = false;
        
        switch ($transaction->category) {
            case 'wallet_funding':
                if ($transaction->external_reference) {
                    $paystackService = app(\App\Services\PaystackService::class);
                    $response = $paystackService->verifyPayment($transaction->external_reference);
                    
                    if ($response['success']) {
                        if ($response['status'] === 'success') {
                            $transaction->wallet->credit($transaction->amount);
                            $transaction->markAsCompleted();
                        } elseif ($response['status'] === 'failed') {
                            $transaction->markAsFailed('Payment failed');
                        }
                        $statusChecked = true;
                    }
                }
                break;
                
            case 'airtime_purchase':
            case 'data_purchase':
            case 'bill_payment':
                if ($transaction->external_reference) {
                    $vtpassService = app(\App\Services\VTPassService::class);
                    $response = $vtpassService->checkTransactionStatus($transaction->external_reference);
                    
                    if ($response['success']) {
                        if (in_array($response['status'], ['delivered', 'successful'])) {
                            $transaction->markAsCompleted();
                        } elseif ($response['status'] === 'failed') {
                            $transaction->markAsFailed('Service delivery failed');
                        }
                        $statusChecked = true;
                    }
                }
                break;
        }
        
        // If we couldn't check status and it's very old, mark as failed
        if (!$statusChecked && $transaction->updated_at < now()->subHours(2)) {
            $transaction->markAsFailed('Transaction stuck in processing state');
            $this->warn("Marked stuck transaction {$transaction->id} as failed");
        }
    }
}