<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\Transaction;
use App\Models\Webhook;
use App\Services\PaystackService;

class WebhookController extends Controller
{
    /**
     * Handle Paystack webhook.
     */
    public function paystack(Request $request)
    {
        // Verify webhook signature
        $signature = $request->header('x-paystack-signature');
        $payload = $request->getContent();
        
        if (!$this->verifyPaystackSignature($payload, $signature)) {
            Log::warning('Invalid Paystack webhook signature');
            return response('Unauthorized', 401);
        }

        $data = $request->all();
        
        // Store webhook for processing
        $webhook = Webhook::create([
            'provider' => 'paystack',
            'event_type' => $data['event'] ?? 'unknown',
            'external_reference' => $data['data']['reference'] ?? null,
            'payload' => $data,
            'status' => 'pending',
        ]);

        try {
            $this->processPaystackWebhook($webhook);
        } catch (\Exception $e) {
            Log::error('Paystack webhook processing failed: ' . $e->getMessage());
            $webhook->markAsFailed($e->getMessage());
        }

        return response('OK', 200);
    }

    /**
     * Handle Monnify webhook.
     */
    public function monnify(Request $request)
    {
        $data = $request->all();
        
        // Store webhook for processing
        $webhook = Webhook::create([
            'provider' => 'monnify',
            'event_type' => $data['eventType'] ?? 'unknown',
            'external_reference' => $data['eventData']['transactionReference'] ?? null,
            'payload' => $data,
            'status' => 'pending',
        ]);

        try {
            $this->processMonnifyWebhook($webhook);
        } catch (\Exception $e) {
            Log::error('Monnify webhook processing failed: ' . $e->getMessage());
            $webhook->markAsFailed($e->getMessage());
        }

        return response('OK', 200);
    }

    /**
     * Handle Anchor webhook.
     */
    public function anchor(Request $request)
    {
        $data = $request->all();
        
        // Store webhook for processing
        $webhook = Webhook::create([
            'provider' => 'anchor',
            'event_type' => $data['event'] ?? 'unknown',
            'external_reference' => $data['data']['reference'] ?? null,
            'payload' => $data,
            'status' => 'pending',
        ]);

        try {
            $this->processAnchorWebhook($webhook);
        } catch (\Exception $e) {
            Log::error('Anchor webhook processing failed: ' . $e->getMessage());
            $webhook->markAsFailed($e->getMessage());
        }

        return response('OK', 200);
    }

    /**
     * Handle VTPass webhook.
     */
    public function vtpass(Request $request)
    {
        $data = $request->all();
        
        // Store webhook for processing
        $webhook = Webhook::create([
            'provider' => 'vtpass',
            'event_type' => 'transaction_update',
            'external_reference' => $data['requestId'] ?? null,
            'payload' => $data,
            'status' => 'pending',
        ]);

        try {
            $this->processVTPassWebhook($webhook);
        } catch (\Exception $e) {
            Log::error('VTPass webhook processing failed: ' . $e->getMessage());
            $webhook->markAsFailed($e->getMessage());
        }

        return response('OK', 200);
    }

    /**
     * Process Paystack webhook.
     */
    protected function processPaystackWebhook(Webhook $webhook): void
    {
        $data = $webhook->payload;
        $event = $data['event'];
        
        switch ($event) {
            case 'charge.success':
                $this->handlePaystackChargeSuccess($data['data']);
                break;
                
            case 'transfer.success':
                $this->handlePaystackTransferSuccess($data['data']);
                break;
                
            case 'transfer.failed':
                $this->handlePaystackTransferFailed($data['data']);
                break;
                
            default:
                Log::info("Unhandled Paystack event: {$event}");
                break;
        }
        
        $webhook->markAsProcessed();
    }

    /**
     * Process Monnify webhook.
     */
    protected function processMonnifyWebhook(Webhook $webhook): void
    {
        $data = $webhook->payload;
        $eventType = $data['eventType'];
        
        switch ($eventType) {
            case 'SUCCESSFUL_TRANSACTION':
                $this->handleMonnifySuccessfulTransaction($data['eventData']);
                break;
                
            case 'FAILED_TRANSACTION':
                $this->handleMonnifyFailedTransaction($data['eventData']);
                break;
                
            default:
                Log::info("Unhandled Monnify event: {$eventType}");
                break;
        }
        
        $webhook->markAsProcessed();
    }

    /**
     * Process Anchor webhook.
     */
    protected function processAnchorWebhook(Webhook $webhook): void
    {
        $data = $webhook->payload;
        $event = $data['event'];
        
        switch ($event) {
            case 'payment.successful':
                $this->handleAnchorPaymentSuccess($data['data']);
                break;
                
            case 'payment.failed':
                $this->handleAnchorPaymentFailed($data['data']);
                break;
                
            default:
                Log::info("Unhandled Anchor event: {$event}");
                break;
        }
        
        $webhook->markAsProcessed();
    }

    /**
     * Process VTPass webhook.
     */
    protected function processVTPassWebhook(Webhook $webhook): void
    {
        $data = $webhook->payload;
        
        if (isset($data['requestId'])) {
            $this->handleVTPassTransactionUpdate($data);
        }
        
        $webhook->markAsProcessed();
    }

    /**
     * Handle Paystack charge success.
     */
    protected function handlePaystackChargeSuccess(array $data): void
    {
        $reference = $data['reference'];
        
        $transaction = Transaction::where('external_reference', $reference)->first();
        
        if ($transaction && $transaction->status === 'pending') {
            DB::transaction(function () use ($transaction, $data) {
                // Credit user's wallet
                $wallet = $transaction->wallet;
                $wallet->credit($transaction->amount);
                
                // Mark transaction as completed
                $transaction->update([
                    'status' => 'completed',
                    'processed_at' => now(),
                    'metadata' => array_merge($transaction->metadata ?? [], [
                        'paystack_data' => $data,
                        'gateway_response' => $data['gateway_response'] ?? null,
                    ]),
                ]);
                
                Log::info("Wallet funding completed for user {$transaction->user_id}: ₦{$transaction->amount}");
            });
        }
    }

    /**
     * Handle Paystack transfer success.
     */
    protected function handlePaystackTransferSuccess(array $data): void
    {
        $reference = $data['reference'];
        
        $transaction = Transaction::where('external_reference', $reference)->first();
        
        if ($transaction && $transaction->status === 'processing') {
            $transaction->markAsCompleted();
            Log::info("Bank transfer completed: {$reference}");
        }
    }

    /**
     * Handle Paystack transfer failed.
     */
    protected function handlePaystackTransferFailed(array $data): void
    {
        $reference = $data['reference'];
        
        $transaction = Transaction::where('external_reference', $reference)->first();
        
        if ($transaction && $transaction->status === 'processing') {
            DB::transaction(function () use ($transaction, $data) {
                // Refund user's wallet
                $wallet = $transaction->wallet;
                $wallet->credit($transaction->total_amount);
                
                // Mark transaction as failed
                $transaction->markAsFailed($data['failure_reason'] ?? 'Transfer failed');
                
                Log::info("Bank transfer failed and refunded: {$reference}");
            });
        }
    }

    /**
     * Handle Monnify successful transaction.
     */
    protected function handleMonnifySuccessfulTransaction(array $data): void
    {
        $reference = $data['transactionReference'];
        
        // This would be for virtual account funding
        // Find user by account reference and credit their wallet
        Log::info("Monnify transaction successful: {$reference}");
    }

    /**
     * Handle Monnify failed transaction.
     */
    protected function handleMonnifyFailedTransaction(array $data): void
    {
        $reference = $data['transactionReference'];
        Log::info("Monnify transaction failed: {$reference}");
    }

    /**
     * Handle Anchor payment success.
     */
    protected function handleAnchorPaymentSuccess(array $data): void
    {
        $reference = $data['reference'];
        Log::info("Anchor payment successful: {$reference}");
    }

    /**
     * Handle Anchor payment failed.
     */
    protected function handleAnchorPaymentFailed(array $data): void
    {
        $reference = $data['reference'];
        Log::info("Anchor payment failed: {$reference}");
    }

    /**
     * Handle VTPass transaction update.
     */
    protected function handleVTPassTransactionUpdate(array $data): void
    {
        $requestId = $data['requestId'];
        
        $transaction = Transaction::where('external_reference', $requestId)->first();
        
        if ($transaction) {
            $status = $data['status'] ?? 'unknown';
            
            if ($status === 'delivered' || $status === 'successful') {
                $transaction->markAsCompleted();
            } elseif ($status === 'failed') {
                $transaction->markAsFailed($data['response_description'] ?? 'Service delivery failed');
            }
            
            Log::info("VTPass transaction updated: {$requestId} - {$status}");
        }
    }

    /**
     * Verify Paystack webhook signature.
     */
    protected function verifyPaystackSignature(string $payload, ?string $signature): bool
    {
        if (!$signature) {
            return false;
        }
        
        $secretKey = config('services.paystack.secret_key');
        $expectedSignature = hash_hmac('sha512', $payload, $secretKey);
        
        return hash_equals($expectedSignature, $signature);
    }
}