# Nigerian Fintech Platform

A comprehensive Laravel-based fintech platform specifically designed for the Nigerian market, providing digital wallet services, peer-to-peer transfers, bill payments, virtual bank accounts, and virtual card issuing.

## Features

### 🔐 Authentication & Security
- Multi-factor authentication (email + phone verification)
- Laravel Sanctum API authentication
- Transaction PIN protection
- Secure password hashing
- Session management

### 🆔 KYC Verification
- BVN (Bank Verification Number) integration via Dojah API
- NIN (National Identification Number) verification via VerifyMe API
- Document upload and verification
- Compliance with Nigerian financial regulations

### 💰 Digital Wallet System
- Multi-currency wallet support (primarily NGN)
- Real-time balance tracking
- Transaction limits and controls
- Automated transaction logging

### 🏦 Virtual Bank Accounts
- Monnify integration for virtual account creation
- Anchor API support as backup provider
- Automatic wallet funding via bank transfers
- Multiple bank account options per user

### 💸 Payment Processing
- Paystack integration for card payments
- P2P transfers between platform users
- Bank transfers to external accounts
- Transaction fee calculation and management

### 📱 Bill Payment Services
- Airtime purchases (MTN, Glo, Airtel, 9mobile)
- Data bundle purchases
- Electricity bill payments
- Cable TV subscriptions
- VTPass API integration

### 💳 Virtual Card Management
- Virtual card issuing via Bloc.io and Wallets Africa
- Card freeze/unfreeze functionality
- Transaction monitoring
- Spending limits and controls

### 👨‍💼 Admin Panel
- **Dedicated Admin Authentication**: Separate login system with guard isolation
- **Multi-role Support**: Super Admin, Admin, Support, and Viewer roles
- **KYC Management**: Review and approve/reject KYC submissions  
- **User Management**: View and manage user accounts
- **Transaction Monitoring**: Real-time transaction oversight
- **System Dashboard**: Overview statistics and system status
- **Permission-based Access**: Role-based permission system
- **Secure Admin Routes**: Protected with admin guard middleware

### 🔗 Webhook Integration
- Real-time payment notifications
- Automatic transaction updates
- Webhook signature verification
- Retry mechanisms for failed webhooks

## Tech Stack

- **Framework**: Laravel 11
- **Database**: MySQL/PostgreSQL
- **Authentication**: Laravel Sanctum
- **Queue**: Database/Redis
- **Storage**: Local/S3
- **Frontend**: Blade Templates (Mobile-first)

## Installation

### System Requirements

**Minimum Requirements:**
- PHP 8.2 or higher with required extensions
- Composer 2.0+
- MySQL 8.0+ or PostgreSQL 13+
- Node.js 18+ and NPM/Yarn
- Redis 6.0+ (for caching and queues)
- Minimum 2GB RAM, 10GB storage
- SSL certificate for production (required for webhooks)

**PHP Extensions Required:**
- BCMath, Ctype, cURL, DOM, Fileinfo, Filter, Hash, Mbstring, OpenSSL, PCRE, PDO, Session, Tokenizer, XML

### Local Development Setup

1. **Clone the repository**
   ```bash
   git clone https://github.com/codemoll/Fintech.git
   cd Fintech
   ```

2. **Install dependencies**
   ```bash
   composer install
   npm install
   ```

3. **Environment configuration**
   ```bash
   cp .env.example .env
   php artisan key:generate
   ```

4. **Configure database**
   Update your `.env` file with database credentials:
   ```env
   DB_CONNECTION=mysql
   DB_HOST=127.0.0.1
   DB_PORT=3306
   DB_DATABASE=fintech_nigeria
   DB_USERNAME=your_username
   DB_PASSWORD=your_password
   ```

5. **Run migrations and setup**
   ```bash
   php artisan migrate
   php artisan db:seed
   ```

6. **Compile assets**
   ```bash
   npm run build
   ```

7. **Start the development server**
   ```bash
   php artisan serve
   
   # In a separate terminal, start queue workers
   php artisan queue:work
   ```

## cPanel Hosting Deployment Guide

### Prerequisites for cPanel Hosting

- **Shared hosting with:**
  - PHP 8.2+ support
  - MySQL database access
  - SSL certificate installed
  - SSH access (preferred) or File Manager
  - At least 2GB storage space
  - Cron job access

### Step-by-Step cPanel Deployment

#### 1. Prepare Your Local Project

```bash
# Clone and prepare the project
git clone https://github.com/codemoll/Fintech.git
cd Fintech

# Install dependencies
composer install --optimize-autoloader --no-dev
npm install
npm run build

# Create deployment archive (exclude unnecessary files)
zip -r fintech-deploy.zip . -x "node_modules/*" ".git/*" "tests/*" "storage/logs/*" "*.md"
```

#### 2. Upload Files via cPanel

**Option A: SSH Upload (Preferred)**
```bash
# Upload to your hosting account
scp fintech-deploy.zip username@yourhost.com:~/
ssh username@yourhost.com
cd ~/
unzip fintech-deploy.zip
mv Fintech/* public_html/
```

**Option B: File Manager Upload**
1. Login to cPanel → File Manager
2. Navigate to `public_html` directory
3. Upload `fintech-deploy.zip`
4. Extract the archive
5. Move contents from `Fintech/` folder to `public_html/`

#### 3. Database Setup

1. **Create Database:**
   - cPanel → MySQL Databases
   - Create database: `username_fintech`
   - Create user: `username_fintech_user`
   - Add user to database with ALL PRIVILEGES

2. **Import Database Schema:**
   ```bash
   # Via SSH
   php artisan migrate --force
   
   # Or via phpMyAdmin
   # Import the SQL schema manually if SSH not available
   ```

#### 4. Environment Configuration

1. **Copy and configure `.env` file:**
   ```bash
   cp .env.example .env
   ```

2. **Edit `.env` with cPanel details:**
   ```env
   APP_NAME="Your Fintech Platform"
   APP_ENV=production
   APP_KEY=base64:YOUR_GENERATED_KEY_HERE
   APP_DEBUG=false
   APP_URL=https://yourdomain.com

   # Database Configuration
   DB_CONNECTION=mysql
   DB_HOST=localhost
   DB_PORT=3306
   DB_DATABASE=username_fintech
   DB_USERNAME=username_fintech_user
   DB_PASSWORD=your_database_password

   # Production Settings
   SESSION_DRIVER=database
   QUEUE_CONNECTION=database
   CACHE_STORE=database
   
   # Mail Configuration (using cPanel email)
   MAIL_MAILER=smtp
   MAIL_HOST=mail.yourdomain.com
   MAIL_PORT=465
   MAIL_USERNAME=noreply@yourdomain.com
   MAIL_PASSWORD=your_email_password
   MAIL_ENCRYPTION=ssl
   MAIL_FROM_ADDRESS=noreply@yourdomain.com
   MAIL_FROM_NAME="${APP_NAME}"

   # Payment Provider Keys (PRODUCTION)
   PAYSTACK_PUBLIC_KEY=pk_live_xxxxxxxxxxxxx
   PAYSTACK_SECRET_KEY=sk_live_xxxxxxxxxxxxx
   PAYSTACK_PAYMENT_URL=https://api.paystack.co
   PAYSTACK_MERCHANT_EMAIL=your@email.com

   # Add other production service keys here
   ```

#### 5. File Permissions Setup

```bash
# Set correct permissions
chmod -R 755 public_html/
chmod -R 775 public_html/storage/
chmod -R 775 public_html/bootstrap/cache/
chmod 644 public_html/.env

# Secure sensitive files
chmod 600 public_html/.env
chmod -R 750 public_html/storage/logs/
```

#### 6. Laravel Optimization for Production

```bash
# Clear and cache configurations
php artisan config:cache
php artisan route:cache
php artisan view:cache

# Generate application key if not set
php artisan key:generate --force

# Run migrations
php artisan migrate --force
```

#### 7. Subdomain Setup (Optional but Recommended)

1. **Create Subdomain:**
   - cPanel → Subdomains
   - Create: `api.yourdomain.com` → points to `public_html/public`

2. **SSL Certificate:**
   - cPanel → SSL/TLS → Let's Encrypt
   - Enable for main domain and subdomain

#### 8. Cron Jobs Setup

1. **Access Cron Jobs:**
   - cPanel → Cron Jobs

2. **Add Laravel Scheduler:**
   ```bash
   # Run every minute
   * * * * * cd /home/username/public_html && php artisan schedule:run >> /dev/null 2>&1
   ```

3. **Add Queue Worker (if supported):**
   ```bash
   # Every 5 minutes, restart queue workers
   */5 * * * * cd /home/username/public_html && php artisan queue:restart
   ```

#### 9. Webhook URLs Configuration

Update your payment provider webhooks to point to:
```
https://yourdomain.com/api/webhooks/paystack
https://yourdomain.com/api/webhooks/monnify
https://yourdomain.com/api/webhooks/anchor
https://yourdomain.com/api/webhooks/vtpass
```

#### 10. Security Headers (.htaccess)

Create/update `.htaccess` in `public_html/public/`:
```apache
<IfModule mod_rewrite.c>
    <IfModule mod_negotiation.c>
        Options -MultiViews -Indexes
    </IfModule>

    RewriteEngine On

    # Handle Angular and Vue History API fallback requests
    RewriteCond %{REQUEST_FILENAME} -d [OR]
    RewriteCond %{REQUEST_FILENAME} -f
    RewriteRule ^ ^$1 [N]

    RewriteCond %{REQUEST_URI} (\.\w+$) [NC]
    RewriteRule ^(.*)$ public/$1

    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteRule ^ server.php

    # Security Headers
    Header always set X-Frame-Options DENY
    Header always set X-Content-Type-Options nosniff
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Strict-Transport-Security "max-age=31536000; includeSubDomains"
    Header always set Content-Security-Policy "default-src 'self'"
</IfModule>

# Block access to sensitive files
<Files .env>
    Order allow,deny
    Deny from all
</Files>

<Files composer.lock>
    Order allow,deny
    Deny from all
</Files>
```

### Post-Deployment Verification

1. **Test Application:**
   ```bash
   # Check if app loads
   curl -I https://yourdomain.com
   
   # Test API endpoints
   curl https://yourdomain.com/api/auth/register
   ```

2. **Check Logs:**
   ```bash
   tail -f storage/logs/laravel.log
   ```

3. **Verify Database Connection:**
   ```bash
   php artisan tinker
   # Test: DB::connection()->getPdo();
   ```

### Troubleshooting Common cPanel Issues

#### Issue: 500 Internal Server Error
**Solutions:**
1. Check `.env` file exists and has correct values
2. Verify file permissions (755 for directories, 644 for files)
3. Run `php artisan config:clear`
4. Check error logs in cPanel → Error Logs

#### Issue: Database Connection Failed
**Solutions:**
1. Verify database credentials in `.env`
2. Check if database user has proper privileges
3. Confirm database exists and is accessible

#### Issue: Queue Jobs Not Processing
**Solutions:**
1. Verify cron job is set up correctly
2. Use `database` queue driver instead of `redis`
3. Check if hosting supports background processes

#### Issue: Assets Not Loading
**Solutions:**
1. Ensure `npm run build` was executed
2. Check if `public/` directory exists
3. Verify asset URLs in configuration

### Maintenance Mode

```bash
# Enable maintenance mode
php artisan down --message="Upgrading system" --retry=60

# Disable maintenance mode
php artisan up
```

### Backup Strategy

1. **Database Backup:**
   ```bash
   # Daily database backup
   mysqldump -u username -p database_name > backup_$(date +%Y%m%d).sql
   ```

2. **File Backup:**
   ```bash
   # Weekly file backup
   tar -czf backup_files_$(date +%Y%m%d).tar.gz public_html/
   ```

### Production Monitoring

1. **Enable Error Logging:**
   ```env
   LOG_CHANNEL=daily
   LOG_LEVEL=error
   ```

2. **Monitor Key Metrics:**
   - Application response time
   - Database query performance
   - Failed queue jobs
   - Error rates

3. **Set Up Alerts:**
   - Email alerts for application errors
   - Transaction failure notifications
   - System resource monitoring

## Configuration

### Payment Providers

#### Paystack
```env
PAYSTACK_PUBLIC_KEY=pk_test_xxxxxxxxxxxxxxxxxxxxxxxxxx
PAYSTACK_SECRET_KEY=sk_test_xxxxxxxxxxxxxxxxxxxxxxxxxx
PAYSTACK_PAYMENT_URL=https://api.paystack.co
PAYSTACK_MERCHANT_EMAIL=merchant@example.com
```

#### Monnify
```env
MONNIFY_API_KEY=MK_TEST_xxxxxxxxxxxxxxxxxxxxxxxxxx
MONNIFY_SECRET_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
MONNIFY_CONTRACT_CODE=xxxxxxxxxxxxxxxxxxxxxxxxxx
MONNIFY_BASE_URL=https://sandbox.monnify.com
```

### KYC Providers

#### Dojah
```env
DOJAH_APP_ID=xxxxxxxxxxxxxxxxxxxxxxxxxx
DOJAH_SECRET_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
DOJAH_BASE_URL=https://sandbox.dojah.io
```

#### VerifyMe
```env
VERIFYME_USERNAME=xxxxxxxxxxxxxxxxxxxxxxxxxx
VERIFYME_PASSWORD=xxxxxxxxxxxxxxxxxxxxxxxxxx
VERIFYME_BASE_URL=https://developer.verifymy.co.uk
```

### Bill Payment

#### VTPass
```env
VTPASS_USERNAME=xxxxxxxxxxxxxxxxxxxxxxxxxx
VTPASS_PASSWORD=xxxxxxxxxxxxxxxxxxxxxxxxxx
VTPASS_BASE_URL=https://sandbox.vtpass.com
```

### Virtual Cards

#### Bloc.io
```env
BLOC_SECRET_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
BLOC_PUBLIC_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
BLOC_BASE_URL=https://blochq.io
```

#### Wallets Africa
```env
WALLETS_AFRICA_PUBLIC_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
WALLETS_AFRICA_SECRET_KEY=xxxxxxxxxxxxxxxxxxxxxxxxxx
WALLETS_AFRICA_BASE_URL=https://api.wallets.africa
```

## API Documentation

### Authentication Endpoints

```http
POST /api/auth/register
POST /api/auth/login
POST /api/auth/verify-phone
POST /api/auth/logout
```

### Wallet Endpoints

```http
GET /api/wallet
POST /api/wallet/fund
GET /api/wallet/transactions
```

### Transaction Endpoints

```http
GET /api/transactions
POST /api/transactions/p2p
POST /api/transactions/bank-transfer
```

### KYC Endpoints

```http
GET /api/kyc/status
POST /api/kyc/bvn
POST /api/kyc/nin
POST /api/kyc/upload-document
```

### Bill Payment Endpoints

```http
GET /api/bills/providers
POST /api/bills/airtime
POST /api/bills/data
POST /api/bills/electricity
```

## Security Best Practices

### Transaction Security
- All financial transactions require PIN verification
- Transaction limits enforced at wallet level
- Real-time fraud detection (to be implemented)
- Secure API communication with HTTPS

### Data Protection
- PII encryption for sensitive data
- Secure storage of KYC documents
- Regular security audits recommended
- GDPR/Nigerian Data Protection compliance

### API Security
- Rate limiting on all endpoints
- Request signature verification for webhooks
- Token-based authentication with expiration
- Input validation and sanitization

## Deployment

### Production Environment

1. **Server Requirements**
   - PHP 8.2+ with required extensions
   - MySQL/PostgreSQL database
   - Redis for caching and queues
   - Nginx/Apache web server
   - SSL certificate (required)

2. **Environment Variables**
   ```bash
   APP_ENV=production
   APP_DEBUG=false
   APP_URL=https://your-domain.com
   ```

3. **Optimize for Production**
   ```bash
   php artisan config:cache
   php artisan route:cache
   php artisan view:cache
   composer install --optimize-autoloader --no-dev
   ```

4. **Queue Workers**
   ```bash
   php artisan queue:work --daemon
   ```

5. **Scheduled Tasks**
   Add to crontab:
   ```bash
   * * * * * cd /path-to-your-project && php artisan schedule:run >> /dev/null 2>&1
   ```

### Docker Deployment (Optional)

```dockerfile
# Use the official PHP image
FROM php:8.2-fpm

# Install dependencies
RUN apt-get update && apt-get install -y \
    git \
    curl \
    libpng-dev \
    libonig-dev \
    libxml2-dev \
    zip \
    unzip

# Install PHP extensions
RUN docker-php-ext-install pdo_mysql mbstring exif pcntl bcmath gd

# Get Composer
COPY --from=composer:latest /usr/bin/composer /usr/bin/composer

# Set working directory
WORKDIR /var/www

# Copy project files
COPY . /var/www

# Install dependencies
RUN composer install --optimize-autoloader --no-dev

# Set permissions
RUN chown -R www-data:www-data /var/www
```

## Testing

<!-- TODO: CRITICAL - No testing infrastructure exists -->
<!-- TODO: Create tests directory and phpunit.xml configuration -->
<!-- TODO: Implement comprehensive test suite for production readiness -->

### Running Tests
```bash
php artisan test  # TODO: Set up testing infrastructure first
```

### Feature Testing
```bash
php artisan test --filter=FeatureTest  # TODO: Create feature tests for all endpoints
```

### Unit Testing
```bash
php artisan test --filter=UnitTest  # TODO: Create unit tests for models and services
```

<!-- TODO: Add testing guidelines and examples -->
<!-- TODO: Set up test database configuration -->
<!-- TODO: Create factories for all models -->
<!-- TODO: Add integration tests for external APIs -->

## Admin Access & Management

### Admin Authentication

The platform includes a separate admin authentication system with dedicated routes and guards:

#### Accessing the Admin Panel

1. **Admin Login URL**: `/admin/login`
2. **Default Admin Dashboard**: `/admin/dashboard`

#### Creating Admin Users

To create admin users, use Laravel Tinker or database seeding:

```bash
php artisan tinker

# Create a super admin
App\Models\Admin::create([
    'name' => 'Super Admin',
    'email' => 'admin@yourfintech.com',
    'password' => bcrypt('secure_password'),
    'role' => 'super_admin',
    'is_active' => true
]);

# Create a regular admin
App\Models\Admin::create([
    'name' => 'KYC Manager',
    'email' => 'kyc@yourfintech.com',
    'password' => bcrypt('secure_password'),
    'role' => 'admin',
    'permissions' => ['users.view', 'kyc.view', 'kyc.approve'],
    'is_active' => true
]);
```

#### Admin Roles & Permissions

| Role | Permissions | Description |
|------|-------------|-------------|
| `super_admin` | All permissions (`*`) | Full system access |
| `admin` | User management, KYC, transactions, wallets | Standard admin access |
| `support` | User view, transaction view, support tickets | Customer support role |
| `viewer` | Dashboard view, reports view | Read-only access |

### Admin Dashboard Features

- **User Statistics**: Total users, new registrations
- **KYC Management**: Pending verifications, approval queue
- **Transaction Overview**: Transaction counts and monitoring
- **System Status**: Real-time system health indicators
- **Quick Actions**: Direct access to common admin tasks

## KYC Verification Flow

### For Users

After registration, users are redirected to complete KYC verification:

1. **KYC Dashboard** (`/kyc`): Overview of verification options
2. **BVN Verification** (`/kyc/bvn`): Bank Verification Number verification
3. **NIN Verification** (`/kyc/nin`): National Identification Number verification  
4. **Document Upload** (`/kyc/document`): Government-issued ID upload

### KYC Forms & Requirements

#### BVN Verification
- 11-digit Bank Verification Number
- Full name (as registered with bank)
- Date of birth
- Real-time verification via Dojah API

#### NIN Verification  
- 11-digit National Identification Number
- Full name (as on NIN slip)
- Date of birth
- Real-time verification via VerifyMe API

#### Document Upload
- Accepted documents: International Passport, Driver's License, Voter's Card
- File formats: PDF, JPG, PNG (max 2MB)
- Manual review by admin team
- Processing time: 24-48 hours

### For Admins

Admins can review and manage KYC submissions through the admin panel:

1. **KYC Queue**: View pending verifications
2. **Document Review**: Approve/reject uploaded documents
3. **Verification History**: Track KYC completion status
4. **Compliance Reports**: Generate KYC compliance reports

## New Routes Added

### Admin Routes
```
GET|POST /admin/login          # Admin authentication
GET     /admin/dashboard       # Admin dashboard  
POST    /admin/logout          # Admin logout
```

### KYC Routes
```
GET     /kyc                   # KYC overview dashboard
GET     /kyc/bvn              # BVN verification form
POST    /kyc/bvn              # Submit BVN verification
GET     /kyc/nin              # NIN verification form  
POST    /kyc/nin              # Submit NIN verification
GET     /kyc/document         # Document upload form
POST    /kyc/document         # Submit document upload
```

## Contributing

1. Fork the repository
2. Create a feature branch (`git checkout -b feature/amazing-feature`)
3. Commit your changes (`git commit -m 'Add some amazing feature'`)
4. Push to the branch (`git push origin feature/amazing-feature`)
5. Open a Pull Request

## License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## Support

For support and questions:
- Email: support@yourfintech.com
- Documentation: https://docs.yourfintech.com
- Issues: https://github.com/codemoll/Fintech/issues

## Roadmap

- [ ] Advanced fraud detection
- [ ] Crypto wallet integration
- [ ] International transfers
- [ ] Merchant payment gateway
- [ ] Mobile applications (iOS/Android)
- [ ] Investment platform integration
- [ ] Loan/credit services
- [ ] Insurance products integration

## Compliance

This platform is designed to comply with:
- Central Bank of Nigeria (CBN) regulations
- Nigerian Data Protection Regulation (NDPR)
- Anti-Money Laundering (AML) requirements
- Know Your Customer (KYC) compliance

**Note**: Always consult with legal and compliance experts before deploying financial services in production.