<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\Wallet;
use App\Models\Transaction;
use App\Services\PaystackService;

class WalletController extends Controller
{
    protected $paystackService;

    public function __construct(PaystackService $paystackService)
    {
        $this->paystackService = $paystackService;
    }

    /**
     * Show wallet dashboard.
     */
    public function index()
    {
        $user = Auth::user();
        $wallet = $user->wallet;
        $recentTransactions = $user->transactions()
            ->latest()
            ->limit(10)
            ->get();

        return view('wallet.index', compact('wallet', 'recentTransactions'));
    }

    /**
     * Fund wallet via Paystack.
     */
    public function fund(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:100|max:500000',
        ]);

        $user = Auth::user();
        $wallet = $user->wallet;

        if (!$wallet) {
            return back()->with('error', 'Wallet not found.');
        }

        DB::beginTransaction();
        try {
            // Create pending transaction
            $transaction = Transaction::create([
                'user_id' => $user->id,
                'wallet_id' => $wallet->id,
                'type' => 'credit',
                'category' => 'wallet_funding',
                'amount' => $request->amount,
                'fee' => 0, // No fee for wallet funding
                'total_amount' => $request->amount,
                'currency' => 'NGN',
                'status' => 'pending',
                'description' => 'Wallet funding via Paystack',
                'payment_method' => 'paystack',
            ]);

            // Initialize Paystack payment
            $paymentData = $this->paystackService->initializePayment([
                'amount' => $request->amount,
                'email' => $user->email,
                'reference' => $transaction->reference,
                'currency' => 'NGN',
                'callback_url' => route('wallet.index'),
                'metadata' => [
                    'user_id' => $user->id,
                    'transaction_id' => $transaction->id,
                    'type' => 'wallet_funding',
                ],
            ]);

            if (!$paymentData['success']) {
                DB::rollBack();
                return back()->with('error', $paymentData['message']);
            }

            // Update transaction with external reference
            $transaction->update([
                'external_reference' => $paymentData['reference'],
                'metadata' => [
                    'paystack_authorization_url' => $paymentData['authorization_url'],
                    'paystack_access_code' => $paymentData['access_code'],
                ],
            ]);

            DB::commit();

            // Redirect to Paystack payment page
            return redirect($paymentData['authorization_url']);
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Payment initialization failed. Please try again.');
        }
    }
}