<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Models\KycVerification;
use App\Services\DojahService;
use App\Services\VerifyMeService;

class KycController extends Controller
{
    protected $dojahService;
    protected $verifyMeService;

    public function __construct(DojahService $dojahService, VerifyMeService $verifyMeService)
    {
        $this->dojahService = $dojahService;
        $this->verifyMeService = $verifyMeService;
    }

    /**
     * Show KYC verification page.
     */
    public function index()
    {
        $user = Auth::user();
        $kycVerifications = $user->kycVerifications;
        
        return view('kyc.index', compact('kycVerifications'));
    }

    /**
     * Show BVN verification form.
     */
    public function showBvnForm()
    {
        return view('kyc.bvn');
    }

    /**
     * Show NIN verification form.
     */
    public function showNinForm()
    {
        return view('kyc.nin');
    }

    /**
     * Show document upload form.
     */
    public function showDocumentForm()
    {
        return view('kyc.document');
    }

    /**
     * Verify BVN.
     */
    public function verifyBvn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'bvn' => 'required|numeric|digits:11',
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'date_of_birth' => 'required|date',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = Auth::user();

        // Check if BVN verification already exists
        $existingVerification = KycVerification::where('user_id', $user->id)
            ->where('type', 'bvn')
            ->first();

        if ($existingVerification && $existingVerification->isVerified()) {
            return back()->with('error', 'BVN already verified for this account.');
        }

        try {
            // Call Dojah API for BVN verification
            $verificationResult = $this->dojahService->verifyBvn([
                'bvn' => $request->bvn,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'date_of_birth' => $request->date_of_birth,
            ]);

            if ($existingVerification) {
                $existingVerification->delete();
            }

            $kycVerification = KycVerification::create([
                'user_id' => $user->id,
                'type' => 'bvn',
                'verification_id' => $request->bvn,
                'provider' => 'dojah',
                'provider_reference' => $verificationResult['reference'],
                'status' => $verificationResult['status'] === 'verified' ? 'verified' : 'failed',
                'verification_data' => $verificationResult['data'] ?? [],
                'failure_reason' => $verificationResult['message'] ?? null,
                'verified_at' => $verificationResult['status'] === 'verified' ? now() : null,
            ]);

            if ($kycVerification->isVerified()) {
                return redirect()->route('dashboard')
                    ->with('success', 'BVN verification successful! You can now access all features.');
            } else {
                return back()->with('error', 'BVN verification failed: ' . $kycVerification->failure_reason);
            }
        } catch (\Exception $e) {
            return back()->with('error', 'Verification failed. Please try again later.');
        }
    }

    /**
     * Verify NIN.
     */
    public function verifyNin(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nin' => 'required|numeric|digits:11',
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'date_of_birth' => 'required|date',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = Auth::user();

        try {
            // Call VerifyMe API for NIN verification
            $verificationResult = $this->verifyMeService->verifyNin([
                'nin' => $request->nin,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'date_of_birth' => $request->date_of_birth,
            ]);

            $kycVerification = KycVerification::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'type' => 'nin'
                ],
                [
                    'verification_id' => $request->nin,
                    'provider' => 'verifyme',
                    'provider_reference' => $verificationResult['reference'],
                    'status' => $verificationResult['status'] === 'verified' ? 'verified' : 'failed',
                    'verification_data' => $verificationResult['data'] ?? [],
                    'failure_reason' => $verificationResult['message'] ?? null,
                    'verified_at' => $verificationResult['status'] === 'verified' ? now() : null,
                ]
            );

            if ($kycVerification->isVerified()) {
                return back()->with('success', 'NIN verification successful!');
            } else {
                return back()->with('error', 'NIN verification failed: ' . $kycVerification->failure_reason);
            }
        } catch (\Exception $e) {
            return back()->with('error', 'Verification failed. Please try again later.');
        }
    }

    /**
     * Upload KYC document.
     */
    public function uploadDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'document_type' => 'required|in:passport,drivers_license,voters_card',
            'document' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = Auth::user();
        
        // Store the document
        $documentPath = $request->file('document')->store('kyc-documents', 'local');

        $kycVerification = KycVerification::updateOrCreate(
            [
                'user_id' => $user->id,
                'type' => $request->document_type
            ],
            [
                'document_path' => $documentPath,
                'status' => 'pending',
            ]
        );

        return back()->with('success', 'Document uploaded successfully! It will be reviewed within 24 hours.');
    }
}