<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Models\Transaction;
use App\Services\PaystackService;

class TransactionController extends Controller
{
    protected $paystackService;

    public function __construct(PaystackService $paystackService)
    {
        $this->paystackService = $paystackService;
    }

    /**
     * Show transaction history.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        $query = $user->transactions()->latest();

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $transactions = $query->paginate(20);

        // Calculate summary statistics
        $summaryQuery = $user->transactions();
        
        // Apply same filters for summary
        if ($request->filled('type')) {
            $summaryQuery->where('type', $request->type);
        }
        if ($request->filled('category')) {
            $summaryQuery->where('category', $request->category);
        }
        if ($request->filled('status')) {
            $summaryQuery->where('status', $request->status);
        }
        if ($request->filled('date_from')) {
            $summaryQuery->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $summaryQuery->whereDate('created_at', '<=', $request->date_to);
        }

        $summary = [
            'total_credit' => $summaryQuery->where('type', 'credit')->sum('amount'),
            'total_debit' => $summaryQuery->where('type', 'debit')->sum('amount'),
            'total_count' => $summaryQuery->count(),
            'total_fees' => $summaryQuery->sum('fee'),
        ];

        return view('transactions.index', compact('transactions', 'summary'));
    }

    /**
     * Show P2P transfer form.
     */
    public function showP2PForm()
    {
        return view('transactions.p2p');
    }

    /**
     * Process P2P transfer.
     */
    public function p2pTransfer(Request $request)
    {
        $request->validate([
            'recipient_identifier' => 'required|string', // email or phone
            'amount' => 'required|numeric|min:100|max:100000',
            'description' => 'nullable|string|max:255',
            'transaction_pin' => 'required|string',
        ]);

        $user = Auth::user();
        $wallet = $user->wallet;

        // Verify transaction PIN
        if (!$user->transaction_pin || !Hash::check($request->transaction_pin, $user->transaction_pin)) {
            return back()->withErrors(['transaction_pin' => 'Invalid transaction PIN.']);
        }

        // Find recipient by email or phone
        $recipient = User::where('email', $request->recipient_identifier)
            ->orWhere('phone', $request->recipient_identifier)
            ->first();

        if (!$recipient) {
            return back()->withErrors(['recipient_identifier' => 'Recipient not found.']);
        }

        if ($recipient->id === $user->id) {
            return back()->withErrors(['recipient_identifier' => 'Cannot transfer to yourself.']);
        }

        $recipientWallet = $recipient->wallet;
        if (!$recipientWallet) {
            return back()->withErrors(['recipient_identifier' => 'Recipient wallet not found.']);
        }

        $amount = $request->amount;
        $fee = $this->calculateP2PFee($amount);
        $totalAmount = $amount + $fee;

        // Check if user can perform transaction
        if (!$wallet->canTransact($totalAmount)) {
            return back()->with('error', 'Insufficient balance or transaction limit exceeded.');
        }

        DB::beginTransaction();
        try {
            // Lock balance for the transaction
            $wallet->lockBalance($totalAmount);

            // Create debit transaction for sender
            $debitTransaction = Transaction::create([
                'user_id' => $user->id,
                'wallet_id' => $wallet->id,
                'type' => 'debit',
                'category' => 'p2p_transfer',
                'amount' => $amount,
                'fee' => $fee,
                'total_amount' => $totalAmount,
                'currency' => 'NGN',
                'status' => 'processing',
                'description' => $request->description ?: "P2P transfer to {$recipient->full_name}",
                'recipient_user_id' => $recipient->id,
                'metadata' => [
                    'transfer_type' => 'p2p',
                    'recipient_name' => $recipient->full_name,
                    'recipient_email' => $recipient->email,
                ],
            ]);

            // Create credit transaction for recipient
            $creditTransaction = Transaction::create([
                'user_id' => $recipient->id,
                'wallet_id' => $recipientWallet->id,
                'type' => 'credit',
                'category' => 'p2p_transfer',
                'amount' => $amount,
                'fee' => 0,
                'total_amount' => $amount,
                'currency' => 'NGN',
                'status' => 'processing',
                'description' => "P2P transfer from {$user->full_name}",
                'metadata' => [
                    'transfer_type' => 'p2p',
                    'sender_name' => $user->full_name,
                    'sender_email' => $user->email,
                    'related_transaction_id' => $debitTransaction->id,
                ],
            ]);

            // Process the transfer
            $wallet->debit($totalAmount);
            $wallet->unlockBalance($totalAmount);
            $recipientWallet->credit($amount);

            // Mark transactions as completed
            $debitTransaction->markAsCompleted();
            $creditTransaction->markAsCompleted();

            DB::commit();

            return redirect()->route('transactions.index')
                ->with('success', "Successfully transferred ₦{$amount} to {$recipient->full_name}");
        } catch (\Exception $e) {
            DB::rollBack();
            $wallet->unlockBalance($totalAmount);
            
            return back()->with('error', 'Transfer failed. Please try again.');
        }
    }

    /**
     * Show bank transfer form.
     */
    public function showBankTransferForm()
    {
        $banks = $this->paystackService->getBanks();
        
        return view('transactions.bank-transfer', [
            'banks' => $banks['success'] ? $banks['banks'] : []
        ]);
    }

    /**
     * Process bank transfer.
     */
    public function bankTransfer(Request $request)
    {
        $request->validate([
            'account_number' => 'required|string|digits:10',
            'bank_code' => 'required|string',
            'amount' => 'required|numeric|min:100|max:100000',
            'description' => 'nullable|string|max:255',
            'transaction_pin' => 'required|string',
        ]);

        $user = Auth::user();
        $wallet = $user->wallet;

        // Verify transaction PIN
        if (!$user->transaction_pin || !Hash::check($request->transaction_pin, $user->transaction_pin)) {
            return back()->withErrors(['transaction_pin' => 'Invalid transaction PIN.']);
        }

        // Resolve account name
        $accountDetails = $this->paystackService->resolveAccount(
            $request->account_number,
            $request->bank_code
        );

        if (!$accountDetails['success']) {
            return back()->withErrors(['account_number' => 'Could not resolve account details.']);
        }

        $amount = $request->amount;
        $fee = $this->calculateBankTransferFee($amount);
        $totalAmount = $amount + $fee;

        // Check if user can perform transaction
        if (!$wallet->canTransact($totalAmount)) {
            return back()->with('error', 'Insufficient balance or transaction limit exceeded.');
        }

        DB::beginTransaction();
        try {
            // Create transaction
            $transaction = Transaction::create([
                'user_id' => $user->id,
                'wallet_id' => $wallet->id,
                'type' => 'debit',
                'category' => 'bank_transfer',
                'amount' => $amount,
                'fee' => $fee,
                'total_amount' => $totalAmount,
                'currency' => 'NGN',
                'status' => 'processing',
                'description' => $request->description ?: "Bank transfer to {$accountDetails['account_name']}",
                'recipient_account_number' => $request->account_number,
                'recipient_bank_code' => $request->bank_code,
                'metadata' => [
                    'transfer_type' => 'bank',
                    'recipient_name' => $accountDetails['account_name'],
                ],
            ]);

            // TODO: Implement actual bank transfer via Paystack or other provider
            // For now, mark as completed
            $wallet->debit($totalAmount);
            $transaction->markAsCompleted();

            DB::commit();

            return redirect()->route('transactions.index')
                ->with('success', "Successfully transferred ₦{$amount} to {$accountDetails['account_name']}");
        } catch (\Exception $e) {
            DB::rollBack();
            
            return back()->with('error', 'Transfer failed. Please try again.');
        }
    }

    /**
     * Calculate P2P transfer fee.
     */
    protected function calculateP2PFee(float $amount): float
    {
        // Free P2P transfers under ₦5,000
        if ($amount < 5000) {
            return 0;
        }

        // ₦25 flat fee for transfers above ₦5,000
        return 25;
    }

    /**
     * Calculate bank transfer fee.
     */
    protected function calculateBankTransferFee(float $amount): float
    {
        // ₦50 flat fee for all bank transfers
        return 50;
    }
}