@extends('layouts.admin-sidebar')

@section('title', 'Profit Margins Management')
@section('page-title', 'Profit Margins')
@section('page-description', 'Configure profit margins for airtime and data services')

@section('content')
<div class="space-y-6">
    <!-- Page Header with Actions -->
    <div class="flex justify-between items-center">
        <div>
            <h2 class="text-2xl font-bold text-gray-900">Profit Margins Management</h2>
            <p class="text-gray-600 mt-1">Set and manage profit margins for different networks and services</p>
        </div>
        <div class="flex space-x-3">
            <button class="btn-secondary" onclick="resetToDefaults()">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                </svg>
                Reset to Defaults
            </button>
            <button class="btn-primary" onclick="saveAllMargins()">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7H5a2 2 0 00-2 2v9a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-3m-1 4l-3-3m0 0l-3 3m3-3v12"></path>
                </svg>
                Save All Changes
            </button>
        </div>
    </div>

    <!-- Global Settings -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Global Margin Settings</h3>
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Default Airtime Margin (%)</label>
                <input type="number" 
                       value="{{ $margins['settings']['default_airtime_margin'] }}" 
                       min="{{ $margins['settings']['minimum_margin'] }}" 
                       max="{{ $margins['settings']['maximum_margin'] }}" 
                       step="0.1"
                       class="input-primary" 
                       id="default_airtime_margin">
                <p class="text-xs text-gray-500 mt-1">Applied to new networks</p>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Default Data Margin (%)</label>
                <input type="number" 
                       value="{{ $margins['settings']['default_data_margin'] }}" 
                       min="{{ $margins['settings']['minimum_margin'] }}" 
                       max="{{ $margins['settings']['maximum_margin'] }}" 
                       step="0.1"
                       class="input-primary" 
                       id="default_data_margin">
                <p class="text-xs text-gray-500 mt-1">Applied to new networks</p>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Minimum Margin (%)</label>
                <input type="number" 
                       value="{{ $margins['settings']['minimum_margin'] }}" 
                       min="0" 
                       max="5" 
                       step="0.1"
                       class="input-primary" 
                       id="minimum_margin">
                <p class="text-xs text-gray-500 mt-1">System minimum</p>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Maximum Margin (%)</label>
                <input type="number" 
                       value="{{ $margins['settings']['maximum_margin'] }}" 
                       min="5" 
                       max="50" 
                       step="0.1"
                       class="input-primary" 
                       id="maximum_margin">
                <p class="text-xs text-gray-500 mt-1">System maximum</p>
            </div>
        </div>
    </div>

    <!-- Network-Specific Margins -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Network-Specific Margins</h3>
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
            @foreach($margins['networks'] as $network_code => $network)
            <div class="border border-gray-200 rounded-lg p-4">
                <!-- Network Header -->
                <div class="flex items-center justify-between mb-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 rounded-lg flex items-center justify-center mr-3
                            {{ $network_code === 'mtn' ? 'bg-yellow-100' : 
                               ($network_code === 'glo' ? 'bg-green-100' : 
                               ($network_code === 'airtel' ? 'bg-red-100' : 'bg-green-100')) }}">
                            <span class="font-bold text-sm
                                {{ $network_code === 'mtn' ? 'text-yellow-800' : 
                                   ($network_code === 'glo' ? 'text-green-800' : 
                                   ($network_code === 'airtel' ? 'text-red-800' : 'text-green-800')) }}">
                                {{ strtoupper(substr($network['name'], 0, 3)) }}
                            </span>
                        </div>
                        <div>
                            <h4 class="text-lg font-semibold text-gray-900">{{ $network['name'] }}</h4>
                            <span class="inline-flex px-2 py-1 text-xs font-medium rounded-full
                                {{ $network['status'] === 'active' ? 'bg-emerald-100 text-emerald-800' : 'bg-red-100 text-red-800' }}">
                                {{ ucfirst($network['status']) }}
                            </span>
                        </div>
                    </div>
                    <div class="flex items-center space-x-2">
                        <button class="text-gray-400 hover:text-gray-600" onclick="toggleNetworkStatus('{{ $network_code }}')">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"></path>
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                            </svg>
                        </button>
                    </div>
                </div>

                <!-- Margin Controls -->
                <div class="grid grid-cols-2 gap-4">
                    <!-- Airtime Margin -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Airtime Margin (%)</label>
                        <div class="relative">
                            <input type="number" 
                                   value="{{ $network['airtime_margin'] }}" 
                                   min="{{ $margins['settings']['minimum_margin'] }}" 
                                   max="{{ $margins['settings']['maximum_margin'] }}" 
                                   step="0.1"
                                   class="input-primary" 
                                   id="airtime_margin_{{ $network_code }}"
                                   onchange="calculateProfit('{{ $network_code }}', 'airtime')">
                            <div class="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                <span class="text-gray-500 text-sm">%</span>
                            </div>
                        </div>
                        <div class="mt-2 text-xs text-gray-600">
                            <span>Example: ₦100 → ₦{{ number_format(100 + ($network['airtime_margin'] / 100 * 100), 2) }}</span>
                        </div>
                    </div>

                    <!-- Data Margin -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Data Margin (%)</label>
                        <div class="relative">
                            <input type="number" 
                                   value="{{ $network['data_margin'] }}" 
                                   min="{{ $margins['settings']['minimum_margin'] }}" 
                                   max="{{ $margins['settings']['maximum_margin'] }}" 
                                   step="0.1"
                                   class="input-primary" 
                                   id="data_margin_{{ $network_code }}"
                                   onchange="calculateProfit('{{ $network_code }}', 'data')">
                            <div class="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                <span class="text-gray-500 text-sm">%</span>
                            </div>
                        </div>
                        <div class="mt-2 text-xs text-gray-600">
                            <span>Example: ₦1000 → ₦{{ number_format(1000 + ($network['data_margin'] / 100 * 1000), 2) }}</span>
                        </div>
                    </div>
                </div>

                <!-- Quick Margin Presets -->
                <div class="mt-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Quick Presets</label>
                    <div class="flex space-x-2">
                        <button class="px-3 py-1 text-xs bg-gray-100 hover:bg-gray-200 rounded" 
                                onclick="setMargins('{{ $network_code }}', 1.5, 2.0)">Conservative</button>
                        <button class="px-3 py-1 text-xs bg-blue-100 hover:bg-blue-200 text-blue-800 rounded" 
                                onclick="setMargins('{{ $network_code }}', 2.5, 3.0)">Balanced</button>
                        <button class="px-3 py-1 text-xs bg-green-100 hover:bg-green-200 text-green-800 rounded" 
                                onclick="setMargins('{{ $network_code }}', 3.5, 4.0)">Aggressive</button>
                    </div>
                </div>
            </div>
            @endforeach
        </div>
    </div>

    <!-- Margin Analytics -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Margin Performance Analytics</h3>
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <!-- Revenue Projection -->
            <div>
                <h4 class="text-sm font-medium text-gray-700 mb-3">Monthly Revenue Projection</h4>
                <div class="space-y-3">
                    @foreach($margins['networks'] as $network_code => $network)
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-gray-600">{{ $network['name'] }}</span>
                        <div class="text-right">
                            <div class="text-sm font-medium text-gray-900">₦{{ number_format(rand(50000, 200000), 0) }}</div>
                            <div class="text-xs text-gray-500">{{ $network['airtime_margin'] }}% avg</div>
                        </div>
                    </div>
                    @endforeach
                    <div class="border-t pt-2 flex justify-between items-center font-semibold">
                        <span class="text-sm text-gray-900">Total Projected</span>
                        <span class="text-sm text-gray-900">₦{{ number_format(rand(500000, 1000000), 0) }}</span>
                    </div>
                </div>
            </div>

            <!-- Competitive Analysis -->
            <div>
                <h4 class="text-sm font-medium text-gray-700 mb-3">Competitive Positioning</h4>
                <div class="space-y-3">
                    @foreach($margins['networks'] as $network_code => $network)
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-gray-600">{{ $network['name'] }}</span>
                        <div class="flex items-center">
                            @php $competitive_score = rand(70, 95); @endphp
                            <div class="w-16 h-2 bg-gray-200 rounded-full mr-2">
                                <div class="h-2 bg-gradient-to-r from-green-500 to-emerald-600 rounded-full" 
                                     style="width: {{ $competitive_score }}%"></div>
                            </div>
                            <span class="text-xs font-medium text-gray-900">{{ $competitive_score }}%</span>
                        </div>
                    </div>
                    @endforeach
                </div>
            </div>

            <!-- Optimization Suggestions -->
            <div>
                <h4 class="text-sm font-medium text-gray-700 mb-3">Optimization Suggestions</h4>
                <div class="space-y-3">
                    <div class="p-3 bg-blue-50 rounded-lg">
                        <div class="flex items-start">
                            <svg class="w-4 h-4 text-blue-600 mt-0.5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <div>
                                <p class="text-xs font-medium text-blue-800">MTN Data Margin</p>
                                <p class="text-xs text-blue-700">Consider increasing to 3.2% for better profitability</p>
                            </div>
                        </div>
                    </div>
                    <div class="p-3 bg-green-50 rounded-lg">
                        <div class="flex items-start">
                            <svg class="w-4 h-4 text-green-600 mt-0.5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <div>
                                <p class="text-xs font-medium text-green-800">Airtel Performance</p>
                                <p class="text-xs text-green-700">Current margins are optimal for market competition</p>
                            </div>
                        </div>
                    </div>
                    <div class="p-3 bg-yellow-50 rounded-lg">
                        <div class="flex items-start">
                            <svg class="w-4 h-4 text-yellow-600 mt-0.5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                            </svg>
                            <div>
                                <p class="text-xs font-medium text-yellow-800">9mobile Strategy</p>
                                <p class="text-xs text-yellow-700">Low margin may impact customer retention</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function calculateProfit(networkCode, type) {
    const marginInput = document.getElementById(`${type}_margin_${networkCode}`);
    const margin = parseFloat(marginInput.value);
    
    // Update example calculations
    // Implementation would update the UI to show new profit calculations
    console.log(`Updated ${type} margin for ${networkCode}: ${margin}%`);
}

function setMargins(networkCode, airtimeMargin, dataMargin) {
    document.getElementById(`airtime_margin_${networkCode}`).value = airtimeMargin;
    document.getElementById(`data_margin_${networkCode}`).value = dataMargin;
    
    // Trigger recalculation
    calculateProfit(networkCode, 'airtime');
    calculateProfit(networkCode, 'data');
}

function toggleNetworkStatus(networkCode) {
    if(confirm('Are you sure you want to toggle this network status?')) {
        // Send toggle request to backend
        fetch(`/admin/airtime/networks/${networkCode}/toggle`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('success', `Network status toggled for ${networkCode.toUpperCase()}`);
                setTimeout(() => location.reload(), 1000);
            } else {
                showAlert('error', data.message || 'Failed to toggle network status');
            }
        })
        .catch(error => {
            showAlert('error', 'Error: ' + error.message);
        });
    }
}

function resetToDefaults() {
    if(confirm('Are you sure you want to reset all margins to default values? This action cannot be undone.')) {
        // Reset all input fields to default values
        @foreach($margins['networks'] as $network_code => $network)
        document.getElementById('airtime_margin_{{ $network_code }}').value = '{{ $margins["settings"]["default_airtime_margin"] }}';
        document.getElementById('data_margin_{{ $network_code }}').value = '{{ $margins["settings"]["default_data_margin"] }}';
        @endforeach
        
        showAlert('info', 'All margins reset to default values. Click "Save All Changes" to apply.');
    }
}

function saveAllMargins() {
    // Collect all margin data
    const marginData = {};
    
    // Collect global settings
    marginData.settings = {
        default_airtime_margin: document.getElementById('default_airtime_margin').value,
        default_data_margin: document.getElementById('default_data_margin').value,
        minimum_margin: document.getElementById('minimum_margin').value,
        maximum_margin: document.getElementById('maximum_margin').value
    };
    
    // Collect network-specific margins
    marginData.networks = {};
    @foreach($margins['networks'] as $network_code => $network)
    marginData.networks['{{ $network_code }}'] = {
        airtime_margin: document.getElementById('airtime_margin_{{ $network_code }}').value,
        data_margin: document.getElementById('data_margin_{{ $network_code }}').value
    };
    @endforeach
    
    // Send to server
    fetch('/admin/airtime/margins', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify(marginData)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('success', 'Margin settings saved successfully!');
        } else {
            showAlert('error', data.message || 'Failed to save margin settings');
        }
    })
    .catch(error => {
        showAlert('error', 'Error: ' + error.message);
    });
}

function showAlert(type, message) {
    // Create alert element
    const alertDiv = document.createElement('div');
    alertDiv.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg max-w-sm ${getAlertClasses(type)}`;
    alertDiv.innerHTML = `
        <div class="flex items-center">
            <div class="flex-shrink-0">
                ${getAlertIcon(type)}
            </div>
            <div class="ml-3">
                <p class="text-sm font-medium">${message}</p>
            </div>
            <div class="ml-4 flex-shrink-0">
                <button onclick="this.parentElement.parentElement.parentElement.remove()" class="inline-flex text-gray-400 hover:text-gray-600">
                    <svg class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(alertDiv);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (alertDiv.parentElement) {
            alertDiv.remove();
        }
    }, 5000);
}

function getAlertClasses(type) {
    switch(type) {
        case 'success': return 'bg-green-50 border border-green-200 text-green-800';
        case 'error': return 'bg-red-50 border border-red-200 text-red-800';
        case 'warning': return 'bg-yellow-50 border border-yellow-200 text-yellow-800';
        case 'info': return 'bg-blue-50 border border-blue-200 text-blue-800';
        default: return 'bg-gray-50 border border-gray-200 text-gray-800';
    }
}

function getAlertIcon(type) {
    switch(type) {
        case 'success': return '<svg class="h-5 w-5 text-green-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
        case 'error': return '<svg class="h-5 w-5 text-red-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
        case 'warning': return '<svg class="h-5 w-5 text-yellow-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z"></path></svg>';
        case 'info': return '<svg class="h-5 w-5 text-blue-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
        default: return '<svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
    }
}
</script>
@endsection