<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AirtimeProvider;
use App\Models\AirtimeMargin;
use App\Models\AirtimeOrder;
use App\Models\ApiConfig;
use App\Models\AuditLog;
use App\Models\Setting;
use App\Models\Transaction;
use App\Services\Airtime\AirtimeProviderFactory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class AirtimeController extends Controller
{
    /**
     * Display airtime and data orders.
     */
    public function orders(Request $request)
    {
        try {
            // Get both transactions and dedicated airtime orders
            $transactionQuery = Transaction::with(['user'])
                ->whereIn('category', ['airtime', 'data', 'airtime_purchase', 'data_purchase']);

            $airtimeOrdersQuery = AirtimeOrder::with(['user', 'provider']);

            // Apply filters to both queries
            if ($request->filled('search')) {
                $search = $request->search;
                $transactionQuery->where(function($q) use ($search) {
                    $q->where('reference', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%")
                      ->orWhereHas('user', function($subq) use ($search) {
                          $subq->where('first_name', 'like', "%{$search}%")
                               ->orWhere('last_name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                      });
                });

                $airtimeOrdersQuery->where(function($q) use ($search) {
                    $q->where('reference', 'like', "%{$search}%")
                      ->orWhere('phone_number', 'like', "%{$search}%")
                      ->orWhereHas('user', function($subq) use ($search) {
                          $subq->where('first_name', 'like', "%{$search}%")
                               ->orWhere('last_name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                      });
                });
            }

            if ($request->filled('status')) {
                $transactionQuery->where('status', $request->status);
                $airtimeOrdersQuery->where('status', $request->status);
            }

            if ($request->filled('network')) {
                $airtimeOrdersQuery->where('network', $request->network);
                $transactionQuery->whereJsonContains('metadata->network', $request->network);
            }

            if ($request->filled('service_type')) {
                $airtimeOrdersQuery->where('service_type', $request->service_type);
                
                $categoryMap = ['airtime' => ['airtime', 'airtime_purchase'], 'data' => ['data', 'data_purchase']];
                if (isset($categoryMap[$request->service_type])) {
                    $transactionQuery->whereIn('category', $categoryMap[$request->service_type]);
                }
            }

            if ($request->filled('date_from')) {
                $transactionQuery->whereDate('created_at', '>=', $request->date_from);
                $airtimeOrdersQuery->whereDate('created_at', '>=', $request->date_from);
            }

            if ($request->filled('date_to')) {
                $transactionQuery->whereDate('created_at', '<=', $request->date_to);
                $airtimeOrdersQuery->whereDate('created_at', '<=', $request->date_to);
            }

            // Get paginated results from both sources
            $transactions = $transactionQuery->latest()->limit(10)->get();
            $airtimeOrders = $airtimeOrdersQuery->latest()->limit(10)->get();

            // Merge and sort by creation date
            $allOrders = collect()
                ->concat($transactions->map(function($txn) {
                    return (object)[
                        'id' => $txn->id,
                        'reference' => $txn->reference,
                        'user' => $txn->user,
                        'amount' => $txn->amount,
                        'fee' => $txn->fee ?? 0,
                        'status' => $txn->status,
                        'created_at' => $txn->created_at,
                        'type' => 'transaction',
                        'source' => $txn,
                        'network' => $txn->metadata['network'] ?? 'Unknown',
                        'phone_number' => $txn->metadata['phone'] ?? 'N/A',
                        'service_type' => str_contains($txn->category, 'data') ? 'data' : 'airtime',
                        'provider_name' => $txn->metadata['provider'] ?? 'Legacy',
                    ];
                }))
                ->concat($airtimeOrders->map(function($order) {
                    return (object)[
                        'id' => $order->id,
                        'reference' => $order->reference,
                        'user' => $order->user,
                        'amount' => $order->amount,
                        'fee' => 0, // Airtime orders don't have separate fees
                        'status' => $order->status,
                        'created_at' => $order->created_at,
                        'type' => 'airtime_order',
                        'source' => $order,
                        'network' => $order->network,
                        'phone_number' => $order->phone_number,
                        'service_type' => $order->service_type,
                        'provider_name' => $order->provider->name ?? 'Unknown',
                    ];
                }))
                ->sortByDesc('created_at')
                ->take(20);

            // Create a paginator for the merged results
            $currentPage = $request->get('page', 1);
            $perPage = 20;
            $orders = new \Illuminate\Pagination\LengthAwarePaginator(
                $allOrders->forPage($currentPage, $perPage)->values(),
                $allOrders->count(),
                $perPage,
                $currentPage,
                ['path' => $request->url(), 'pageName' => 'page']
            );
            $orders->withQueryString();

            // Calculate comprehensive stats
            $totalTransactions = Transaction::whereIn('category', ['airtime', 'data', 'airtime_purchase', 'data_purchase'])->count();
            $totalAirtimeOrders = AirtimeOrder::count();
            
            $stats = [
                'total_orders' => $totalTransactions + $totalAirtimeOrders,
                'airtime_orders' => Transaction::whereIn('category', ['airtime', 'airtime_purchase'])->count() + 
                                  AirtimeOrder::where('service_type', 'airtime')->count(),
                'data_orders' => Transaction::whereIn('category', ['data', 'data_purchase'])->count() + 
                               AirtimeOrder::where('service_type', 'data')->count(),
                'total_volume' => Transaction::whereIn('category', ['airtime', 'data', 'airtime_purchase', 'data_purchase'])->sum('amount') +
                                AirtimeOrder::sum('amount'),
                'success_rate' => $this->calculateSuccessRate(),
                'successful_orders' => Transaction::whereIn('category', ['airtime', 'data', 'airtime_purchase', 'data_purchase'])
                    ->where('status', 'completed')->count() + AirtimeOrder::where('status', 'completed')->count(),
                'failed_orders' => Transaction::whereIn('category', ['airtime', 'data', 'airtime_purchase', 'data_purchase'])
                    ->where('status', 'failed')->count() + AirtimeOrder::where('status', 'failed')->count(),
                'today_orders' => Transaction::whereIn('category', ['airtime', 'data', 'airtime_purchase', 'data_purchase'])
                    ->whereDate('created_at', today())->count() + AirtimeOrder::whereDate('created_at', today())->count(),
            ];

            return view('admin.airtime.orders', compact('orders', 'stats'));
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error loading airtime orders: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            // Create empty collection for display
            $orders = new \Illuminate\Pagination\LengthAwarePaginator(
                collect([]), 0, 20, 1, ['path' => request()->url()]
            );
            
            $stats = [
                'total_orders' => 0,
                'airtime_orders' => 0,
                'data_orders' => 0,
                'total_volume' => 0,
                'success_rate' => 0,
                'successful_orders' => 0,
                'failed_orders' => 0,
                'today_orders' => 0,
            ];

            return view('admin.airtime.orders', compact('orders', 'stats'))
                ->with('error', 'Unable to load transaction data. This may be due to missing database tables or columns. Please run database migrations.');
        }
    }

    /**
     * Display failed airtime and data transactions.
     */
    public function failed(Request $request)
    {
        try {
            // Handle both old and new category naming conventions
            $query = Transaction::with(['user'])
                              ->whereIn('category', ['airtime', 'data', 'airtime_purchase', 'data_purchase'])
                              ->where('status', 'failed');

            // Apply filters
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('reference', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%")
                      ->orWhereHas('user', function($subq) use ($search) {
                          $subq->where('first_name', 'like', "%{$search}%")
                               ->orWhere('last_name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                      });
                });
            }

            if ($request->filled('category')) {
                $query->where('category', $request->category);
            }

            if ($request->filled('provider')) {
                $query->whereJsonContains('metadata->provider', $request->provider);
            }

            if ($request->filled('network')) {
                $query->whereJsonContains('metadata->network', $request->network);
            }

            if ($request->filled('date_from')) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }

            if ($request->filled('date_to')) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            $failedTransactions = $query->latest()->paginate(20)->withQueryString();

            $stats = [
                'total_failed' => $failedTransactions->total(),
                'airtime_failed' => Transaction::whereIn('category', ['airtime', 'airtime_purchase'])->where('status', 'failed')->count(),
                'data_failed' => Transaction::whereIn('category', ['data', 'data_purchase'])->where('status', 'failed')->count(),
                'failed_volume' => $query->sum('amount'),
            ];

            return view('admin.airtime.failed', compact('failedTransactions', 'stats'));
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error loading failed transactions: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            // Create empty collection for display
            $failedTransactions = new \Illuminate\Pagination\LengthAwarePaginator(
                collect([]), 0, 20, 1, ['path' => request()->url()]
            );
            
            $stats = [
                'total_failed' => 0,
                'airtime_failed' => 0,
                'data_failed' => 0,
                'failed_volume' => 0,
            ];

            return view('admin.airtime.failed', compact('failedTransactions', 'stats'))
                ->with('error', 'Unable to load failed transaction data. Please run database migrations.');
        }
    }

    /**
     * Manage airtime/data providers.
     */
    public function providers()
    {
        try {
            // Use the correct scope method name and add error handling
            $providers = AirtimeProvider::orderedByPriority()->get();

            return view('admin.airtime.providers', compact('providers'));
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error loading airtime providers: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            // Fallback: try to get providers without priority ordering
            try {
                $providers = AirtimeProvider::all();
                return view('admin.airtime.providers', compact('providers'));
            } catch (\Exception $fallbackError) {
                // If table doesn't exist, show empty state
                $providers = collect([]);
                return view('admin.airtime.providers', compact('providers'))
                    ->with('error', 'Airtime providers table not found. Please run database migrations.');
            }
        }
    }

    /**
     * Show the form for creating a new airtime provider.
     */
    public function createProvider()
    {
        $supportedModules = [
            'vtpass' => 'VTPass',
            'buypower' => 'BuyPower',
            'clubkonnect' => 'ClubKonnect',
            'smartsmssolutions' => 'SmartSMS Solutions',
            'custom' => 'Custom Provider'
        ];

        $supportedServices = [
            'airtime' => 'Airtime',
            'data' => 'Data Bundles',
            'electricity' => 'Electricity Bills',
            'cable' => 'Cable TV',
            'internet' => 'Internet Bills'
        ];

        $supportedNetworks = [
            'mtn' => 'MTN',
            'glo' => 'Glo',
            'airtel' => 'Airtel',
            '9mobile' => '9mobile'
        ];

        return view('admin.airtime.providers.create', compact('supportedModules', 'supportedServices', 'supportedNetworks'));
    }

    /**
     * Show the form for editing an airtime provider.
     */
    public function editProvider(AirtimeProvider $provider)
    {
        $supportedModules = [
            'vtpass' => 'VTPass',
            'buypower' => 'BuyPower',
            'clubkonnect' => 'ClubKonnect',
            'smartsmssolutions' => 'SmartSMS Solutions',
            'custom' => 'Custom Provider'
        ];

        $supportedServices = [
            'airtime' => 'Airtime',
            'data' => 'Data Bundles',
            'electricity' => 'Electricity Bills',
            'cable' => 'Cable TV',
            'internet' => 'Internet Bills'
        ];

        $supportedNetworks = [
            'mtn' => 'MTN',
            'glo' => 'Glo',
            'airtel' => 'Airtel',
            '9mobile' => '9mobile'
        ];

        return view('admin.airtime.providers.edit', compact('provider', 'supportedModules', 'supportedServices', 'supportedNetworks'));
    }

    /**
     * Store a new airtime provider.
     */
    public function storeProvider(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'module' => 'required|in:vtpass,buypower,clubkonnect,smartsmssolutions,custom',
            'api_credentials' => 'required|array',
            'supported_services' => 'required|array',
            'supported_networks' => 'required|array',
            'priority' => 'required|integer|min:0|max:100',
            'is_test_mode' => 'boolean',
        ]);

        $provider = AirtimeProvider::create([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'module' => $request->module,
            'api_credentials' => $request->api_credentials,
            'supported_services' => $request->supported_services,
            'supported_networks' => $request->supported_networks,
            'priority' => $request->priority,
            'is_active' => true,
            'is_test_mode' => $request->boolean('is_test_mode'),
        ]);

        // Create audit log
        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'airtime_provider_created',
            'model' => 'AirtimeProvider',
            'model_id' => $provider->id,
            'description' => "Created new airtime provider: {$provider->name}",
            'ip_address' => request()->ip(),
        ]);

        return redirect()->route('admin.airtime.providers')->with('success', 'Airtime provider created successfully.');
    }

    /**
     * Update an airtime provider.
     */
    public function updateProvider(Request $request, AirtimeProvider $provider)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'module' => 'required|in:vtpass,buypower,clubkonnect,smartsmssolutions,custom',
                'api_credentials' => 'required|array',
                'supported_services' => 'required|array',
                'supported_networks' => 'required|array',
                'priority' => 'required|integer|min:0|max:100',
                'is_active' => 'boolean',
                'is_test_mode' => 'boolean',
            ]);

            $oldValues = $provider->toArray();

            $provider->update([
                'name' => $request->name,
                'slug' => Str::slug($request->name),
                'module' => $request->module,
                'api_credentials' => $request->api_credentials,
                'supported_services' => $request->supported_services,
                'supported_networks' => $request->supported_networks,
                'priority' => $request->priority,
                'is_active' => $request->boolean('is_active'),
                'is_test_mode' => $request->boolean('is_test_mode'),
            ]);

            // Create audit log
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'airtime_provider_updated',
                'model' => 'AirtimeProvider',
                'model_id' => $provider->id,
                'old_values' => $oldValues,
                'new_values' => $provider->toArray(),
                'description' => "Updated airtime provider: {$provider->name}",
                'ip_address' => request()->ip(),
            ]);

            if (request()->expectsJson()) {
                return response()->json(['success' => true, 'message' => 'Provider updated successfully']);
            }

            return redirect()->route('admin.airtime.providers')->with('success', 'Airtime provider updated successfully.');
        } catch (\Exception $e) {
            if (request()->expectsJson()) {
                return response()->json(['success' => false, 'message' => 'Failed to update provider'], 500);
            }
            
            return redirect()->route('admin.airtime.providers')->with('error', 'Failed to update provider');
        }
    }

    /**
     * Delete an airtime provider.
     */
    public function deleteProvider(AirtimeProvider $provider)
    {
        try {
            // Create audit log before deletion
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'airtime_provider_deleted',
                'model' => 'AirtimeProvider',
                'model_id' => $provider->id,
                'old_values' => $provider->toArray(),
                'description' => "Deleted airtime provider: {$provider->name}",
                'ip_address' => request()->ip(),
            ]);

            $provider->delete();

            if (request()->expectsJson()) {
                return response()->json(['success' => true, 'message' => 'Provider deleted successfully']);
            }

            return redirect()->route('admin.airtime.providers')->with('success', 'Airtime provider deleted successfully.');
        } catch (\Exception $e) {
            if (request()->expectsJson()) {
                return response()->json(['success' => false, 'message' => 'Failed to delete provider'], 500);
            }
            
            return redirect()->route('admin.airtime.providers')->with('error', 'Failed to delete provider');
        }
    }

    /**
     * Test provider connectivity.
     */
    public function testProvider(AirtimeProvider $provider)
    {
        try {
            $result = $provider->testConnectivity();
            
            if ($result['success']) {
                return response()->json([
                    'success' => true,
                    'response_time' => $result['response_time'],
                    'message' => $result['message']
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $result['message']
                ], 422);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Manage profit margins.
     */
    public function margins()
    {
        try {
            $margins = AirtimeMargin::getAllMarginsGrouped();
            
            // Get active providers for integration info
            $providers = AirtimeProvider::active()
                ->orderBy('priority', 'desc')
                ->get()
                ->groupBy(function($provider) {
                    // Group providers by supported networks
                    return collect($provider->supported_networks)->first();
                });

            // Ensure all networks have entries and integrate with provider data
            $networks = ['mtn', 'glo', 'airtel', '9mobile'];
            $serviceTypes = ['airtime', 'data'];
            
            $networkStats = [];
            
            foreach ($networks as $network) {
                // Get providers for this network
                $networkProviders = AirtimeProvider::active()
                    ->whereJsonContains('supported_networks', $network)
                    ->get();
                
                $networkStats[$network] = [
                    'providers_count' => $networkProviders->count(),
                    'active_providers' => $networkProviders->pluck('name')->implode(', '),
                    'avg_response_time' => $networkProviders->avg('avg_response_time') ?? 0,
                    'success_rate' => $networkProviders->avg('success_rate') ?? 100,
                ];
                
                foreach ($serviceTypes as $serviceType) {
                    if (!isset($margins[$network][$serviceType])) {
                        try {
                            // Create default margin if it doesn't exist
                            $defaultMargin = ($serviceType === 'airtime') ? 2.5 : 3.0;
                            $margins[$network][$serviceType] = AirtimeMargin::create([
                                'network' => $network,
                                'service_type' => $serviceType,
                                'margin_percentage' => $defaultMargin,
                                'fixed_fee' => 0.00,
                                'min_margin_amount' => 1.00,
                                'is_active' => true,
                            ]);
                        } catch (\Exception $createError) {
                            // If we can't create the margin, set a default array structure
                            $defaultMargin = ($serviceType === 'airtime') ? 2.5 : 3.0;
                            $margins[$network][$serviceType] = (object)[
                                'id' => null,
                                'network' => $network,
                                'service_type' => $serviceType,
                                'margin_percentage' => $defaultMargin,
                                'fixed_fee' => 0.00,
                                'min_margin_amount' => 1.00,
                                'is_active' => true,
                            ];
                        }
                    }
                }
            }

            return view('admin.airtime.margins', compact('margins', 'networkStats'));
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error loading airtime margins: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            // If table doesn't exist, create empty margins structure for display
            $margins = [];
            $networkStats = [];
            $networks = ['mtn', 'glo', 'airtel', '9mobile'];
            $serviceTypes = ['airtime', 'data'];
            
            foreach ($networks as $network) {
                $networkStats[$network] = [
                    'providers_count' => 0,
                    'active_providers' => 'No providers configured',
                    'avg_response_time' => 0,
                    'success_rate' => 0,
                ];
                
                foreach ($serviceTypes as $serviceType) {
                    $defaultMargin = ($serviceType === 'airtime') ? 2.5 : 3.0;
                    $margins[$network][$serviceType] = (object)[
                        'id' => null,
                        'network' => $network,
                        'service_type' => $serviceType,
                        'margin_percentage' => $defaultMargin,
                        'fixed_fee' => 0.00,
                        'min_margin_amount' => 1.00,
                        'is_active' => true,
                    ];
                }
            }

            return view('admin.airtime.margins', compact('margins', 'networkStats'))
                ->with('error', 'Airtime margins table not found. Please run database migrations to enable margin management.');
        }
    }

    /**
     * Update margins.
     */
    public function updateMargins(Request $request)
    {
        try {
            $request->validate([
                'margins' => 'required|array',
                'margins.*.network' => 'required|in:mtn,glo,airtel,9mobile',
                'margins.*.service_type' => 'required|in:airtime,data',
                'margins.*.margin_percentage' => 'required|numeric|min:0.5|max:50',
                'margins.*.fixed_fee' => 'nullable|numeric|min:0|max:1000',
                'margins.*.min_margin_amount' => 'nullable|numeric|min:0|max:1000',
                'margins.*.max_margin_amount' => 'nullable|numeric|min:0|max:10000',
                'margins.*.is_active' => 'nullable|boolean',
            ], [
                'margins.*.margin_percentage.min' => 'Margin percentage must be at least 0.5%',
                'margins.*.margin_percentage.max' => 'Margin percentage cannot exceed 50%',
                'margins.*.network.in' => 'Invalid network selected',
                'margins.*.service_type.in' => 'Invalid service type selected',
            ]);

            $updatedMargins = [];
            $totalUpdated = 0;

            DB::beginTransaction();

            foreach ($request->margins as $key => $marginData) {
                // Skip if required fields are missing
                if (!isset($marginData['network']) || !isset($marginData['service_type']) || !isset($marginData['margin_percentage'])) {
                    continue;
                }

                $margin = AirtimeMargin::updateOrCreate(
                    [
                        'network' => $marginData['network'],
                        'service_type' => $marginData['service_type'],
                    ],
                    [
                        'margin_percentage' => $marginData['margin_percentage'],
                        'fixed_fee' => $marginData['fixed_fee'] ?? 0.00,
                        'min_margin_amount' => $marginData['min_margin_amount'] ?? 1.00,
                        'max_margin_amount' => $marginData['max_margin_amount'] ?? null,
                        'is_active' => isset($marginData['is_active']) ? (bool)$marginData['is_active'] : true,
                    ]
                );

                $updatedMargins[] = $margin;
                $totalUpdated++;

                // Create audit log
                AuditLog::create([
                    'admin_id' => auth('admin')->id(),
                    'action' => 'airtime_margin_updated',
                    'model' => 'AirtimeMargin',
                    'model_id' => $margin->id,
                    'description' => "Updated {$margin->network} {$margin->service_type} margin to {$margin->margin_percentage}%",
                    'ip_address' => request()->ip(),
                ]);
            }

            DB::commit();

            // Prepare success message with details
            $message = "Successfully updated {$totalUpdated} margin(s).";
            if ($totalUpdated > 0) {
                $networks = collect($updatedMargins)->groupBy('network')->keys()->implode(', ');
                $message .= " Networks: " . strtoupper($networks);
            }

            return back()->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error updating airtime margins: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return back()->with('error', 'Failed to update margins. Please try again or contact support if the problem persists.');
        }
    }

    /**
     * Resend a failed transaction.
     */
    public function resendTransaction(Transaction $transaction)
    {
        if ($transaction->status !== 'failed') {
            return back()->withErrors(['error' => 'Only failed transactions can be resent.']);
        }

        // Reset transaction status to pending for retry
        $transaction->update([
            'status' => 'pending',
            'metadata' => array_merge(
                json_decode($transaction->metadata, true) ?? [],
                ['resent_at' => now(), 'resent_by' => auth('admin')->id()]
            ),
        ]);

        // Log admin action
        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'resend_transaction',
            'target_type' => 'Transaction',
            'target_id' => $transaction->id,
            'data' => ['category' => $transaction->category, 'amount' => $transaction->amount],
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', 'Transaction has been queued for retry.');
    }

    /**
     * Calculate overall success rate.
     */
    private function calculateSuccessRate(): float
    {
        $totalTransactions = Transaction::whereIn('category', ['airtime', 'data', 'airtime_purchase', 'data_purchase'])->count();
        $totalAirtimeOrders = AirtimeOrder::count();
        $totalOrders = $totalTransactions + $totalAirtimeOrders;

        if ($totalOrders === 0) {
            return 0.0;
        }

        $successfulTransactions = Transaction::whereIn('category', ['airtime', 'data', 'airtime_purchase', 'data_purchase'])
            ->where('status', 'completed')->count();
        $successfulAirtimeOrders = AirtimeOrder::where('status', 'completed')->count();
        $totalSuccessful = $successfulTransactions + $successfulAirtimeOrders;

        return round(($totalSuccessful / $totalOrders) * 100, 1);
    }
}