<?php

namespace App\Services;

use App\Models\ApiConfig;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Transaction;
use Exception;

class PaystackService
{
    protected $config;
    protected $apiConfig;

    public function __construct()
    {
        // Don't load config in constructor to avoid errors during application bootstrap
        // Config will be loaded when needed in other methods
    }

    /**
     * Get API configuration.
     */
    protected function getApiConfig(): ?ApiConfig
    {
        if (!$this->apiConfig) {
            $this->apiConfig = ApiConfig::where('slug', 'paystack')->where('is_active', true)->first();
            if ($this->apiConfig) {
                $this->config = $this->apiConfig->config ?? [];
            }
        }
        return $this->apiConfig;
    }

    /**
     * Ensure service is configured before API calls.
     */
    protected function ensureConfigured(): void
    {
        if (!$this->getApiConfig()) {
            throw new Exception('Paystack API configuration not found or not active');
        }
    }

    /**
     * Initialize payment transaction.
     */
    public function initializePayment(array $data): array
    {
        try {
            $this->ensureConfigured();
            
            $response = $this->makeApiCall('/transaction/initialize', [
                'amount' => $data['amount'] * 100, // Convert to kobo
                'email' => $data['email'],
                'reference' => $data['reference'],
                'currency' => $data['currency'] ?? 'NGN',
                'callback_url' => $data['callback_url'] ?? null,
                'metadata' => $data['metadata'] ?? [],
            ]);

            if ($response['status']) {
                return [
                    'success' => true,
                    'data' => [
                        'authorization_url' => $response['data']['authorization_url'],
                        'access_code' => $response['data']['access_code'],
                        'reference' => $response['data']['reference'],
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Payment initialization failed',
            ];
        } catch (Exception $e) {
            Log::error('Paystack payment initialization failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Payment initialization failed. Please try again.',
            ];
        }
    }

    /**
     * Verify payment transaction.
     */
    public function verifyPayment(string $reference): array
    {
        try {
            $this->ensureConfigured();
            
            $response = $this->makeApiCall("/transaction/verify/{$reference}", [], 'GET');

            if ($response['status'] && $response['data']['status'] === 'success') {
                return [
                    'success' => true,
                    'data' => [
                        'status' => 'success',
                        'amount' => $response['data']['amount'] / 100, // Convert from kobo
                        'currency' => $response['data']['currency'],
                        'reference' => $response['data']['reference'],
                        'customer' => $response['data']['customer'],
                        'authorization' => $response['data']['authorization'],
                        'fees' => $response['data']['fees'] / 100,
                        'paid_at' => $response['data']['paid_at'],
                    ],
                ];
            }

            return [
                'success' => false,
                'status' => $response['data']['status'] ?? 'failed',
                'message' => $response['message'] ?? 'Payment verification failed',
            ];
        } catch (Exception $e) {
            Log::error('Paystack payment verification failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Payment verification failed.',
            ];
        }
    }

    /**
     * Create transfer recipient.
     */
    public function createTransferRecipient(array $data): array
    {
        try {
            $response = $this->makeApiCall('/transferrecipient', [
                'type' => 'nuban',
                'name' => $data['name'],
                'account_number' => $data['account_number'],
                'bank_code' => $data['bank_code'],
                'currency' => $data['currency'] ?? 'NGN',
            ]);

            if ($response['status']) {
                return [
                    'success' => true,
                    'data' => [
                        'recipient_code' => $response['data']['recipient_code'],
                        'details' => $response['data'],
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Recipient creation failed',
            ];
        } catch (Exception $e) {
            Log::error('Paystack recipient creation failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Recipient creation failed.',
            ];
        }
    }

    /**
     * Initiate transfer.
     */
    public function initiateTransfer(array $data): array
    {
        try {
            $response = $this->makeApiCall('/transfer', [
                'source' => 'balance',
                'amount' => $data['amount'] * 100, // Convert to kobo
                'recipient' => $data['recipient_code'],
                'reason' => $data['reason'] ?? 'Transfer',
                'reference' => $data['reference'],
            ]);

            if ($response['status']) {
                return [
                    'success' => true,
                    'data' => [
                        'transfer_code' => $response['data']['transfer_code'],
                        'reference' => $response['data']['reference'],
                        'status' => $response['data']['status'],
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Transfer initiation failed',
            ];
        } catch (Exception $e) {
            Log::error('Paystack transfer initiation failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Transfer initiation failed.',
            ];
        }
    }

    /**
     * Get list of banks.
     */
    public function getBanks(): array
    {
        try {
            $response = $this->makeApiCall('/bank', [], 'GET');

            if ($response['status']) {
                return [
                    'success' => true,
                    'data' => $response['data'],
                ];
            }

            return [
                'success' => false,
                'message' => 'Could not fetch banks',
            ];
        } catch (Exception $e) {
            Log::error('Paystack get banks failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Could not fetch banks.',
            ];
        }
    }

    /**
     * Resolve account number.
     */
    public function resolveAccount(string $accountNumber, string $bankCode): array
    {
        try {
            $response = $this->makeApiCall("/bank/resolve?account_number={$accountNumber}&bank_code={$bankCode}", [], 'GET');

            if ($response['status']) {
                return [
                    'success' => true,
                    'data' => [
                        'account_name' => $response['data']['account_name'],
                        'account_number' => $response['data']['account_number'],
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => 'Could not resolve account',
            ];
        } catch (Exception $e) {
            Log::error('Paystack account resolution failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Could not resolve account.',
            ];
        }
    }

    /**
     * Make API call to Paystack.
     */
    protected function makeApiCall(string $endpoint, array $data = [], string $method = 'POST'): array
    {
        if (config('app.env') === 'local' && !$this->getApiConfig()?->secret_key) {
            // Return mock response in local environment when no API key is configured
            Log::info("Paystack API Call (MOCK): {$method} {$endpoint}", $data);
            
            // Return mock successful response based on endpoint
            if (str_contains($endpoint, 'initialize')) {
                return [
                    'status' => true,
                    'data' => [
                        'authorization_url' => 'https://checkout.paystack.com/mock',
                        'access_code' => 'mock_access_code',
                        'reference' => $data['reference'] ?? 'mock_reference',
                    ]
                ];
            }
            
            return ['status' => true, 'data' => ['mock' => true]];
        }

        $url = rtrim($this->getApiConfig()->effective_base_url, '/') . $endpoint;
        $timeout = $this->config['timeout'] ?? 30;
        $retryAttempts = $this->config['retry_attempts'] ?? 3;

        $headers = [
            'Authorization' => 'Bearer ' . $this->getApiConfig()->secret_key,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ];

        for ($attempt = 1; $attempt <= $retryAttempts; $attempt++) {
            try {
                $request = Http::withHeaders($headers)->timeout($timeout);

                $response = match (strtoupper($method)) {
                    'GET' => $request->get($url, $data),
                    'POST' => $request->post($url, $data),
                    'PUT' => $request->put($url, $data),
                    'DELETE' => $request->delete($url, $data),
                    default => throw new Exception("Unsupported HTTP method: {$method}"),
                };

                if ($response->successful()) {
                    return $response->json();
                }

                if ($attempt === $retryAttempts) {
                    throw new Exception("API call failed with status: {$response->status()}, Body: {$response->body()}");
                }

                sleep($attempt); // Progressive delay

            } catch (Exception $e) {
                if ($attempt === $retryAttempts) {
                    throw $e;
                }
                sleep($attempt);
            }
        }

        throw new Exception('Max retry attempts exceeded');
    }

    /**
     * Verify webhook signature.
     */
    public function verifyWebhookSignature(string $payload, string $signature): bool
    {
        $secretKey = $this->getApiConfig()?->secret_key;
        if (!$secretKey) {
            return false;
        }
        
        $expectedSignature = hash_hmac('sha512', $payload, $secretKey);
        
        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Check if service is configured and active.
     */
    public function isConfigured(): bool
    {
        $config = $this->getApiConfig();
        return $config && 
               $config->is_active && 
               !empty($config->secret_key);
    }

    /**
     * Get service configuration.
     */
    public function getConfig(): ?ApiConfig
    {
        return $this->getApiConfig();
    }

    /**
     * Test API connection.
     */
    public function testConnection(): array
    {
        try {
            return $this->getBanks();
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage(),
            ];
        }
    }
}