<?php

namespace App\Services;

use App\Models\ApiConfig;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class TermiiService
{
    protected $config;
    protected $apiConfig;

    public function __construct()
    {
        // Don't load config in constructor to avoid errors during application bootstrap
        // Config will be loaded when needed in other methods
    }

    /**
     * Get API configuration.
     */
    protected function getApiConfig(): ?ApiConfig
    {
        if (!$this->apiConfig) {
            $this->apiConfig = ApiConfig::where('slug', 'termii')->where('is_active', true)->first();
            if ($this->apiConfig) {
                $this->config = $this->apiConfig->config ?? [];
            }
        }
        return $this->apiConfig;
    }

    /**
     * Ensure service is configured before API calls.
     */
    protected function ensureConfigured(): void
    {
        if (!$this->getApiConfig()) {
            throw new Exception('Termii API configuration not found or not active');
        }
    }

    /**
     * Send SMS message.
     */
    public function sendSms(string $to, string $message, array $options = []): array
    {
        try {
            $this->ensureConfigured();
            
            $formattedPhone = $this->formatPhoneNumber($to);
            
            $payload = [
                'to' => $formattedPhone,
                'from' => $options['from'] ?? config('app.name'),
                'sms' => $message,
                'type' => $options['type'] ?? 'plain',
                'channel' => $options['channel'] ?? 'generic',
                'api_key' => $this->getApiConfig()->api_key,
            ];

            // Log the SMS request details
            Log::info('Termii SMS Request Initiated', [
                'service' => 'TermiiService',
                'method' => 'sendSms',
                'original_phone' => $to,
                'formatted_phone' => $formattedPhone,
                'message_length' => strlen($message),
                'payload' => array_merge($payload, ['api_key' => '[MASKED]']), // Mask API key in logs
                'options' => $options,
                'timestamp' => now(),
            ]);

            $response = $this->makeApiCall('/api/sms/send', $payload, 'POST');

            // Log the full API response
            Log::info('Termii SMS Response Received', [
                'service' => 'TermiiService',
                'method' => 'sendSms',
                'original_phone' => $to,
                'formatted_phone' => $formattedPhone,
                'response' => $response,
                'success' => isset($response['message_id']),
                'timestamp' => now(),
            ]);

            if (isset($response['message_id'])) {
                Log::info('Termii SMS Sent Successfully', [
                    'service' => 'TermiiService',
                    'original_phone' => $to,
                    'formatted_phone' => $formattedPhone,
                    'message_id' => $response['message_id'],
                    'balance' => $response['balance'] ?? null,
                    'timestamp' => now(),
                ]);

                return [
                    'success' => true,
                    'data' => [
                        'message_id' => $response['message_id'],
                        'message' => $response['message'] ?? 'SMS sent successfully',
                        'balance' => $response['balance'] ?? null,
                        'user' => $response['user'] ?? null,
                    ],
                ];
            }

            Log::warning('Termii SMS Send Failed', [
                'service' => 'TermiiService',
                'original_phone' => $to,
                'formatted_phone' => $formattedPhone,
                'error_message' => $response['message'] ?? 'SMS sending failed',
                'error_code' => $response['code'] ?? null,
                'full_response' => $response,
                'timestamp' => now(),
            ]);

            return [
                'success' => false,
                'message' => $response['message'] ?? 'SMS sending failed',
                'error_code' => $response['code'] ?? null,
            ];

        } catch (Exception $e) {
            Log::error('Termii SMS Sending Exception', [
                'service' => 'TermiiService',
                'method' => 'sendSms',
                'original_phone' => $to,
                'formatted_phone' => $formattedPhone ?? null,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'timestamp' => now(),
            ]);

            return [
                'success' => false,
                'message' => 'SMS sending failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Send OTP.
     */
    public function sendOtp(string $to, array $options = []): array
    {
        try {
            $this->ensureConfigured();
            
            $formattedPhone = $this->formatPhoneNumber($to);
            
            $payload = [
                'api_key' => $this->getApiConfig()->api_key,
                'message_type' => $options['message_type'] ?? 'NUMERIC',
                'to' => $formattedPhone,
                'from' => $options['from'] ?? config('app.name'),
                'channel' => $options['channel'] ?? 'generic',
                'pin_attempts' => $options['pin_attempts'] ?? 3,
                'pin_time_to_live' => $options['pin_time_to_live'] ?? 5,
                'pin_length' => $options['pin_length'] ?? 6,
                'pin_placeholder' => $options['pin_placeholder'] ?? '< 1234 >',
                'message_text' => $options['message_text'] ?? 'Your verification code is < 1234 >. Valid for 5 minutes.',
            ];

            // Log the OTP request details
            Log::info('Termii OTP Request Initiated', [
                'service' => 'TermiiService',
                'method' => 'sendOtp',
                'original_phone' => $to,
                'formatted_phone' => $formattedPhone,
                'payload' => array_merge($payload, ['api_key' => '[MASKED]']), // Mask API key in logs
                'options' => $options,
                'timestamp' => now(),
            ]);

            $response = $this->makeApiCall('/api/sms/otp/send', $payload, 'POST');

            // Log the full API response
            Log::info('Termii OTP Response Received', [
                'service' => 'TermiiService',
                'method' => 'sendOtp',
                'original_phone' => $to,
                'formatted_phone' => $formattedPhone,
                'response' => $response,
                'success' => isset($response['pinId']),
                'timestamp' => now(),
            ]);

            if (isset($response['pinId'])) {
                Log::info('Termii OTP Sent Successfully', [
                    'service' => 'TermiiService',
                    'original_phone' => $to,
                    'formatted_phone' => $formattedPhone,
                    'pin_id' => $response['pinId'],
                    'sms_status' => $response['smsStatus'] ?? 'sent',
                    'timestamp' => now(),
                ]);

                return [
                    'success' => true,
                    'data' => [
                        'pin_id' => $response['pinId'],
                        'to' => $response['to'],
                        'sms_status' => $response['smsStatus'] ?? 'sent',
                        'message' => $response['message'] ?? 'OTP sent successfully',
                    ],
                ];
            }

            Log::warning('Termii OTP Send Failed', [
                'service' => 'TermiiService',
                'original_phone' => $to,
                'formatted_phone' => $formattedPhone,
                'error_message' => $response['message'] ?? 'OTP sending failed',
                'error_code' => $response['code'] ?? null,
                'full_response' => $response,
                'timestamp' => now(),
            ]);

            return [
                'success' => false,
                'message' => $response['message'] ?? 'OTP sending failed',
                'error_code' => $response['code'] ?? null,
            ];

        } catch (Exception $e) {
            Log::error('Termii OTP Sending Exception', [
                'service' => 'TermiiService',
                'method' => 'sendOtp',
                'original_phone' => $to,
                'formatted_phone' => $formattedPhone ?? null,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'timestamp' => now(),
            ]);

            return [
                'success' => false,
                'message' => 'OTP sending failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Verify OTP.
     */
    public function verifyOtp(string $pinId, string $pin): array
    {
        try {
            $this->ensureConfigured();
            
            $payload = [
                'api_key' => $this->getApiConfig()->api_key,
                'pin_id' => $pinId,
                'pin' => $pin,
            ];

            $response = $this->makeApiCall('/api/sms/otp/verify', $payload, 'POST');

            if (isset($response['pinId'])) {
                return [
                    'success' => true,
                    'data' => [
                        'pin_id' => $response['pinId'],
                        'verified' => $response['verified'] ?? false,
                        'message' => $response['message'] ?? 'OTP verified successfully',
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'OTP verification failed',
                'error_code' => $response['code'] ?? null,
            ];

        } catch (Exception $e) {
            Log::error('Termii OTP verification failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'OTP verification failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get account balance.
     */
    public function getBalance(): array
    {
        try {
            $this->ensureConfigured();
            
            $response = $this->makeApiCall('/api/get-balance', [
                'api_key' => $this->getApiConfig()->api_key,
            ], 'GET');

            if (isset($response['user'])) {
                return [
                    'success' => true,
                    'data' => [
                        'user' => $response['user'],
                        'balance' => $response['balance'] ?? 0,
                        'currency' => $response['currency'] ?? 'NGN',
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Failed to get balance',
            ];

        } catch (Exception $e) {
            Log::error('Termii get balance failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to get balance: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Send WhatsApp message.
     */
    public function sendWhatsApp(string $to, string $message, array $options = []): array
    {
        try {
            $this->ensureConfigured();
            
            $payload = [
                'api_key' => $this->getApiConfig()->api_key,
                'to' => $this->formatPhoneNumber($to),
                'from' => $options['from'] ?? config('app.name'),
                'sms' => $message,
                'type' => 'plain',
                'channel' => 'whatsapp',
            ];

            $response = $this->makeApiCall('/api/sms/send', $payload, 'POST');

            if (isset($response['message_id'])) {
                return [
                    'success' => true,
                    'data' => [
                        'message_id' => $response['message_id'],
                        'message' => $response['message'] ?? 'WhatsApp message sent successfully',
                        'balance' => $response['balance'] ?? null,
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'WhatsApp message sending failed',
                'error_code' => $response['code'] ?? null,
            ];

        } catch (Exception $e) {
            Log::error('Termii WhatsApp sending failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'WhatsApp message sending failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Send voice call.
     */
    public function sendVoiceCall(string $to, string $code, array $options = []): array
    {
        try {
            $this->ensureConfigured();
            
            $payload = [
                'api_key' => $this->getApiConfig()->api_key,
                'phone_number' => $this->formatPhoneNumber($to),
                'code' => $code,
            ];

            $response = $this->makeApiCall('/api/sms/otp/send/voice', $payload, 'POST');

            if (isset($response['pinId'])) {
                return [
                    'success' => true,
                    'data' => [
                        'pin_id' => $response['pinId'],
                        'message' => $response['message'] ?? 'Voice call initiated successfully',
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Voice call failed',
                'error_code' => $response['code'] ?? null,
            ];

        } catch (Exception $e) {
            Log::error('Termii voice call failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Voice call failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get inbox messages.
     */
    public function getInbox(): array
    {
        try {
            $this->ensureConfigured();
            $response = $this->makeApiCall('/api/sms/inbox', [
                'api_key' => $this->getApiConfig()->api_key,
            ], 'GET');

            if (isset($response['data'])) {
                return [
                    'success' => true,
                    'data' => $response['data'],
                    'links' => $response['links'] ?? [],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Failed to get inbox',
            ];

        } catch (Exception $e) {
            Log::error('Termii get inbox failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to get inbox: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Search phone number.
     */
    public function searchPhoneNumber(string $phoneNumber): array
    {
        try {
            $this->ensureConfigured();
            $response = $this->makeApiCall('/api/insight/number/query', [
                'api_key' => $this->getApiConfig()->api_key,
                'phone_number' => $this->formatPhoneNumber($phoneNumber),
            ], 'GET');

            if (isset($response['result'])) {
                return [
                    'success' => true,
                    'data' => $response['result'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Phone number search failed',
            ];

        } catch (Exception $e) {
            Log::error('Termii phone number search failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Phone number search failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get message history.
     */
    public function getMessageHistory(): array
    {
        $this->ensureConfigured();
        try {
            $response = $this->makeApiCall('/api/sms/inbox', [
                'api_key' => $this->getApiConfig()->api_key,
            ], 'GET');

            if (isset($response['data'])) {
                return [
                    'success' => true,
                    'data' => $response['data'],
                    'pagination' => [
                        'current_page' => $response['current_page'] ?? 1,
                        'last_page' => $response['last_page'] ?? 1,
                        'total' => $response['total'] ?? 0,
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Failed to get message history',
            ];

        } catch (Exception $e) {
            Log::error('Termii get message history failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to get message history: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Format phone number for Termii API.
     */
    protected function formatPhoneNumber(string $phoneNumber): string
    {
        // Remove any non-numeric characters
        $phoneNumber = preg_replace('/[^0-9]/', '', $phoneNumber);
        
        // Add country code if not present
        if (!str_starts_with($phoneNumber, '234') && str_starts_with($phoneNumber, '0')) {
            $phoneNumber = '234' . substr($phoneNumber, 1);
        } elseif (!str_starts_with($phoneNumber, '234') && !str_starts_with($phoneNumber, '0')) {
            $phoneNumber = '234' . $phoneNumber;
        }

        return $phoneNumber;
    }

    /**
     * Make API call to Termii.
     */
    protected function makeApiCall(string $endpoint, array $data = [], string $method = 'GET'): array
    {
        $url = rtrim($this->getApiConfig()->effective_base_url, '/') . $endpoint;
        $timeout = $this->config['timeout'] ?? 30;
        $retryAttempts = $this->config['retry_attempts'] ?? 3;

        $headers = [
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ];

        // Log API call attempt
        Log::info('Termii API Call Initiated', [
            'service' => 'TermiiService',
            'method' => $method,
            'endpoint' => $endpoint,
            'url' => $url,
            'data' => array_key_exists('api_key', $data) ? array_merge($data, ['api_key' => '[MASKED]']) : $data,
            'headers' => $headers,
            'timeout' => $timeout,
            'retry_attempts' => $retryAttempts,
            'timestamp' => now(),
        ]);

        for ($attempt = 1; $attempt <= $retryAttempts; $attempt++) {
            try {
                Log::info('Termii API Call Attempt', [
                    'service' => 'TermiiService',
                    'attempt' => $attempt,
                    'max_attempts' => $retryAttempts,
                    'endpoint' => $endpoint,
                    'timestamp' => now(),
                ]);

                $request = Http::withHeaders($headers)->timeout($timeout);

                $response = match (strtoupper($method)) {
                    'GET' => $request->get($url, $data),
                    'POST' => $request->post($url, $data),
                    'PUT' => $request->put($url, $data),
                    'DELETE' => $request->delete($url, $data),
                    default => throw new Exception("Unsupported HTTP method: {$method}"),
                };

                $responseData = $response->json();

                Log::info('Termii API Call Response', [
                    'service' => 'TermiiService',
                    'attempt' => $attempt,
                    'endpoint' => $endpoint,
                    'status_code' => $response->status(),
                    'response_body' => $responseData,
                    'response_headers' => $response->headers(),
                    'success' => $response->successful(),
                    'timestamp' => now(),
                ]);

                if ($response->successful()) {
                    return $responseData;
                }

                Log::warning('Termii API Call Failed', [
                    'service' => 'TermiiService',
                    'attempt' => $attempt,
                    'endpoint' => $endpoint,
                    'status_code' => $response->status(),
                    'response_body' => $response->body(),
                    'will_retry' => $attempt < $retryAttempts,
                    'timestamp' => now(),
                ]);

                if ($attempt === $retryAttempts) {
                    throw new Exception("API call failed with status: {$response->status()}, Body: {$response->body()}");
                }

                sleep($attempt); // Progressive delay

            } catch (Exception $e) {
                Log::error('Termii API Call Exception', [
                    'service' => 'TermiiService',
                    'attempt' => $attempt,
                    'endpoint' => $endpoint,
                    'error' => $e->getMessage(),
                    'will_retry' => $attempt < $retryAttempts,
                    'timestamp' => now(),
                ]);

                if ($attempt === $retryAttempts) {
                    throw $e;
                }
                sleep($attempt);
            }
        }

        throw new Exception('Max retry attempts exceeded');
    }

    /**
     * Check if service is configured and active.
     */
    public function isConfigured(): bool
    {
        $config = $this->getApiConfig();
        return $config && 
               $config->is_active && 
               !empty($config->api_key);
    }

    /**
     * Get service configuration.
     */
    public function getConfig(): ?ApiConfig
    {
        return $this->getApiConfig();
    }

    /**
     * Test API connection.
     */
    public function testConnection(): array
    {
        try {
            return $this->getBalance();
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage(),
            ];
        }
    }
}