@extends('layouts.responsive')

@section('title', 'Virtual Account')

@section('content')
<div class="space-y-8">
    <!-- Header Section -->
    <div class="text-center">
        <div class="w-20 h-20 bg-gradient-to-br from-indigo-500 to-indigo-700 rounded-2xl flex items-center justify-center mx-auto mb-6">
            <i class="fas fa-university text-white text-3xl"></i>
        </div>
        <h2 class="text-3xl font-bold text-white mb-4">Virtual Account</h2>
        <p class="text-gray-300 max-w-md mx-auto">
            Get a dedicated account number to receive payments and fund your wallet automatically
        </p>
    </div>

    <!-- KYC Required Notice -->
    @if(!auth()->user()->hasCompletedKyc())
    <div class="bg-gradient-to-r from-orange-500/20 to-yellow-500/20 border border-orange-500/30 rounded-2xl p-6">
        <div class="flex items-center">
            <div class="w-12 h-12 bg-orange-500/20 rounded-full flex items-center justify-center mr-4">
                <i class="fas fa-exclamation-triangle text-orange-400 text-xl"></i>
            </div>
            <div class="flex-1">
                <h4 class="text-orange-400 font-bold text-lg mb-2">KYC Verification Required</h4>
                <p class="text-gray-300 text-sm mb-4">Complete your identity verification to unlock virtual account features and receive payments.</p>
                <a href="{{ route('kyc.index') }}" class="inline-flex items-center px-4 py-2 bg-orange-500 hover:bg-orange-600 text-white font-semibold rounded-xl transition-colors duration-200">
                    <i class="fas fa-shield-check mr-2"></i>Complete KYC
                </a>
            </div>
        </div>
    </div>
    @endif

    <!-- Virtual Account Display -->
    @if($virtualAccount ?? false)
    <div class="bg-dark-800/50 backdrop-blur-sm border border-dark-700/50 rounded-2xl p-6">
        <div class="flex justify-between items-start mb-6">
            <h3 class="text-xl font-bold text-white">Your Virtual Account</h3>
            <div class="px-3 py-1 bg-green-500/20 text-green-400 border border-green-500/30 rounded-full text-xs font-semibold">
                <i class="fas fa-circle text-[8px] mr-1"></i>Active
            </div>
        </div>

        <div class="bg-gradient-to-br from-indigo-600 to-indigo-800 rounded-2xl p-6 text-white mb-6">
            <div class="flex justify-between items-start mb-4">
                <div>
                    <p class="text-indigo-200 text-sm font-medium">Account Number</p>
                    <p class="text-3xl font-mono font-bold tracking-wider">{{ $virtualAccount->account_number }}</p>
                </div>
                <button onclick="copyToClipboard('{{ $virtualAccount->account_number }}')" 
                        class="p-3 bg-white/15 rounded-xl hover:bg-white/25 transition-all duration-200 backdrop-blur-sm">
                    <i class="fas fa-copy text-lg"></i>
                </button>
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <p class="text-indigo-200 text-xs uppercase tracking-wide font-medium">Account Name</p>
                    <p class="text-white font-semibold">{{ $virtualAccount->account_name }}</p>
                </div>
                <div class="text-right">
                    <p class="text-indigo-200 text-xs uppercase tracking-wide font-medium">Bank</p>
                    <p class="text-white font-semibold">{{ $virtualAccount->bank_name }}</p>
                </div>
            </div>
        </div>

        <!-- Usage Instructions -->
        <div class="bg-blue-500/20 border border-blue-500/30 rounded-2xl p-6">
            <h4 class="text-blue-400 font-bold text-lg mb-4">How to Use</h4>
            <div class="space-y-3">
                <div class="flex items-start">
                    <div class="w-8 h-8 bg-blue-500/20 rounded-full flex items-center justify-center mr-3 flex-shrink-0 mt-0.5">
                        <span class="text-blue-400 font-bold text-sm">1</span>
                    </div>
                    <div>
                        <p class="text-gray-300 text-sm">Share your account number and bank name with anyone who wants to send you money</p>
                    </div>
                </div>
                <div class="flex items-start">
                    <div class="w-8 h-8 bg-blue-500/20 rounded-full flex items-center justify-center mr-3 flex-shrink-0 mt-0.5">
                        <span class="text-blue-400 font-bold text-sm">2</span>
                    </div>
                    <div>
                        <p class="text-gray-300 text-sm">When they transfer money, it will automatically credit your wallet within minutes</p>
                    </div>
                </div>
                <div class="flex items-start">
                    <div class="w-8 h-8 bg-blue-500/20 rounded-full flex items-center justify-center mr-3 flex-shrink-0 mt-0.5">
                        <span class="text-blue-400 font-bold text-sm">3</span>
                    </div>
                    <div>
                        <p class="text-gray-300 text-sm">You'll receive instant notifications for all incoming payments</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Transactions -->
        @if($recentTransactions ?? false)
        <div class="mt-8">
            <div class="flex justify-between items-center mb-4">
                <h4 class="text-lg font-bold text-white">Recent Credits</h4>
                <a href="{{ route('transactions.index', ['type' => 'credit', 'category' => 'virtual_account']) }}" 
                   class="text-primary-400 text-sm font-semibold hover:text-primary-300 transition-colors">View All</a>
            </div>
            
            <div class="space-y-3">
                @foreach($recentTransactions->take(5) as $transaction)
                <div class="flex items-center justify-between p-4 bg-dark-700/50 rounded-xl">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-green-500/20 text-green-400 rounded-full flex items-center justify-center mr-3">
                            <i class="fas fa-arrow-down"></i>
                        </div>
                        <div>
                            <p class="text-white font-medium text-sm">{{ $transaction->description }}</p>
                            <p class="text-gray-400 text-xs">{{ $transaction->created_at->format('M d, Y H:i') }}</p>
                        </div>
                    </div>
                    <div class="text-right">
                        <p class="text-green-400 font-bold">+₦{{ number_format($transaction->amount, 2) }}</p>
                    </div>
                </div>
                @endforeach
            </div>
        </div>
        @endif
    </div>
    @else
    <!-- Create Virtual Account -->
    <div class="bg-dark-800/50 backdrop-blur-sm border border-dark-700/50 rounded-2xl p-8">
        <div class="text-center">
            <div class="w-24 h-24 bg-indigo-500/20 rounded-2xl flex items-center justify-center mx-auto mb-6">
                <i class="fas fa-university text-indigo-400 text-4xl"></i>
            </div>
            <h3 class="text-2xl font-bold text-white mb-4">Create Your Virtual Account</h3>
            <p class="text-gray-300 mb-8 max-w-md mx-auto">
                Get a dedicated account number to receive payments from any Nigerian bank. All payments are automatically credited to your wallet.
            </p>
            
            @if(auth()->user()->hasCompletedKyc())
            <form method="POST" action="{{ route('virtual-account.create') }}" class="max-w-md mx-auto">
                @csrf
                <div class="mb-6">
                    <label for="account_name" class="form-label">
                        <i class="fas fa-user mr-2"></i>
                        Account Name
                    </label>
                    <input type="text" id="account_name" name="account_name" 
                           value="{{ old('account_name', auth()->user()->full_name) }}" 
                           class="form-input @error('account_name') border-red-500 @enderror" 
                           placeholder="Enter account name"
                           required>
                    @error('account_name')
                        <p class="text-red-400 text-sm mt-2 flex items-center">
                            <i class="fas fa-exclamation-circle mr-2"></i>{{ $message }}
                        </p>
                    @enderror
                </div>
                
                <button type="submit" class="w-full py-4 bg-indigo-600 hover:bg-indigo-700 text-white font-semibold rounded-2xl transition-colors duration-200 flex items-center justify-center">
                    <i class="fas fa-plus mr-2"></i>
                    <span id="submit-text">Create Virtual Account</span>
                </button>
            </form>
            @else
            <a href="{{ route('kyc.index') }}" class="inline-flex items-center px-6 py-4 bg-orange-600 hover:bg-orange-700 text-white font-semibold rounded-2xl transition-colors duration-200">
                <i class="fas fa-shield-check mr-2"></i>Complete KYC First
            </a>
            @endif
        </div>
    </div>
    @endif

    <!-- Benefits Section -->
    <div class="grid gap-6 md:grid-cols-3">
        <div class="bg-dark-800/50 backdrop-blur-sm border border-dark-700/50 rounded-2xl p-6">
            <div class="w-12 h-12 bg-green-500/20 rounded-xl flex items-center justify-center mb-4">
                <i class="fas fa-lightning-bolt text-green-400 text-xl"></i>
            </div>
            <h4 class="text-white font-bold text-lg mb-2">Instant Credits</h4>
            <p class="text-gray-400 text-sm">Receive payments instantly and get automatic wallet credit notifications.</p>
        </div>

        <div class="bg-dark-800/50 backdrop-blur-sm border border-dark-700/50 rounded-2xl p-6">
            <div class="w-12 h-12 bg-blue-500/20 rounded-xl flex items-center justify-center mb-4">
                <i class="fas fa-shield-alt text-blue-400 text-xl"></i>
            </div>
            <h4 class="text-white font-bold text-lg mb-2">100% Secure</h4>
            <p class="text-gray-400 text-sm">Bank-grade security with full transaction monitoring and fraud protection.</p>
        </div>

        <div class="bg-dark-800/50 backdrop-blur-sm border border-dark-700/50 rounded-2xl p-6">
            <div class="w-12 h-12 bg-purple-500/20 rounded-xl flex items-center justify-center mb-4">
                <i class="fas fa-coins text-purple-400 text-xl"></i>
            </div>
            <h4 class="text-white font-bold text-lg mb-2">No Hidden Fees</h4>
            <p class="text-gray-400 text-sm">Free to create and receive payments. No monthly charges or hidden costs.</p>
        </div>
    </div>
</div>

@push('scripts')
<script>
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(function() {
        showToast('Account number copied to clipboard!', 'success');
    }).catch(function() {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = text;
        textArea.style.position = 'fixed';
        textArea.style.opacity = '0';
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        showToast('Account number copied!', 'success');
    });
    
    // Add haptic feedback
    if (navigator.vibrate) {
        navigator.vibrate([50, 100, 50]);
    }
}

// Form submission handling
document.addEventListener('DOMContentLoaded', function() {
    const form = document.querySelector('form[action*="virtual-account"]');
    if (form) {
        form.addEventListener('submit', function(e) {
            const submitButton = this.querySelector('button[type="submit"]');
            const submitText = document.getElementById('submit-text');
            
            if (submitButton && submitText) {
                submitButton.disabled = true;
                submitText.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Creating...';
            }
        });
    }
});

function showToast(message, type = 'info') {
    const toast = document.createElement('div');
    const iconClass = type === 'success' ? 'fa-check-circle' : type === 'error' ? 'fa-exclamation-circle' : 'fa-info-circle';
    const bgClass = type === 'success' ? 'bg-green-600' : type === 'error' ? 'bg-red-600' : 'bg-blue-600';
    
    toast.className = `fixed top-4 right-4 p-4 rounded-2xl text-white z-50 ${bgClass} flex items-center shadow-2xl max-w-sm`;
    toast.innerHTML = `
        <i class="fas ${iconClass} mr-3 text-lg"></i>
        <span class="font-medium">${message}</span>
    `;
    
    document.body.appendChild(toast);
    
    // Animate in
    toast.style.transform = 'translateX(100%)';
    setTimeout(() => {
        toast.style.transform = 'translateX(0)';
        toast.style.transition = 'transform 0.3s ease-out';
    }, 10);
    
    // Remove after delay
    setTimeout(() => {
        toast.style.transform = 'translateX(100%)';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}
</script>
@endpush
@endsection