<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\VirtualAccount;
use App\Services\MonnifyService;

/**
 * Virtual Account Web Controller
 * 
 * TODO: Add comprehensive input validation
 * TODO: Add CSRF protection for all forms
 * TODO: Add user-friendly error messages
 * TODO: Add success/error flash messages
 */
class VirtualAccountController extends Controller
{
    protected $monnifyService;

    public function __construct(MonnifyService $monnifyService)
    {
        $this->monnifyService = $monnifyService;
    }

    /**
     * Display virtual accounts page
     */
    public function index(Request $request)
    {
        try {
            $user = $request->user();
            
            // Get user's virtual account (only one allowed per user)
            $virtualAccount = VirtualAccount::where('user_id', $user->id)
                ->where('is_active', true)
                ->first();

            $recentTransactions = null;
            
            if ($virtualAccount) {
                // Get recent transactions related to virtual account funding
                $recentTransactions = $user->transactions()
                    ->where('type', 'credit')
                    ->where('category', 'wallet_funding')
                    ->where('description', 'like', '%virtual account%')
                    ->latest()
                    ->limit(5)
                    ->get();
            }

            return view('virtual-account.index', compact('virtualAccount', 'recentTransactions'));

        } catch (\Exception $e) {
            \Log::error('Failed to load virtual accounts page', [
                'user_id' => $request->user()?->id,
                'error' => $e->getMessage()
            ]);
            
            return back()->with('error', 'Failed to load virtual accounts');
        }
    }

    /**
     * Handle virtual account creation (form submission)
     */
    public function create(Request $request)
    {
        // This method handles the POST request to create a virtual account
        return $this->store($request);
    }

    /**
     * Store new virtual account
     * Enhanced with KYC validation and better error handling
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'account_name' => 'required|string|max:100|regex:/^[a-zA-Z\s]+$/',
            ]);

            if ($validator->fails()) {
                return back()
                    ->withErrors($validator)
                    ->withInput();
            }

            $user = $request->user();

            // Gate virtual account creation to KYC-approved users
            if (!$user->hasCompletedKyc()) {
                return back()->with('error', 'Please complete your KYC verification first to create a virtual account.');
            }

            // Check if user already has an active virtual account
            $existingAccount = VirtualAccount::where('user_id', $user->id)
                ->where('is_active', true)
                ->first();

            if ($existingAccount) {
                return back()->with('error', 'You already have an active virtual account. Only one virtual account per user is allowed.');
            }

            // Create virtual account through service
            $accountData = [
                'account_name' => $request->account_name ?: $user->full_name,
                'user_id' => $user->id,
                'email' => $user->email,
                'phone' => $user->phone,
                'currency' => 'NGN',
                // Instead of passing the raw BVN, pass the KYC verification record ID or a verification flag
                'bvn_verified' => $user->kycVerifications()->where('type', 'bvn')->where('status', 'verified')->exists(),
                'bvn_kyc_id' => $user->kycVerifications()->where('type', 'bvn')->where('status', 'verified')->first()?->id ?? null,
            ];

            $response = $this->monnifyService->createVirtualAccount($accountData);

            if (!$response['success']) {
                return back()
                    ->with('error', 'Failed to create virtual account: ' . ($response['message'] ?? 'Unknown error'))
                    ->withInput();
            }

            // Save account to database
            $virtualAccount = VirtualAccount::create([
                'user_id' => $user->id,
                'account_number' => $response['data']['account_number'],
                'account_name' => $response['data']['account_name'],
                'bank_code' => $response['data']['bank_code'] ?? '000',
                'bank_name' => $response['data']['bank_name'] ?? 'Monnify Bank',
                'provider' => 'monnify',
                'provider_reference' => $response['data']['account_reference'] ?? $response['data']['reference'],
                'currency' => 'NGN',
                'is_active' => true,
                'metadata' => [
                    'created_via' => 'web',
                    'creation_ip' => $request->ip(),
                    'kyc_verified_at' => now(),
                ],
            ]);

            return redirect()
                ->route('virtual-account.index')
                ->with('success', 'Virtual account created successfully! You can now receive payments using your new account number.');

        } catch (\Exception $e) {
            \Log::error('Virtual account creation failed', [
                'user_id' => $request->user()?->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->with('error', 'Failed to create virtual account. Please try again or contact support if the problem persists.')
                ->withInput();
        }
    }

    /**
     * Show virtual account details
     * 
     * TODO: Add transaction history
     * TODO: Add account analytics
     */
    public function show(Request $request, $accountId)
    {
        try {
            $user = $request->user();
            
            $account = VirtualAccount::where('id', $accountId)
                ->where('user_id', $user->id)
                ->with(['bank'])
                ->first();

            if (!$account) {
                return redirect()
                    ->route('virtual-account.index')
                    ->with('error', 'Virtual account not found');
            }

            // TODO: Get transaction history for this account
            $transactions = collect(); // Placeholder

            return view('virtual-accounts.show', compact('account', 'transactions'));

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to load account details');
        }
    }

    /**
     * Show edit form
     * 
     * TODO: Add editable account properties
     */
    public function edit(Request $request, $accountId)
    {
        try {
            $user = $request->user();
            
            $account = VirtualAccount::where('id', $accountId)
                ->where('user_id', $user->id)
                ->first();

            if (!$account) {
                return redirect()
                    ->route('virtual-account.index')
                    ->with('error', 'Virtual account not found');
            }

            return view('virtual-accounts.edit', compact('account'));

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to load edit form');
        }
    }

    /**
     * Update virtual account
     * 
     * TODO: Add updateable fields validation
     */
    public function update(Request $request, $accountId)
    {
        try {
            $validator = Validator::make($request->all(), [
                'account_name' => 'required|string|max:100',
                'status' => 'required|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                return back()
                    ->withErrors($validator)
                    ->withInput();
            }

            $user = $request->user();
            
            $account = VirtualAccount::where('id', $accountId)
                ->where('user_id', $user->id)
                ->first();

            if (!$account) {
                return redirect()
                    ->route('virtual-account.index')
                    ->with('error', 'Virtual account not found');
            }

            $account->update([
                'account_name' => $request->account_name,
                'status' => $request->status,
            ]);

            // TODO: Update account with provider if needed
            // TODO: Log account update

            return redirect()
                ->route('virtual-account.show', $account->id)
                ->with('success', 'Virtual account updated successfully');

        } catch (\Exception $e) {
            return back()
                ->with('error', 'Failed to update virtual account')
                ->withInput();
        }
    }

    /**
     * Deactivate virtual account
     * 
     * TODO: Add confirmation modal
     * TODO: Add balance check before deactivation
     */
    public function destroy(Request $request, $accountId)
    {
        try {
            $user = $request->user();
            
            $account = VirtualAccount::where('id', $accountId)
                ->where('user_id', $user->id)
                ->first();

            if (!$account) {
                return redirect()
                    ->route('virtual-account.index')
                    ->with('error', 'Virtual account not found');
            }

            // TODO: Check for pending transactions
            // TODO: Deactivate with provider

            $account->update(['status' => 'inactive']);

            // TODO: Log account deactivation

            return redirect()
                ->route('virtual-account.index')
                ->with('success', 'Virtual account deactivated successfully');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to deactivate virtual account');
        }
    }
}