<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Service extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'category',
        'provider_code',
        'provider_name',
        'available_amounts',
        'minimum_amount',
        'maximum_amount',
        'commission_percentage',
        'commission_cap',
        'is_active',
        'metadata',
    ];

    protected function casts(): array
    {
        return [
            'available_amounts' => 'array',
            'minimum_amount' => 'decimal:2',
            'maximum_amount' => 'decimal:2',
            'commission_percentage' => 'decimal:2',
            'commission_cap' => 'decimal:2',
            'is_active' => 'boolean',
            'metadata' => 'array',
        ];
    }

    /**
     * Scope a query to only include active services.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to filter by category.
     */
    public function scopeCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Calculate commission for given amount.
     */
    public function calculateCommission(float $amount): float
    {
        $commission = ($amount * $this->commission_percentage) / 100;
        
        if ($this->commission_cap && $commission > $this->commission_cap) {
            $commission = $this->commission_cap;
        }
        
        return $commission;
    }

    /**
     * Check if amount is within service limits.
     */
    public function isAmountValid(float $amount): bool
    {
        if ($this->minimum_amount && $amount < $this->minimum_amount) {
            return false;
        }
        
        if ($this->maximum_amount && $amount > $this->maximum_amount) {
            return false;
        }
        
        return true;
    }
}