<?php

namespace App\Services\Airtime;

use App\Models\AirtimeProvider;
use App\Services\Airtime\Contracts\AirtimeProviderInterface;
use App\Services\Airtime\Providers\VTPassProvider;
use App\Services\Airtime\Providers\ClubkonnectProvider;
use App\Services\Airtime\Providers\CustomProvider;

class AirtimeProviderFactory
{
    /**
     * Available provider classes.
     */
    protected array $providers = [
        'vtpass' => VTPassProvider::class,
        'clubkonnect' => ClubkonnectProvider::class,
        'buypower' => VTPassProvider::class, // Use VTPass as fallback
        'smartsmssolutions' => VTPassProvider::class, // Use VTPass as fallback
        'custom' => CustomProvider::class,
    ];

    /**
     * Create a provider instance from database configuration.
     */
    public function create(AirtimeProvider $providerModel): AirtimeProviderInterface
    {
        $module = $providerModel->module;
        
        if (!isset($this->providers[$module])) {
            throw new \InvalidArgumentException("Unsupported provider module: {$module}");
        }

        $providerClass = $this->providers[$module];
        
        if (!class_exists($providerClass)) {
            throw new \InvalidArgumentException("Provider class not found: {$providerClass}");
        }

        $provider = new $providerClass();
        
        if (!$provider instanceof AirtimeProviderInterface) {
            throw new \InvalidArgumentException("Provider must implement AirtimeProviderInterface");
        }

        // Configure the provider with credentials
        $credentials = array_merge(
            $providerModel->api_credentials,
            ['test_mode' => $providerModel->is_test_mode]
        );

        $provider->configure($credentials);

        return $provider;
    }

    /**
     * Get the best available provider for a service and network.
     */
    public function getBestProvider(string $serviceType, string $network): ?AirtimeProviderInterface
    {
        $providers = AirtimeProvider::active()
            ->byService($serviceType)
            ->byNetwork($network)
            ->orderedByPriority()
            ->get();

        foreach ($providers as $providerModel) {
            try {
                return $this->create($providerModel);
            } catch (\Exception $e) {
                \Log::warning('Failed to create provider instance', [
                    'provider' => $providerModel->name,
                    'error' => $e->getMessage(),
                ]);
                continue;
            }
        }

        return null;
    }

    /**
     * Get all active providers for a service.
     */
    public function getProvidersForService(string $serviceType): array
    {
        $providers = AirtimeProvider::active()
            ->byService($serviceType)
            ->orderedByPriority()
            ->get();

        $instances = [];
        
        foreach ($providers as $providerModel) {
            try {
                $instances[] = [
                    'model' => $providerModel,
                    'instance' => $this->create($providerModel),
                ];
            } catch (\Exception $e) {
                \Log::warning('Failed to create provider instance', [
                    'provider' => $providerModel->name,
                    'error' => $e->getMessage(),
                ]);
                continue;
            }
        }

        return $instances;
    }

    /**
     * Test all active providers.
     */
    public function testAllProviders(): array
    {
        $providers = AirtimeProvider::active()->get();
        $results = [];

        foreach ($providers as $providerModel) {
            try {
                $provider = $this->create($providerModel);
                $testResult = $provider->testConnection();
                
                $results[] = [
                    'provider' => $providerModel->name,
                    'module' => $providerModel->module,
                    'success' => $testResult['success'],
                    'message' => $testResult['message'],
                    'response_time' => $testResult['response_time'] ?? 0,
                ];

                // Update provider's last test result
                $providerModel->update(['last_test_result' => $testResult]);

            } catch (\Exception $e) {
                $results[] = [
                    'provider' => $providerModel->name,
                    'module' => $providerModel->module,
                    'success' => false,
                    'message' => 'Configuration error: ' . $e->getMessage(),
                    'response_time' => 0,
                ];
            }
        }

        return $results;
    }

    /**
     * Register a custom provider.
     */
    public function registerProvider(string $module, string $providerClass): void
    {
        if (!class_exists($providerClass)) {
            throw new \InvalidArgumentException("Provider class not found: {$providerClass}");
        }

        if (!is_subclass_of($providerClass, AirtimeProviderInterface::class)) {
            throw new \InvalidArgumentException("Provider must implement AirtimeProviderInterface");
        }

        $this->providers[$module] = $providerClass;
    }

    /**
     * Get list of available provider modules.
     */
    public function getAvailableModules(): array
    {
        return array_keys($this->providers);
    }
}