<?php

namespace App\Services\Airtime;

use App\Services\Airtime\Contracts\AirtimeProviderInterface;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

abstract class BaseAirtimeProvider implements AirtimeProviderInterface
{
    protected array $credentials = [];
    protected string $baseUrl;
    protected bool $isTestMode = true;
    protected int $timeout = 30;

    /**
     * Configure the provider with credentials.
     */
    public function configure(array $credentials): void
    {
        $this->credentials = $credentials;
        $this->baseUrl = $credentials['base_url'] ?? '';
        $this->isTestMode = $credentials['test_mode'] ?? true;
    }

    /**
     * Make HTTP request to provider API.
     */
    protected function makeRequest(string $method, string $endpoint, array $data = [], array $headers = []): array
    {
        $url = rtrim($this->baseUrl, '/') . '/' . ltrim($endpoint, '/');

        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders(array_merge($this->getDefaultHeaders(), $headers))
                ->when($method === 'GET', fn($http) => $http->get($url, $data))
                ->when($method === 'POST', fn($http) => $http->post($url, $data))
                ->when($method === 'PUT', fn($http) => $http->put($url, $data))
                ->when($method === 'DELETE', fn($http) => $http->delete($url, $data));

            $responseData = $response->json();

            Log::info('Airtime Provider API Request', [
                'provider' => static::class,
                'method' => $method,
                'url' => $url,
                'status' => $response->status(),
                'response_size' => strlen($response->body()),
            ]);

            return [
                'success' => $response->successful(),
                'status_code' => $response->status(),
                'data' => $responseData,
                'raw_response' => $response->body(),
                'headers' => $response->headers(),
            ];

        } catch (\Exception $e) {
            Log::error('Airtime Provider API Error', [
                'provider' => static::class,
                'method' => $method,
                'url' => $url,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return [
                'success' => false,
                'status_code' => 0,
                'data' => null,
                'error' => $e->getMessage(),
                'raw_response' => null,
                'headers' => [],
            ];
        }
    }

    /**
     * Get default headers for API requests.
     */
    protected function getDefaultHeaders(): array
    {
        return [
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'User-Agent' => 'FinTech-Platform/1.0',
        ];
    }

    /**
     * Format phone number to standard format.
     */
    protected function formatPhoneNumber(string $phoneNumber): string
    {
        // Remove any non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phoneNumber);

        // Convert to international format
        if (strlen($phone) === 11 && substr($phone, 0, 1) === '0') {
            return '234' . substr($phone, 1);
        } elseif (strlen($phone) === 10) {
            return '234' . $phone;
        } elseif (strlen($phone) === 13 && substr($phone, 0, 3) === '234') {
            return $phone;
        }

        return $phone;
    }

    /**
     * Detect network from phone number.
     */
    protected function detectNetwork(string $phoneNumber): ?string
    {
        $phone = $this->formatPhoneNumber($phoneNumber);
        $prefix = substr($phone, 3, 3); // Get first 3 digits after country code

        $networkPrefixes = [
            'mtn' => ['703', '706', '803', '806', '810', '813', '814', '816', '903', '906', '913', '916'],
            'glo' => ['705', '805', '807', '811', '815', '905', '915'],
            'airtel' => ['701', '708', '802', '808', '812', '901', '902', '904', '907', '912'],
            '9mobile' => ['704', '709', '804', '809', '817', '818', '908', '909'],
        ];

        foreach ($networkPrefixes as $network => $prefixes) {
            if (in_array($prefix, $prefixes)) {
                return $network;
            }
        }

        return null;
    }

    /**
     * Validate required parameters.
     */
    protected function validateParameters(array $params, array $required): void
    {
        $missing = array_diff($required, array_keys($params));
        if (!empty($missing)) {
            throw new \InvalidArgumentException('Missing required parameters: ' . implode(', ', $missing));
        }
    }

    /**
     * Generate a unique transaction reference.
     */
    protected function generateReference(string $prefix = 'TXN'): string
    {
        return $prefix . '_' . time() . '_' . strtoupper(substr(md5(uniqid()), 0, 8));
    }

    /**
     * Default implementation for supported networks.
     */
    public function getSupportedNetworks(): array
    {
        return ['mtn', 'glo', 'airtel', '9mobile'];
    }
}