<?php

if (!function_exists('setting')) {
    /**
     * Get a setting value with default fallback.
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    function setting(string $key, $default = null)
    {
        try {
            return \App\Models\Setting::get($key, $default);
        } catch (\Exception $e) {
            // Fallback to default if database not available
            return $default;
        }
    }
}

if (!function_exists('set_setting')) {
    /**
     * Set a setting value.
     *
     * @param string $key
     * @param mixed $value
     * @param string $type
     * @return bool
     */
    function set_setting(string $key, $value, string $type = 'string'): bool
    {
        try {
            return \App\Models\Setting::set($key, $value, $type);
        } catch (\Exception $e) {
            return false;
        }
    }
}

if (!function_exists('getActionColor')) {
    /**
     * Get color class for audit log action.
     *
     * @param string $action
     * @return string
     */
    function getActionColor(string $action): string
    {
        $colors = [
            'login' => 'bg-green-100 text-green-800',
            'logout' => 'bg-gray-100 text-gray-800',
            'create' => 'bg-blue-100 text-blue-800',
            'update' => 'bg-yellow-100 text-yellow-800',
            'delete' => 'bg-red-100 text-red-800',
            'approve' => 'bg-emerald-100 text-emerald-800',
            'reject' => 'bg-red-100 text-red-800',
            'send' => 'bg-purple-100 text-purple-800',
            'broadcast' => 'bg-indigo-100 text-indigo-800',
        ];
        
        foreach($colors as $key => $color) {
            if(str_contains($action, $key)) {
                return $color;
            }
        }
        
        return 'bg-gray-100 text-gray-800';
    }
}

if (!function_exists('getLocationFromIP')) {
    /**
     * Get location from IP address.
     *
     * @param string $ip
     * @return string
     */
    function getLocationFromIP(string $ip): string
    {
        // Mock location data - in a real implementation, use a GeoIP service like MaxMind or IPStack
        $locations = [
            '127.0.0.1' => 'Local',
            '192.168.1.1' => 'Local Network',
            '::1' => 'Local (IPv6)',
            '10.0.0.1' => 'Private Network',
        ];
        
        // Return known locations
        if (isset($locations[$ip])) {
            return $locations[$ip];
        }
        
        // For unknown IPs, generate mock location based on IP patterns
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
            // Public IP - generate mock location
            $cities = ['Lagos, Nigeria', 'Abuja, Nigeria', 'Kano, Nigeria', 'Port Harcourt, Nigeria', 'Ibadan, Nigeria'];
            $hash = crc32($ip);
            return $cities[abs($hash) % count($cities)];
        }
        
        return 'Unknown Location';
    }
}