@extends('layouts.admin-sidebar')

@section('title', 'Airtime & Data Providers')
@section('page-title', 'Service Providers')
@section('page-description', 'Manage airtime and data service providers and their settings')

@section('content')
<div class="space-y-6">
    @if(isset($error))
    <div class="bg-red-50 border border-red-200 rounded-lg p-4">
        <div class="flex">
            <svg class="w-5 h-5 text-red-400 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
            </svg>
            <div>
                <h3 class="text-sm font-medium text-red-800">Database Error</h3>
                <p class="text-sm text-red-700 mt-1">{{ $error }}</p>
            </div>
        </div>
    </div>
    @endif

    <!-- Page Header with Actions -->
    <div class="flex justify-between items-center">
        <div>
            <h2 class="text-2xl font-bold text-gray-900">Service Provider Management</h2>
            <p class="text-gray-600 mt-1">Configure and monitor your airtime and data service providers</p>
        </div>
        <div class="flex space-x-3">
            <button class="btn-secondary" onclick="testAllProviders()">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                </svg>
                Test All
            </button>
            <button class="btn-primary" onclick="openProviderModal()">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                </svg>
                Add Provider
            </button>
        </div>
    </div>

    <!-- Providers Grid -->
    <div id="providers-grid" class="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-6">
        @if($providers->count() > 0)
        @foreach($providers as $provider)
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow">
            <!-- Provider Header -->
            <div class="flex items-center justify-between mb-4">
                <div class="flex items-center">
                    <div class="w-12 h-12 bg-gradient-to-br from-blue-500 to-blue-600 rounded-lg flex items-center justify-center">
                        <span class="text-white font-bold text-lg">{{ strtoupper(substr($provider->name ?? 'UN', 0, 2)) }}</span>
                    </div>
                    <div class="ml-3">
                        <h3 class="text-lg font-semibold text-gray-900">{{ $provider->name ?? 'Unknown Provider' }}</h3>
                        <span class="inline-flex px-2 py-1 text-xs font-medium rounded-full
                            {{ $provider->priority > 50 ? 'bg-green-100 text-green-800' : 
                               ($provider->priority > 20 ? 'bg-blue-100 text-blue-800' : 'bg-gray-100 text-gray-800') }}">
                            {{ $provider->priority > 50 ? 'Primary' : ($provider->priority > 20 ? 'Secondary' : 'Backup') }}
                        </span>
                    </div>
                </div>
                <div class="flex items-center">
                    <span class="inline-flex px-2 py-1 text-xs font-medium rounded-full
                        {{ $provider->is_active ? 'bg-emerald-100 text-emerald-800' : 'bg-red-100 text-red-800' }}">
                        {{ $provider->is_active ? 'Active' : 'Inactive' }}
                    </span>
                </div>
            </div>

            <!-- Provider Stats -->
            <div class="grid grid-cols-2 gap-4 mb-4">
                <div class="text-center p-3 bg-gray-50 rounded-lg">
                    <div class="text-2xl font-bold text-gray-900">{{ number_format($provider->success_rate ?? 0, 1) }}%</div>
                    <div class="text-xs text-gray-600">Success Rate</div>
                </div>
                <div class="text-center p-3 bg-gray-50 rounded-lg">
                    <div class="text-2xl font-bold text-gray-900">{{ number_format(($provider->avg_response_time ?? 0) / 1000, 1) }}s</div>
                    <div class="text-xs text-gray-600">Avg Response</div>
                </div>
            </div>

            <!-- Transaction Stats -->
            <div class="flex justify-between items-center mb-4 text-sm">
                <span class="text-gray-600">Priority:</span>
                <span class="font-semibold text-gray-900">{{ $provider->priority ?? 0 }}</span>
            </div>
            <div class="flex justify-between items-center mb-4 text-sm">
                <span class="text-gray-600">Last Used:</span>
                <span class="font-semibold text-gray-900">{{ $provider->last_used_at ? $provider->last_used_at->diffForHumans() : 'Never' }}</span>
            </div>

            <!-- Supported Services -->
            <div class="mb-4">
                <label class="text-sm font-medium text-gray-700 mb-2 block">Supported Services</label>
                <div class="flex flex-wrap gap-1">
                    @if($provider->supported_services)
                        @foreach($provider->supported_services as $service)
                        <span class="inline-flex px-2 py-1 text-xs font-medium bg-blue-100 text-blue-800 rounded">
                            {{ $service }}
                        </span>
                        @endforeach
                    @else
                        <span class="text-xs text-gray-500">No services configured</span>
                    @endif
                </div>
            </div>

            <!-- Provider Actions -->
            <div class="flex space-x-2 pt-4 border-t border-gray-200">
                <button class="flex-1 btn-secondary text-sm py-2" 
                        onclick="testProvider({{ $provider->id }})">
                    <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    Test
                </button>
                <button class="flex-1 btn-primary text-sm py-2" 
                        onclick="editProvider({{ $provider->id }})">
                    <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                    </svg>
                    Edit
                </button>
                @if($provider->is_active)
                <button class="btn-danger text-sm py-2 px-3" 
                        onclick="toggleProviderStatus({{ $provider->id }}, false)">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                </button>
                @else
                <button class="btn-success text-sm py-2 px-3" 
                        onclick="toggleProviderStatus({{ $provider->id }}, true)">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                </button>
                @endif
                <button class="btn-danger text-sm py-2 px-3" 
                        onclick="deleteProvider({{ $provider->id }}, '{{ $provider->name }}')">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                    </svg>
                </button>
            </div>
        </div>
        @endforeach
        @else
        <div class="col-span-full">
            <div class="text-center py-12">
                <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"></path>
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                </svg>
                <h3 class="mt-2 text-sm font-medium text-gray-900">No providers configured</h3>
                <p class="mt-1 text-sm text-gray-500">Get started by adding your first service provider.</p>
                <div class="mt-6">
                    <button type="button" class="btn-primary" onclick="openProviderModal()">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                        </svg>
                        Add Provider
                    </button>
                </div>
            </div>
        </div>
        @endif
    </div>
</div>

<script>
// CSRF token for AJAX requests
const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');

function testProvider(providerId) {
    const button = event.target.closest('button');
    const originalContent = button.innerHTML;
    button.innerHTML = '<svg class="animate-spin w-4 h-4 mr-1" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4z"></path></svg>Testing...';
    button.disabled = true;

    fetch(`/admin/airtime/providers/${providerId}/test`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken,
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert(`Provider tested successfully! Response time: ${data.response_time || 'N/A'}ms`);
        } else {
            alert(`Test failed: ${data.message || 'Unknown error'}`);
        }
    })
    .catch(error => {
        alert(`Test failed: ${error.message}`);
    })
    .finally(() => {
        button.innerHTML = originalContent;
        button.disabled = false;
    });
}

function testAllProviders() {
    if (!confirm('Are you sure you want to test all providers? This may take a few moments.')) {
        return;
    }

    const providers = @json($providers->pluck('id'));
    let completed = 0;
    const total = providers.length;

    if (total === 0) {
        alert('No providers to test.');
        return;
    }

    alert(`Testing ${total} providers...`);

    providers.forEach(providerId => {
        fetch(`/admin/airtime/providers/${providerId}/test`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': csrfToken,
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            completed++;
            if (completed === total) {
                alert('All providers tested successfully!');
                setTimeout(() => location.reload(), 1500);
            }
        })
        .catch(error => {
            completed++;
            if (completed === total) {
                alert('Testing completed with some errors. Check individual results.');
                setTimeout(() => location.reload(), 1500);
            }
        });
    });
}

function editProvider(providerId) {
    window.location.href = `/admin/airtime/providers/${providerId}/edit`;
}

function toggleProviderStatus(providerId, status) {
    const action = status ? 'activate' : 'deactivate';
    if (!confirm(`Are you sure you want to ${action} this provider?`)) {
        return;
    }

    fetch(`/admin/airtime/providers/${providerId}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken,
            'Accept': 'application/json'
        },
        body: JSON.stringify({ is_active: status })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success !== false) {
            alert(`Provider ${action}d successfully!`);
            setTimeout(() => location.reload(), 1000);
        } else {
            alert(data.message || `Failed to ${action} provider`);
        }
    })
    .catch(error => {
        alert(`Error: ${error.message}`);
    });
}

function openProviderModal() {
    window.location.href = '/admin/airtime/providers/create';
}

function deleteProvider(providerId, providerName) {
    if (confirm(`Are you sure you want to delete ${providerName}? This action cannot be undone.`)) {
        fetch(`/admin/airtime/providers/${providerId}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': csrfToken,
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success !== false) {
                alert('Provider deleted successfully!');
                setTimeout(() => location.reload(), 1000);
            } else {
                alert(data.message || 'Failed to delete provider');
            }
        })
        .catch(error => {
            alert(`Error: ${error.message}`);
        });
    }
}
</script>

@endsection