@extends('layouts.mobile')

@section('title', 'Transaction History')

@section('content')
<div class="space-y-6">
    <!-- Transaction Summary Cards -->
    <div class="grid grid-cols-2 gap-4">
        <div class="card bg-gradient-to-r from-success-600/20 to-success-700/20 border-success-600/30">
            <div class="text-center">
                <p class="text-success-400 text-sm font-medium mb-1">Total Credit</p>
                <p class="text-2xl font-bold text-white">₦{{ number_format($summary['total_credit'] ?? 0, 2) }}</p>
            </div>
        </div>
        
        <div class="card bg-gradient-to-r from-error-600/20 to-error-700/20 border-error-600/30">
            <div class="text-center">
                <p class="text-error-400 text-sm font-medium mb-1">Total Debit</p>
                <p class="text-2xl font-bold text-white">₦{{ number_format($summary['total_debit'] ?? 0, 2) }}</p>
            </div>
        </div>
    </div>

    <!-- Filter Button -->
    <div class="flex justify-between items-center">
        <h2 class="text-lg font-semibold text-white">Transactions</h2>
        <button onclick="toggleFilters()" class="flex items-center px-4 py-2 bg-dark-700 rounded-lg text-gray-300 hover:text-white transition-colors">
            <i class="fas fa-filter mr-2"></i>
            <span>Filter</span>
        </button>
    </div>

    <!-- Filters Panel -->
    <div id="filtersPanel" class="hidden">
        <div class="card">
            <h3 class="text-lg font-semibold text-white mb-4">Filter Transactions</h3>
            <form method="GET" action="{{ route('transactions.index') }}" class="space-y-4">
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label for="type" class="form-label">Type</label>
                        <select name="type" id="type" class="form-input">
                            <option value="">All Types</option>
                            <option value="credit" {{ request('type') === 'credit' ? 'selected' : '' }}>Credit</option>
                            <option value="debit" {{ request('type') === 'debit' ? 'selected' : '' }}>Debit</option>
                        </select>
                    </div>
                    
                    <div>
                        <label for="status" class="form-label">Status</label>
                        <select name="status" id="status" class="form-input">
                            <option value="">All Status</option>
                            <option value="completed" {{ request('status') === 'completed' ? 'selected' : '' }}>Completed</option>
                            <option value="pending" {{ request('status') === 'pending' ? 'selected' : '' }}>Pending</option>
                            <option value="processing" {{ request('status') === 'processing' ? 'selected' : '' }}>Processing</option>
                            <option value="failed" {{ request('status') === 'failed' ? 'selected' : '' }}>Failed</option>
                        </select>
                    </div>
                </div>
                
                <div>
                    <label for="category" class="form-label">Category</label>
                    <select name="category" id="category" class="form-input">
                        <option value="">All Categories</option>
                        <option value="wallet_funding" {{ request('category') === 'wallet_funding' ? 'selected' : '' }}>Wallet Funding</option>
                        <option value="p2p_transfer" {{ request('category') === 'p2p_transfer' ? 'selected' : '' }}>P2P Transfer</option>
                        <option value="bank_transfer" {{ request('category') === 'bank_transfer' ? 'selected' : '' }}>Bank Transfer</option>
                        <option value="airtime_purchase" {{ request('category') === 'airtime_purchase' ? 'selected' : '' }}>Airtime</option>
                        <option value="data_purchase" {{ request('category') === 'data_purchase' ? 'selected' : '' }}>Data</option>
                        <option value="bill_payment" {{ request('category') === 'bill_payment' ? 'selected' : '' }}>Bill Payment</option>
                        <option value="virtual_card_transaction" {{ request('category') === 'virtual_card_transaction' ? 'selected' : '' }}>Virtual Card</option>
                    </select>
                </div>
                
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label for="date_from" class="form-label">From Date</label>
                        <input type="date" name="date_from" id="date_from" value="{{ request('date_from') }}" class="form-input">
                    </div>
                    
                    <div>
                        <label for="date_to" class="form-label">To Date</label>
                        <input type="date" name="date_to" id="date_to" value="{{ request('date_to') }}" class="form-input">
                    </div>
                </div>
                
                <div class="flex space-x-3">
                    @if(request()->hasAny(['type', 'category', 'status', 'date_from', 'date_to']))
                    <a href="{{ route('transactions.index') }}" class="flex-1 btn btn-secondary">
                        Clear Filters
                    </a>
                    @endif
                    <button type="submit" class="flex-1 btn btn-primary">
                        Apply Filters
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Transactions List -->
    <div class="space-y-4">
        @if($transactions && $transactions->count() > 0)
            @foreach($transactions as $transaction)
            <div class="transaction-item cursor-pointer" onclick="showTransactionDetails('{{ $transaction->id }}')">
                <div class="flex items-center flex-1">
                    <div class="transaction-icon {{ $transaction->type === 'credit' ? 'bg-success-600/20 text-success-400' : 'bg-error-600/20 text-error-400' }}">
                        @if($transaction->category === 'wallet_funding')
                            <i class="fas fa-plus"></i>
                        @elseif($transaction->category === 'p2p_transfer')
                            <i class="fas fa-exchange-alt"></i>
                        @elseif($transaction->category === 'bank_transfer')
                            <i class="fas fa-university"></i>
                        @elseif($transaction->category === 'airtime_purchase')
                            <i class="fas fa-mobile-alt"></i>
                        @elseif($transaction->category === 'data_purchase')
                            <i class="fas fa-wifi"></i>
                        @elseif($transaction->category === 'bill_payment')
                            <i class="fas fa-bolt"></i>
                        @elseif($transaction->category === 'virtual_card_transaction')
                            <i class="fas fa-credit-card"></i>
                        @else
                            <i class="fas fa-money-bill-wave"></i>
                        @endif
                    </div>
                    <div class="ml-3 flex-1">
                        <p class="text-white font-medium text-sm">{{ $transaction->description }}</p>
                        <p class="text-gray-400 text-xs">{{ $transaction->created_at->format('M d, Y H:i') }}</p>
                        @if($transaction->reference)
                        <p class="text-gray-500 text-xs font-mono">{{ $transaction->reference }}</p>
                        @endif
                    </div>
                </div>
                <div class="text-right">
                    <p class="font-semibold text-sm {{ $transaction->type === 'credit' ? 'text-success-400' : 'text-error-400' }}">
                        {{ $transaction->type === 'credit' ? '+' : '-' }}₦{{ number_format($transaction->amount, 2) }}
                    </p>
                    <div class="status-badge mt-1 
                        {{ $transaction->status === 'completed' ? 'status-success' : 
                           ($transaction->status === 'pending' || $transaction->status === 'processing' ? 'status-pending' : 'status-failed') }}">
                        {{ ucfirst($transaction->status) }}
                    </div>
                    @if($transaction->fee && $transaction->fee > 0)
                    <p class="text-gray-500 text-xs mt-1">Fee: ₦{{ number_format($transaction->fee, 2) }}</p>
                    @endif
                </div>
            </div>
            @endforeach
            
            <!-- Pagination -->
            @if($transactions->hasPages())
            <div class="flex justify-center mt-6">
                <div class="flex space-x-2">
                    @if($transactions->onFirstPage())
                        <button disabled class="px-3 py-2 bg-dark-800 text-gray-500 rounded-lg">
                            <i class="fas fa-chevron-left"></i>
                        </button>
                    @else
                        <a href="{{ $transactions->previousPageUrl() }}" class="px-3 py-2 bg-dark-700 text-white rounded-lg hover:bg-dark-600">
                            <i class="fas fa-chevron-left"></i>
                        </a>
                    @endif
                    
                    <span class="px-3 py-2 bg-primary-600 text-white rounded-lg">
                        {{ $transactions->currentPage() }} of {{ $transactions->lastPage() }}
                    </span>
                    
                    @if($transactions->hasMorePages())
                        <a href="{{ $transactions->nextPageUrl() }}" class="px-3 py-2 bg-dark-700 text-white rounded-lg hover:bg-dark-600">
                            <i class="fas fa-chevron-right"></i>
                        </a>
                    @else
                        <button disabled class="px-3 py-2 bg-dark-800 text-gray-500 rounded-lg">
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    @endif
                </div>
            </div>
            @endif
        @else
            <div class="text-center py-12">
                <div class="w-16 h-16 bg-dark-800 rounded-full flex items-center justify-center mx-auto mb-4">
                    <i class="fas fa-receipt text-gray-500 text-2xl"></i>
                </div>
                <p class="text-gray-400 font-medium mb-2">No transactions found</p>
                @if(request()->hasAny(['type', 'category', 'status', 'date_from', 'date_to']))
                    <p class="text-gray-500 text-sm mb-4">Try adjusting your filters</p>
                    <a href="{{ route('transactions.index') }}" class="btn btn-secondary">Clear Filters</a>
                @else
                    <p class="text-gray-500 text-sm mb-4">Your transaction history will appear here</p>
                    <a href="{{ route('wallet.index') }}" class="btn btn-primary">Fund Wallet</a>
                @endif
            </div>
        @endif
    </div>

    <!-- Export Options -->
    @if($transactions && $transactions->count() > 0)
    <div class="card">
        <h3 class="text-lg font-semibold text-white mb-4">Export Transactions</h3>
        <div class="grid grid-cols-2 gap-4">
            <button onclick="exportTransactions('csv')" class="btn btn-secondary flex items-center justify-center">
                <i class="fas fa-file-csv mr-2"></i>
                Export CSV
            </button>
            <button onclick="exportTransactions('pdf')" class="btn btn-secondary flex items-center justify-center">
                <i class="fas fa-file-pdf mr-2"></i>
                Export PDF
            </button>
        </div>
    </div>
    @endif
</div>

<!-- Transaction Details Modal -->
<div id="transactionModal" class="fixed inset-0 bg-black/50 backdrop-blur-sm hidden z-50">
    <div class="absolute bottom-0 left-0 right-0 bg-dark-900 rounded-t-3xl p-6 slide-up">
        <div class="w-12 h-1 bg-gray-600 rounded-full mx-auto mb-6"></div>
        <div id="transactionDetails">
            <!-- Transaction details will be loaded here -->
        </div>
        <button onclick="closeTransactionModal()" class="w-full btn btn-secondary mt-4">Close</button>
    </div>
</div>

@push('scripts')
<script>
function toggleFilters() {
    const panel = document.getElementById('filtersPanel');
    panel.classList.toggle('hidden');
}

function showTransactionDetails(transactionId) {
    // TODO: Implement transaction details loading
    document.getElementById('transactionModal').classList.remove('hidden');
    document.getElementById('transactionDetails').innerHTML = `
        <div class="text-center py-8">
            <div class="w-8 h-8 border-2 border-primary-600 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
            <p class="text-gray-400">Loading transaction details...</p>
        </div>
    `;
    
    // Simulate loading (replace with actual API call)
    setTimeout(() => {
        document.getElementById('transactionDetails').innerHTML = `
            <h3 class="text-xl font-semibold text-white mb-4">Transaction Details</h3>
            <div class="space-y-3">
                <div class="flex justify-between">
                    <span class="text-gray-400">Transaction ID:</span>
                    <span class="text-white font-mono">${transactionId}</span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-400">Status:</span>
                    <span class="status-badge status-success">Completed</span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-400">Date:</span>
                    <span class="text-white">${new Date().toLocaleDateString()}</span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-400">Amount:</span>
                    <span class="text-success-400 font-semibold">₦1,000.00</span>
                </div>
            </div>
        `;
    }, 1000);
}

function closeTransactionModal() {
    document.getElementById('transactionModal').classList.add('hidden');
}

function exportTransactions(format) {
    const params = new URLSearchParams(window.location.search);
    params.set('export', format);
    
    const exportUrl = `{{ route('transactions.index') }}?${params.toString()}`;
    window.open(exportUrl, '_blank');
}

// Close modal when clicking outside
document.addEventListener('click', function(event) {
    const modal = document.getElementById('transactionModal');
    if (event.target === modal) {
        closeTransactionModal();
    }
});

// Real-time transaction updates
function refreshTransactions() {
    // TODO: Implement real-time updates via WebSocket or polling
    console.log('Refreshing transactions...');
}

// Check for new transactions every 30 seconds
setInterval(refreshTransactions, 30000);
</script>
@endpush
@endsection
    
    <div class="bg-gradient-to-r from-blue-50 to-blue-100 rounded-lg p-4">
        <h3 class="text-sm font-medium text-blue-600 mb-1">Total Transactions</h3>
        <p class="text-xl font-bold text-blue-900">{{ $summary['total_count'] ?? 0 }}</p>
    </div>
    
    <div class="bg-gradient-to-r from-purple-50 to-purple-100 rounded-lg p-4">
        <h3 class="text-sm font-medium text-purple-600 mb-1">Total Fees</h3>
        <p class="text-xl font-bold text-purple-900">₦{{ number_format($summary['total_fees'] ?? 0, 2) }}</p>
    </div>
</div>

<!-- Transactions List -->
<div class="card">
    @if($transactions->count() > 0)
        <div class="space-y-4">
            @foreach($transactions as $transaction)
            <div class="flex justify-between items-center p-4 bg-gray-50 hover:bg-gray-100 rounded-lg transition-colors cursor-pointer"
                 onclick="openTransactionModal('{{ $transaction->id }}')">
                <div class="flex items-center">
                    <div class="w-12 h-12 rounded-full flex items-center justify-center mr-4 {{ $transaction->type === 'credit' ? 'bg-green-100' : 'bg-red-100' }}">
                        <span class="text-xl">
                            @if($transaction->category === 'wallet_funding') 💰
                            @elseif($transaction->category === 'p2p_transfer') 👥
                            @elseif($transaction->category === 'bank_transfer') 🏦
                            @elseif($transaction->category === 'airtime_purchase') 📱
                            @elseif($transaction->category === 'data_purchase') 📶
                            @elseif($transaction->category === 'bill_payment') ⚡
                            @elseif($transaction->category === 'virtual_card_transaction') 💳
                            @elseif($transaction->category === 'refund') 🔄
                            @else 💼
                            @endif
                        </span>
                    </div>
                    <div>
                        <p class="font-medium text-gray-900">{{ $transaction->description }}</p>
                        <p class="text-sm text-gray-600">{{ $transaction->created_at->format('M d, Y H:i A') }}</p>
                        @if($transaction->external_reference)
                        <p class="text-xs text-gray-500">Ref: {{ $transaction->reference }}</p>
                        @endif
                    </div>
                </div>
                <div class="text-right">
                    <p class="font-bold text-lg {{ $transaction->type === 'credit' ? 'text-green-600' : 'text-red-600' }}">
                        {{ $transaction->type === 'credit' ? '+' : '-' }}₦{{ number_format($transaction->amount, 2) }}
                    </p>
                    @if($transaction->fee > 0)
                    <p class="text-xs text-gray-500">Fee: ₦{{ number_format($transaction->fee, 2) }}</p>
                    @endif
                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium mt-1
                        @if($transaction->status === 'completed') bg-green-100 text-green-800
                        @elseif($transaction->status === 'pending' || $transaction->status === 'processing') bg-yellow-100 text-yellow-800
                        @else bg-red-100 text-red-800
                        @endif">
                        {{ ucfirst($transaction->status) }}
                    </span>
                </div>
            </div>
            @endforeach
        </div>
        
        <!-- Pagination -->
        <div class="mt-6">
            {{ $transactions->appends(request()->query())->links() }}
        </div>
    @else
        <div class="text-center py-12 text-gray-500">
            <span class="text-6xl mb-4 block">📊</span>
            <p class="text-lg font-medium mb-2">No transactions found</p>
            <p class="text-sm">Try adjusting your filters or check back later</p>
        </div>
    @endif
</div>

<!-- Transaction Detail Modal -->
<div id="transactionModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-medium text-gray-900">Transaction Details</h3>
                <button onclick="closeTransactionModal()" class="text-gray-400 hover:text-gray-600">
                    <span class="text-2xl">&times;</span>
                </button>
            </div>
            <div id="transactionDetails">
                <!-- Transaction details will be loaded here -->
            </div>
        </div>
    </div>
</div>

<script>
function openTransactionModal(transactionId) {
    document.getElementById('transactionModal').classList.remove('hidden');
    // Load transaction details via AJAX
    loadTransactionDetails(transactionId);
}

function closeTransactionModal() {
    document.getElementById('transactionModal').classList.add('hidden');
}

function loadTransactionDetails(transactionId) {
    // This would normally make an AJAX request to get transaction details
    document.getElementById('transactionDetails').innerHTML = '<p class="text-center">Loading...</p>';
    
    // Placeholder implementation
    setTimeout(() => {
        document.getElementById('transactionDetails').innerHTML = `
            <div class="space-y-3">
                <div class="bg-gray-50 rounded-lg p-3">
                    <p class="text-sm text-gray-600">Transaction ID</p>
                    <p class="font-medium">${transactionId}</p>
                </div>
                <div class="bg-gray-50 rounded-lg p-3">
                    <p class="text-sm text-gray-600">Status</p>
                    <p class="font-medium">Completed</p>
                </div>
                <div class="bg-gray-50 rounded-lg p-3">
                    <p class="text-sm text-gray-600">Date</p>
                    <p class="font-medium">${new Date().toLocaleDateString()}</p>
                </div>
                <button onclick="downloadReceipt('${transactionId}')" class="w-full mt-4 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                    Download Receipt
                </button>
            </div>
        `;
    }, 500);
}

function downloadReceipt(transactionId) {
    alert('Downloading receipt for transaction ' + transactionId);
    // Implementation for downloading receipt
}

// Close modal when clicking outside
document.getElementById('transactionModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeTransactionModal();
    }
});
</script>
@endsection