<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Wallet;
use App\Models\VirtualAccount;
use App\Models\CronLog;
use App\Services\MonnifyService;
use App\Services\PaystackService;

class SyncWalletsCommand extends Command
{
    protected $signature = 'fintech:sync-wallets';
    protected $description = 'Sync wallet balances with external APIs';

    protected $monnifyService;
    protected $paystackService;

    public function __construct(MonnifyService $monnifyService, PaystackService $paystackService)
    {
        parent::__construct();
        $this->monnifyService = $monnifyService;
        $this->paystackService = $paystackService;
    }

    public function handle()
    {
        $startTime = microtime(true);
        $startMemory = memory_get_usage(true);
        
        try {
            $this->info('Starting wallet synchronization...');
            
            $syncedCount = 0;
            $errorCount = 0;
            
            // Get all virtual accounts that need syncing
            $virtualAccounts = VirtualAccount::where('status', 'active')
                ->where('updated_at', '<', now()->subMinutes(5))
                ->with('user.wallet')
                ->get();
            
            foreach ($virtualAccounts as $account) {
                try {
                    if ($account->provider === 'monnify') {
                        $this->syncMonnifyAccount($account);
                    } elseif ($account->provider === 'paystack') {
                        $this->syncPaystackAccount($account);
                    }
                    
                    $syncedCount++;
                    
                } catch (\Exception $e) {
                    $this->error("Error syncing account {$account->account_number}: " . $e->getMessage());
                    $errorCount++;
                }
            }
            
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $message = "Synced {$syncedCount} wallets, {$errorCount} errors";
            $this->info($message);
            
            // Log execution
            CronLog::create([
                'task_name' => 'sync_wallets',
                'status' => $errorCount > 0 ? 'warning' : 'success',
                'message' => $message,
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage
            ]);
            
        } catch (\Exception $e) {
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $this->error('Wallet sync failed: ' . $e->getMessage());
            
            CronLog::create([
                'task_name' => 'sync_wallets',
                'status' => 'failed',
                'message' => 'Wallet sync failed',
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage,
                'error_details' => [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]
            ]);
        }
    }
    
    private function syncMonnifyAccount(VirtualAccount $account)
    {
        $response = $this->monnifyService->getAccountTransactions($account->account_number);
        
        if ($response['success'] && !empty($response['transactions'])) {
            foreach ($response['transactions'] as $transaction) {
                // Check if transaction already processed
                if (!$this->isTransactionProcessed($transaction['reference'])) {
                    $this->processIncomingTransaction($account, $transaction);
                }
            }
        }
        
        // Update last sync time
        $account->touch();
    }
    
    private function syncPaystackAccount(VirtualAccount $account)
    {
        // Implement Paystack account sync if applicable
        $account->touch();
    }
    
    private function isTransactionProcessed($reference)
    {
        return \App\Models\Transaction::where('external_reference', $reference)->exists();
    }
    
    private function processIncomingTransaction(VirtualAccount $account, $transactionData)
    {
        if ($transactionData['type'] === 'credit' && $transactionData['status'] === 'successful') {
            $wallet = $account->user->wallet;
            
            // Create credit transaction
            $transaction = \App\Models\Transaction::create([
                'user_id' => $account->user_id,
                'wallet_id' => $wallet->id,
                'type' => 'credit',
                'category' => 'virtual_account_funding',
                'amount' => $transactionData['amount'],
                'fee' => 0,
                'total_amount' => $transactionData['amount'],
                'currency' => 'NGN',
                'status' => 'completed',
                'description' => 'Virtual account funding',
                'external_reference' => $transactionData['reference'],
                'metadata' => [
                    'virtual_account_number' => $account->account_number,
                    'sender_name' => $transactionData['sender_name'] ?? 'Unknown',
                    'sender_bank' => $transactionData['sender_bank'] ?? 'Unknown'
                ]
            ]);
            
            // Credit wallet
            $wallet->credit($transactionData['amount']);
            
            $this->info("Processed credit: ₦{$transactionData['amount']} for user {$account->user_id}");
        }
    }
}