<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;

class ApiConfigController extends Controller
{
    /**
     * Display API configuration dashboard.
     */
    public function index()
    {
        // For demo purposes, create mock API services
        $mockServices = collect([
            (object)[
                'id' => 1,
                'name' => 'Paystack',
                'category' => 'payment',
                'description' => 'Payment processing and card transactions',
                'base_url' => 'https://api.paystack.co',
                'api_key' => 'pk_test_••••••••••••1234',
                'is_active' => true,
                'is_live_mode' => false,
            ],
            (object)[
                'id' => 2,
                'name' => 'Flutterwave',
                'category' => 'payment',
                'description' => 'Payment gateway and financial services',
                'base_url' => 'https://api.flutterwave.com',
                'api_key' => 'FLWPUBK••••••••••••5678',
                'is_active' => true,
                'is_live_mode' => true,
            ],
            (object)[
                'id' => 3,
                'name' => 'Smile Identity',
                'category' => 'identity',
                'description' => 'Identity verification and KYC services',
                'base_url' => 'https://api.smileidentity.com',
                'api_key' => 'test••••••••••••abcd',
                'is_active' => false,
                'is_live_mode' => false,
            ],
            (object)[
                'id' => 4,
                'name' => 'Termii',
                'category' => 'messaging',
                'description' => 'SMS and messaging services',
                'base_url' => 'https://api.ng.termii.com',
                'api_key' => 'TL••••••••••••efgh',
                'is_active' => true,
                'is_live_mode' => false,
            ],
            (object)[
                'id' => 5,
                'name' => 'VTPass',
                'category' => 'utilities',
                'description' => 'Bill payment and utility services',
                'base_url' => 'https://vtpass.com/api',
                'api_key' => 'VT••••••••••••ijkl',
                'is_active' => true,
                'is_live_mode' => true,
            ],
        ]);
        
        // Group services by category
        $serviceCategories = [
            'payment' => $mockServices->where('category', 'payment'),
            'identity' => $mockServices->where('category', 'identity'),
            'messaging' => $mockServices->where('category', 'messaging'),
            'utilities' => $mockServices->where('category', 'utilities'),
        ];

        // Mock API Health Stats
        $healthStats = [
            'Paystack' => ['status' => 'healthy', 'last_check' => now()->subMinutes(5), 'response_time' => 120],
            'Flutterwave' => ['status' => 'healthy', 'last_check' => now()->subMinutes(3), 'response_time' => 95],
            'Smile Identity' => ['status' => 'unhealthy', 'last_check' => now()->subMinutes(10), 'response_time' => null],
            'Termii' => ['status' => 'healthy', 'last_check' => now()->subMinutes(2), 'response_time' => 200],
            'VTPass' => ['status' => 'error', 'last_check' => now()->subMinutes(15), 'response_time' => null],
        ];

        return view('admin.api-config.index', compact('serviceCategories', 'healthStats'));
    }

    /**
     * Show API service configuration.
     */
    public function show(Service $service)
    {
        return view('admin.api-config.show', compact('service'));
    }

    /**
     * Update API service configuration.
     */
    public function update(Request $request, Service $service)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'base_url' => 'required|url',
            'api_key' => 'nullable|string',
            'secret_key' => 'nullable|string',
            'is_active' => 'boolean',
            'is_live_mode' => 'boolean',
            'timeout' => 'required|integer|min:5|max:120',
            'retry_attempts' => 'required|integer|min:0|max:5',
            'webhook_url' => 'nullable|url',
        ]);

        DB::transaction(function () use ($request, $service) {
            $service->update([
                'name' => $request->name,
                'description' => $request->description,
                'base_url' => $request->base_url,
                'api_key' => $request->api_key,
                'secret_key' => $request->secret_key,
                'is_active' => $request->boolean('is_active'),
                'is_live_mode' => $request->boolean('is_live_mode'),
                'config' => json_encode([
                    'timeout' => $request->timeout,
                    'retry_attempts' => $request->retry_attempts,
                    'webhook_url' => $request->webhook_url,
                    'headers' => $request->headers ?? [],
                ]),
                'updated_at' => now(),
            ]);

            // Log admin action
            \App\Models\AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'update_api_config',
                'target_type' => 'Service',
                'target_id' => $service->id,
                'old_values' => $service->getOriginal(),
                'new_values' => $service->getAttributes(),
                'ip_address' => request()->ip(),
            ]);
        });

        return back()->with('success', 'API configuration updated successfully.');
    }

    /**
     * Test API connection.
     */
    public function testConnection(Service $service)
    {
        try {
            $startTime = microtime(true);
            
            // Build test request based on service type
            $testEndpoint = $this->getTestEndpoint($service);
            $headers = $this->buildHeaders($service);
            
            $response = Http::withHeaders($headers)
                           ->timeout($service->config['timeout'] ?? 30)
                           ->get($service->base_url . $testEndpoint);
            
            $responseTime = round((microtime(true) - $startTime) * 1000, 2); // in milliseconds
            
            $result = [
                'success' => $response->successful(),
                'status_code' => $response->status(),
                'response_time' => $responseTime,
                'response_body' => $response->json() ?? $response->body(),
                'headers' => $response->headers(),
                'tested_at' => now(),
            ];

            // Cache the health status
            Cache::put("api_health_{$service->name}", [
                'status' => $response->successful() ? 'healthy' : 'unhealthy',
                'last_check' => now(),
                'response_time' => $responseTime,
            ], 300); // Cache for 5 minutes

            return response()->json($result);

        } catch (\Exception $e) {
            Cache::put("api_health_{$service->name}", [
                'status' => 'error',
                'last_check' => now(),
                'response_time' => null,
                'error' => $e->getMessage(),
            ], 300);

            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'tested_at' => now(),
            ], 500);
        }
    }

    /**
     * Toggle service active status.
     */
    public function toggleStatus(Service $service)
    {
        DB::transaction(function () use ($service) {
            $oldStatus = $service->is_active;
            $service->update(['is_active' => !$service->is_active]);

            // Log admin action
            \App\Models\AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => $service->is_active ? 'activate_service' : 'deactivate_service',
                'target_type' => 'Service',
                'target_id' => $service->id,
                'old_values' => ['is_active' => $oldStatus],
                'new_values' => ['is_active' => $service->is_active],
                'ip_address' => request()->ip(),
            ]);
        });

        $message = $service->is_active ? 'Service activated successfully.' : 'Service deactivated successfully.';
        return back()->with('success', $message);
    }

    /**
     * Toggle live/test mode.
     */
    public function toggleMode(Service $service)
    {
        DB::transaction(function () use ($service) {
            $oldMode = $service->is_live_mode;
            $service->update(['is_live_mode' => !$service->is_live_mode]);

            // Log admin action
            \App\Models\AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'toggle_service_mode',
                'target_type' => 'Service',
                'target_id' => $service->id,
                'old_values' => ['is_live_mode' => $oldMode],
                'new_values' => ['is_live_mode' => $service->is_live_mode],
                'ip_address' => request()->ip(),
            ]);
        });

        $mode = $service->is_live_mode ? 'live' : 'test';
        return back()->with('success', "Service switched to {$mode} mode successfully.");
    }

    /**
     * View API logs.
     */
    public function logs(Request $request)
    {
        $query = \App\Models\AuditLog::with('admin')
                                    ->whereIn('action', [
                                        'update_api_config',
                                        'activate_service',
                                        'deactivate_service',
                                        'toggle_service_mode',
                                        'test_api_connection'
                                    ]);

        if ($request->filled('service')) {
            $query->where('target_type', 'Service')
                  ->where('target_id', $request->service);
        }

        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->latest()->paginate(20);
        $services = Service::all();

        return view('admin.api-config.logs', compact('logs', 'services'));
    }

    /**
     * Get appropriate test endpoint for service.
     */
    private function getTestEndpoint(Service $service): string
    {
        $testEndpoints = [
            'paystack' => '/bank',
            'flutterwave' => '/banks/NG',
            'mono' => '/coverage',
            'smile_identity' => '/services',
            'termii' => '/sender-id',
            'default' => '/health',
        ];

        return $testEndpoints[strtolower($service->name)] ?? $testEndpoints['default'];
    }

    /**
     * Build headers for API request.
     */
    private function buildHeaders(Service $service): array
    {
        $headers = [
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
        ];

        if ($service->api_key) {
            // Different services use different authentication methods
            switch (strtolower($service->name)) {
                case 'paystack':
                    $headers['Authorization'] = 'Bearer ' . $service->api_key;
                    break;
                case 'flutterwave':
                    $headers['Authorization'] = 'Bearer ' . $service->api_key;
                    break;
                case 'termii':
                    $headers['Authorization'] = 'Bearer ' . $service->api_key;
                    break;
                default:
                    $headers['Authorization'] = 'Bearer ' . $service->api_key;
            }
        }

        // Add custom headers from config
        $customHeaders = $service->config['headers'] ?? [];
        return array_merge($headers, $customHeaders);
    }
}