<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Models\Wallet;
use App\Services\SmsService;

class AuthController extends Controller
{
    protected $smsService;

    public function __construct(SmsService $smsService)
    {
        $this->smsService = $smsService;
    }

    /**
     * Show the login form.
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }

    /**
     * Show the registration form.
     */
    public function showRegistrationForm()
    {
        return view('auth.register');
    }

    /**
     * Show the phone verification form.
     */
    public function showVerifyPhone()
    {
        return view('auth.verify-phone');
    }

    /**
     * Handle user registration.
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'required|string|max:15|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'date_of_birth' => 'required|date|before:18 years ago',
            'gender' => 'required|in:male,female,other',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Create user
        $user = User::create([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
            'date_of_birth' => $request->date_of_birth,
            'gender' => $request->gender,
            'country' => 'Nigeria',
        ]);

        // Create default wallet
        Wallet::create([
            'user_id' => $user->id,
            'currency' => 'NGN',
        ]);

        // Send phone verification OTP
        $this->smsService->sendOtp($user->phone);

        Auth::login($user);

        return redirect()->route('verify.phone')
            ->with('success', 'Registration successful! Please verify your phone number.');
    }

    /**
     * Handle user login.
     */
    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (Auth::attempt($credentials, $request->boolean('remember'))) {
            $request->session()->regenerate();

            // Update last login info
            Auth::user()->update([
                'last_login_at' => now(),
                'last_login_ip' => $request->ip(),
            ]);

            // Redirect based on phone verification and KYC status
            if (!Auth::user()->hasVerifiedPhone()) {
                return redirect()->route('verify.phone');
            }

            if (!Auth::user()->hasCompletedKyc()) {
                return redirect()->route('kyc.index');
            }

            return redirect()->intended('/dashboard');
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ])->onlyInput('email');
    }

    /**
     * Handle phone verification.
     */
    public function verifyPhone(Request $request)
    {
        $request->validate([
            'otp' => 'required|numeric|digits:6',
        ]);

        $user = Auth::user();
        
        if ($this->smsService->verifyOtp($user->phone, $request->otp)) {
            $user->update([
                'phone_verified_at' => now(),
            ]);

            if (!$user->hasCompletedKyc()) {
                return redirect()->route('kyc.index')
                    ->with('success', 'Phone verified successfully! Please complete your KYC verification.');
            }

            return redirect()->route('dashboard')
                ->with('success', 'Phone verified successfully!');
        }

        return back()->withErrors([
            'otp' => 'Invalid or expired OTP code.',
        ]);
    }

    /**
     * Handle user logout.
     */
    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }
}