<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class KycVerification extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'type',
        'verification_id',
        'status',
        'provider',
        'provider_reference',
        'verification_data',
        'document_path',
        'failure_reason',
        'verified_at',
        'expires_at',
    ];

    protected function casts(): array
    {
        return [
            'verification_data' => 'array',
            'verified_at' => 'datetime',
            'expires_at' => 'datetime',
        ];
    }

    /**
     * Get the user that owns the KYC verification.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Check if verification is verified.
     */
    public function isVerified(): bool
    {
        return $this->status === 'verified';
    }

    /**
     * Check if verification has expired.
     */
    public function isExpired(): bool
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * Mark verification as verified.
     */
    public function markAsVerified(array $verificationData = []): void
    {
        $this->update([
            'status' => 'verified',
            'verification_data' => $verificationData,
            'verified_at' => now(),
            'failure_reason' => null,
        ]);
    }

    /**
     * Mark verification as failed.
     */
    public function markAsFailed(string $reason): void
    {
        $this->update([
            'status' => 'failed',
            'failure_reason' => $reason,
        ]);
    }
}