<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\AirtimeProvider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AirtimeController extends Controller
{
    /**
     * Display airtime and data orders.
     */
    public function orders(Request $request)
    {
        $query = Transaction::with(['user'])
                          ->whereIn('category', ['airtime', 'data']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('reference', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('user', function($subq) use ($search) {
                      $subq->where('first_name', 'like', "%{$search}%")
                           ->orWhere('last_name', 'like', "%{$search}%")
                           ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('provider')) {
            $query->whereJsonContains('metadata->provider', $request->provider);
        }

        if ($request->filled('network')) {
            $query->whereJsonContains('metadata->network', $request->network);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        $orders = $query->latest()->paginate(20)->withQueryString();

        // Statistics
        $stats = [
            'total_orders' => Transaction::whereIn('category', ['airtime', 'data'])->count(),
            'successful_orders' => Transaction::whereIn('category', ['airtime', 'data'])->where('status', 'completed')->count(),
            'failed_orders' => Transaction::whereIn('category', ['airtime', 'data'])->where('status', 'failed')->count(),
            'pending_orders' => Transaction::whereIn('category', ['airtime', 'data'])->where('status', 'pending')->count(),
            'total_volume' => Transaction::whereIn('category', ['airtime', 'data'])->where('status', 'completed')->sum('amount'),
            'today_orders' => Transaction::whereIn('category', ['airtime', 'data'])->whereDate('created_at', today())->count(),
            'airtime_orders' => Transaction::where('category', 'airtime')->count(),
            'data_orders' => Transaction::where('category', 'data')->count(),
        ];

        return view('admin.airtime.orders', compact('orders', 'stats'));
    }

    /**
     * Display failed airtime/data transactions.
     */
    public function failed(Request $request)
    {
        $query = Transaction::with(['user'])
                          ->whereIn('category', ['airtime', 'data'])
                          ->where('status', 'failed');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        $failedTransactions = $query->latest()->paginate(20)->withQueryString();

        $stats = [
            'total_failed' => $failedTransactions->total(),
            'airtime_failed' => Transaction::where('category', 'airtime')->where('status', 'failed')->count(),
            'data_failed' => Transaction::where('category', 'data')->where('status', 'failed')->count(),
            'failed_volume' => $query->sum('amount'),
        ];

        return view('admin.airtime.failed', compact('failedTransactions', 'stats'));
    }

    /**
     * Manage airtime/data providers.
     */
    public function providers()
    {
        // Get actual provider data from API configs
        $apiProviders = \App\Models\ApiConfig::where('category', 'utilities')
                                           ->orWhere('category', 'airtime')
                                           ->get();

        $providers = $apiProviders->map(function ($config) {
            // Calculate success rate from recent transactions
            $recentTransactions = Transaction::whereJsonContains('metadata->provider', $config->slug)
                                            ->where('created_at', '>=', now()->subDays(30))
                                            ->count();
            
            $successfulTransactions = Transaction::whereJsonContains('metadata->provider', $config->slug)
                                                ->where('status', 'completed')
                                                ->where('created_at', '>=', now()->subDays(30))
                                                ->count();

            $successRate = $recentTransactions > 0 ? ($successfulTransactions / $recentTransactions) * 100 : 0;
            
            // Get last used timestamp
            $lastTransaction = Transaction::whereJsonContains('metadata->provider', $config->slug)
                                         ->latest()
                                         ->first();

            return [
                'id' => $config->id,
                'name' => $config->name,
                'type' => $config->is_live_mode ? 'Primary' : 'Test',
                'status' => $config->is_active ? 'active' : 'inactive',
                'success_rate' => round($successRate, 1),
                'avg_response_time' => json_decode($config->last_test_response, true)['response_time'] ?? 0,
                'last_used' => $lastTransaction ? $lastTransaction->created_at->diffForHumans() : 'Never',
                'total_transactions' => $recentTransactions,
                'supported_services' => $config->config['supported_services'] ?? ['Airtime', 'Data'],
                'config' => $config,
            ];
        })->toArray();

        return view('admin.airtime.providers', compact('providers'));
    }

    /**
     * Manage profit margins.
     */
    public function margins()
    {
        // Get margin settings from database/cache or use defaults
        $margins = [
            'networks' => [
                'mtn' => [
                    'name' => 'MTN', 
                    'airtime_margin' => setting('margin.mtn.airtime', 2.5), 
                    'data_margin' => setting('margin.mtn.data', 3.0), 
                    'status' => 'active'
                ],
                'glo' => [
                    'name' => 'Glo', 
                    'airtime_margin' => setting('margin.glo.airtime', 2.0), 
                    'data_margin' => setting('margin.glo.data', 2.5), 
                    'status' => 'active'
                ],
                'airtel' => [
                    'name' => 'Airtel', 
                    'airtime_margin' => setting('margin.airtel.airtime', 2.2), 
                    'data_margin' => setting('margin.airtel.data', 2.8), 
                    'status' => 'active'
                ],
                '9mobile' => [
                    'name' => '9mobile', 
                    'airtime_margin' => setting('margin.9mobile.airtime', 1.8), 
                    'data_margin' => setting('margin.9mobile.data', 2.3), 
                    'status' => 'active'
                ],
            ],
            'settings' => [
                'default_airtime_margin' => setting('margin.default.airtime', 2.0),
                'default_data_margin' => setting('margin.default.data', 2.5),
                'minimum_margin' => setting('margin.minimum', 1.0),
                'maximum_margin' => setting('margin.maximum', 10.0),
            ]
        ];

        return view('admin.airtime.margins', compact('margins'));
    }

    /**
     * Resend failed transaction.
     */
    public function resendTransaction(Request $request, Transaction $transaction)
    {
        if ($transaction->status !== 'failed' || !in_array($transaction->category, ['airtime', 'data'])) {
            return back()->withErrors(['error' => 'Invalid transaction for resend.']);
        }

        // In a real implementation, this would retry the transaction with the provider
        $transaction->update([
            'status' => 'pending',
            'metadata' => array_merge(
                json_decode($transaction->metadata, true) ?? [],
                ['resent_at' => now(), 'resent_by' => auth('admin')->id()]
            ),
        ]);

        // Log admin action
        \App\Models\AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'resend_transaction',
            'target_type' => 'Transaction',
            'target_id' => $transaction->id,
            'data' => ['category' => $transaction->category, 'amount' => $transaction->amount],
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', 'Transaction has been queued for retry.');
    }
}