<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\AirtimeProvider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AirtimeController extends Controller
{
    /**
     * Display airtime and data orders.
     */
    public function orders(Request $request)
    {
        $query = Transaction::with(['user'])
                          ->whereIn('category', ['airtime', 'data']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('reference', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('user', function($subq) use ($search) {
                      $subq->where('first_name', 'like', "%{$search}%")
                           ->orWhere('last_name', 'like', "%{$search}%")
                           ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('provider')) {
            $query->whereJsonContains('metadata->provider', $request->provider);
        }

        if ($request->filled('network')) {
            $query->whereJsonContains('metadata->network', $request->network);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        $orders = $query->latest()->paginate(20)->withQueryString();

        // Statistics
        $stats = [
            'total_orders' => Transaction::whereIn('category', ['airtime', 'data'])->count(),
            'successful_orders' => Transaction::whereIn('category', ['airtime', 'data'])->where('status', 'completed')->count(),
            'failed_orders' => Transaction::whereIn('category', ['airtime', 'data'])->where('status', 'failed')->count(),
            'pending_orders' => Transaction::whereIn('category', ['airtime', 'data'])->where('status', 'pending')->count(),
            'total_volume' => Transaction::whereIn('category', ['airtime', 'data'])->where('status', 'completed')->sum('amount'),
            'today_orders' => Transaction::whereIn('category', ['airtime', 'data'])->whereDate('created_at', today())->count(),
            'airtime_orders' => Transaction::where('category', 'airtime')->count(),
            'data_orders' => Transaction::where('category', 'data')->count(),
        ];

        return view('admin.airtime.orders', compact('orders', 'stats'));
    }

    /**
     * Display failed airtime/data transactions.
     */
    public function failed(Request $request)
    {
        $query = Transaction::with(['user'])
                          ->whereIn('category', ['airtime', 'data'])
                          ->where('status', 'failed');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        $failedTransactions = $query->latest()->paginate(20)->withQueryString();

        $stats = [
            'total_failed' => $failedTransactions->total(),
            'airtime_failed' => Transaction::where('category', 'airtime')->where('status', 'failed')->count(),
            'data_failed' => Transaction::where('category', 'data')->where('status', 'failed')->count(),
            'failed_volume' => $query->sum('amount'),
        ];

        return view('admin.airtime.failed', compact('failedTransactions', 'stats'));
    }

    /**
     * Manage airtime/data providers.
     */
    public function providers()
    {
        // Mock provider data - in a real app, this would come from a database
        $providers = [
            [
                'id' => 1,
                'name' => 'VTPass',
                'type' => 'Primary',
                'status' => 'active',
                'success_rate' => 98.5,
                'avg_response_time' => 2.3,
                'last_used' => '2 minutes ago',
                'total_transactions' => 5420,
                'supported_services' => ['Airtime', 'Data', 'Cable TV', 'Electricity'],
            ],
            [
                'id' => 2,
                'name' => 'Buypower',
                'type' => 'Secondary',
                'status' => 'active',
                'success_rate' => 95.2,
                'avg_response_time' => 3.1,
                'last_used' => '15 minutes ago',
                'total_transactions' => 2340,
                'supported_services' => ['Airtime', 'Data'],
            ],
            [
                'id' => 3,
                'name' => 'Payscribe',
                'type' => 'Backup',
                'status' => 'inactive',
                'success_rate' => 92.1,
                'avg_response_time' => 4.2,
                'last_used' => '3 days ago',
                'total_transactions' => 856,
                'supported_services' => ['Airtime', 'Data'],
            ],
        ];

        return view('admin.airtime.providers', compact('providers'));
    }

    /**
     * Manage profit margins.
     */
    public function margins()
    {
        // Mock margin data - in a real app, this would come from settings/config
        $margins = [
            'networks' => [
                'mtn' => ['name' => 'MTN', 'airtime_margin' => 2.5, 'data_margin' => 3.0, 'status' => 'active'],
                'glo' => ['name' => 'Glo', 'airtime_margin' => 2.0, 'data_margin' => 2.5, 'status' => 'active'],
                'airtel' => ['name' => 'Airtel', 'airtime_margin' => 2.2, 'data_margin' => 2.8, 'status' => 'active'],
                '9mobile' => ['name' => '9mobile', 'airtime_margin' => 1.8, 'data_margin' => 2.3, 'status' => 'active'],
            ],
            'settings' => [
                'default_airtime_margin' => 2.0,
                'default_data_margin' => 2.5,
                'minimum_margin' => 1.0,
                'maximum_margin' => 10.0,
            ]
        ];

        return view('admin.airtime.margins', compact('margins'));
    }

    /**
     * Resend failed transaction.
     */
    public function resendTransaction(Request $request, Transaction $transaction)
    {
        if ($transaction->status !== 'failed' || !in_array($transaction->category, ['airtime', 'data'])) {
            return back()->withErrors(['error' => 'Invalid transaction for resend.']);
        }

        // In a real implementation, this would retry the transaction with the provider
        $transaction->update([
            'status' => 'pending',
            'metadata' => array_merge(
                json_decode($transaction->metadata, true) ?? [],
                ['resent_at' => now(), 'resent_by' => auth('admin')->id()]
            ),
        ]);

        // Log admin action
        \App\Models\AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'resend_transaction',
            'target_type' => 'Transaction',
            'target_id' => $transaction->id,
            'data' => ['category' => $transaction->category, 'amount' => $transaction->amount],
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', 'Transaction has been queued for retry.');
    }
}