<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Transaction;
use App\Models\KycVerification;
use App\Models\VirtualCard;
use App\Models\Wallet;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Show the admin dashboard.
     */
    public function index()
    {
        $admin = auth('admin')->user();
        
        // Overview statistics
        $stats = [
            'total_users' => User::count(),
            'active_users' => User::where('is_active', true)->count(),
            'pending_kyc' => KycVerification::where('status', 'pending')->count(),
            'approved_kyc' => KycVerification::where('status', 'approved')->count(),
            'total_transactions' => Transaction::count(),
            'total_transaction_volume' => Transaction::where('status', 'completed')->sum('amount'),
            'total_wallets' => Wallet::count(),
            'active_cards' => VirtualCard::where('status', 'active')->count(),
            'frozen_wallets' => Wallet::where('is_frozen', true)->count(),
            'blocked_users' => User::where('is_blocked', true)->count(),
        ];

        // Recent activities
        $recentUsers = User::latest()->limit(5)->get();
        $recentTransactions = Transaction::with('user')->latest()->limit(10)->get();
        $pendingKyc = KycVerification::with('user')->where('status', 'pending')->latest()->limit(5)->get();

        // Chart data for the last 30 days
        $chartData = $this->getChartData();

        return view('admin.dashboard', compact('admin', 'stats', 'recentUsers', 'recentTransactions', 'pendingKyc', 'chartData'));
    }

    /**
     * Get chart data for dashboard graphs.
     */
    private function getChartData()
    {
        $last30Days = collect(range(29, 0))->map(function ($days) {
            return Carbon::now()->subDays($days)->format('Y-m-d');
        });

        // User registrations over last 30 days
        $userRegistrations = User::selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->where('created_at', '>=', Carbon::now()->subDays(30))
            ->groupBy('date')
            ->pluck('count', 'date')
            ->toArray();

        // Transactions over last 30 days
        $transactionVolume = Transaction::selectRaw('DATE(created_at) as date, SUM(amount) as total, COUNT(*) as count')
            ->where('created_at', '>=', Carbon::now()->subDays(30))
            ->where('status', 'completed')
            ->groupBy('date')
            ->get()
            ->keyBy('date');

        // KYC submissions over last 30 days
        $kycSubmissions = KycVerification::selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->where('created_at', '>=', Carbon::now()->subDays(30))
            ->groupBy('date')
            ->pluck('count', 'date')
            ->toArray();

        return [
            'dates' => $last30Days->values(),
            'user_registrations' => $last30Days->map(fn($date) => $userRegistrations[$date] ?? 0)->values(),
            'transaction_volume' => $last30Days->map(fn($date) => $transactionVolume->get($date)?->total ?? 0)->values(),
            'transaction_count' => $last30Days->map(fn($date) => $transactionVolume->get($date)?->count ?? 0)->values(),
            'kyc_submissions' => $last30Days->map(fn($date) => $kycSubmissions[$date] ?? 0)->values(),
        ];
    }

    /**
     * Get system alerts and notifications for admin.
     */
    public function getAlerts()
    {
        $alerts = [];

        // Check for pending KYC verifications
        $pendingKycCount = KycVerification::where('status', 'pending')->count();
        if ($pendingKycCount > 0) {
            $alerts[] = [
                'type' => 'warning',
                'title' => 'Pending KYC Verifications',
                'message' => "{$pendingKycCount} KYC verification(s) require your attention.",
                'action_url' => route('admin.kyc.index', ['status' => 'pending']),
                'action_text' => 'Review KYC'
            ];
        }

        // Check for high-value pending transactions
        $highValueTransactions = Transaction::where('status', 'pending')
            ->where('amount', '>', 100000)
            ->count();
        if ($highValueTransactions > 0) {
            $alerts[] = [
                'type' => 'danger',
                'title' => 'High-Value Pending Transactions',
                'message' => "{$highValueTransactions} high-value transaction(s) pending approval.",
                'action_url' => route('admin.transactions.index', ['status' => 'pending', 'min_amount' => 100000]),
                'action_text' => 'Review Transactions'
            ];
        }

        // Check for blocked users
        $blockedUsersCount = User::where('is_blocked', true)->count();
        if ($blockedUsersCount > 0) {
            $alerts[] = [
                'type' => 'info',
                'title' => 'Blocked Users',
                'message' => "{$blockedUsersCount} user(s) are currently blocked.",
                'action_url' => route('admin.users.index', ['status' => 'blocked']),
                'action_text' => 'Manage Users'
            ];
        }

        // Check for frozen wallets
        $frozenWalletsCount = Wallet::where('is_frozen', true)->count();
        if ($frozenWalletsCount > 0) {
            $alerts[] = [
                'type' => 'warning',
                'title' => 'Frozen Wallets',
                'message' => "{$frozenWalletsCount} wallet(s) are currently frozen.",
                'action_url' => route('admin.wallets.index', ['status' => 'frozen']),
                'action_text' => 'Review Wallets'
            ];
        }

        return response()->json($alerts);
    }
}