<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VirtualCard;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class VirtualCardController extends Controller
{
    /**
     * Display a listing of virtual cards.
     */
    public function index(Request $request)
    {
        $query = VirtualCard::with(['user', 'wallet']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            })->orWhere('card_number', 'like', "%{$search}%");
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $cards = $query->latest()->paginate(20);

        // Statistics
        $stats = [
            'total_cards' => VirtualCard::count(),
            'active_cards' => VirtualCard::where('status', 'active')->count(),
            'frozen_cards' => VirtualCard::where('status', 'frozen')->count(),
            'terminated_cards' => VirtualCard::where('status', 'terminated')->count(),
            'total_spending' => 0, // Will implement when virtual card transactions are linked
        ];

        return view('admin.virtual-cards.index', compact('cards', 'stats'));
    }

    /**
     * Display the specified virtual card.
     */
    public function show(VirtualCard $virtualCard)
    {
        $virtualCard->load(['user', 'wallet', 'transactions' => function($query) {
            $query->latest()->limit(20);
        }]);

        return view('admin.virtual-cards.show', compact('virtualCard'));
    }

    /**
     * Activate a virtual card.
     */
    public function activate(VirtualCard $virtualCard)
    {
        if ($virtualCard->status === 'terminated') {
            return back()->withErrors(['error' => 'Cannot activate a terminated card.']);
        }

        $virtualCard->update([
            'status' => 'active',
            'activated_at' => now(),
        ]);

        return back()->with('success', 'Virtual card activated successfully.');
    }

    /**
     * Deactivate a virtual card.
     */
    public function deactivate(VirtualCard $virtualCard)
    {
        $virtualCard->update([
            'status' => 'inactive',
        ]);

        return back()->with('success', 'Virtual card deactivated successfully.');
    }

    /**
     * Freeze a virtual card.
     */
    public function freeze(VirtualCard $virtualCard)
    {
        if ($virtualCard->status === 'terminated') {
            return back()->withErrors(['error' => 'Cannot freeze a terminated card.']);
        }

        $virtualCard->update([
            'status' => 'frozen',
            'frozen_at' => now(),
        ]);

        return back()->with('success', 'Virtual card frozen successfully.');
    }

    /**
     * Unfreeze a virtual card.
     */
    public function unfreeze(VirtualCard $virtualCard)
    {
        $virtualCard->update([
            'status' => 'active',
            'frozen_at' => null,
        ]);

        return back()->with('success', 'Virtual card unfrozen successfully.');
    }

    /**
     * Terminate a virtual card.
     */
    public function terminate(VirtualCard $virtualCard)
    {
        $virtualCard->update([
            'status' => 'terminated',
            'terminated_at' => now(),
        ]);

        return back()->with('success', 'Virtual card terminated successfully.');
    }

    /**
     * Reissue a virtual card.
     */
    public function reissue(VirtualCard $virtualCard)
    {
        if ($virtualCard->status !== 'active' && $virtualCard->status !== 'frozen') {
            return back()->withErrors(['error' => 'Can only reissue active or frozen cards.']);
        }

        // Generate new card details
        $newCardNumber = $this->generateCardNumber();
        $newCvv = $this->generateCvv();
        $newExpiryDate = now()->addYears(3);

        $virtualCard->update([
            'card_number' => $newCardNumber,
            'cvv' => $newCvv,
            'expiry_date' => $newExpiryDate,
            'reissued_at' => now(),
        ]);

        return back()->with('success', 'Virtual card reissued successfully with new details.');
    }

    /**
     * Update card limits.
     */
    public function updateLimits(Request $request, VirtualCard $virtualCard)
    {
        $request->validate([
            'daily_limit' => 'required|numeric|min:0|max:1000000',
            'monthly_limit' => 'required|numeric|min:0|max:10000000',
            'per_transaction_limit' => 'required|numeric|min:0|max:500000',
        ]);

        $virtualCard->update([
            'daily_limit' => $request->daily_limit,
            'monthly_limit' => $request->monthly_limit,
            'per_transaction_limit' => $request->per_transaction_limit,
        ]);

        return back()->with('success', 'Card limits updated successfully.');
    }

    /**
     * Update card fees.
     */
    public function updateFees(Request $request, VirtualCard $virtualCard)
    {
        $request->validate([
            'creation_fee' => 'required|numeric|min:0|max:10000',
            'maintenance_fee' => 'required|numeric|min:0|max:1000',
            'transaction_fee' => 'required|numeric|min:0|max:1000',
        ]);

        $virtualCard->update([
            'creation_fee' => $request->creation_fee,
            'maintenance_fee' => $request->maintenance_fee,
            'transaction_fee' => $request->transaction_fee,
        ]);

        return back()->with('success', 'Card fees updated successfully.');
    }

    /**
     * Get card analytics.
     */
    public function analytics()
    {
        $analytics = [
            'cards_by_status' => VirtualCard::selectRaw('status, COUNT(*) as count')
                                          ->groupBy('status')
                                          ->pluck('count', 'status'),
            'cards_by_type' => VirtualCard::selectRaw('type, COUNT(*) as count')
                                        ->groupBy('type')
                                        ->pluck('count', 'type'),
            'monthly_creations' => VirtualCard::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month, COUNT(*) as count')
                                            ->where('created_at', '>=', now()->subMonths(12))
                                            ->groupBy('month')
                                            ->orderBy('month')
                                            ->pluck('count', 'month'),
            'transaction_volume' => VirtualCard::join('transactions', 'virtual_cards.id', '=', 'transactions.virtual_card_id')
                                             ->selectRaw('DATE_FORMAT(transactions.created_at, "%Y-%m") as month, SUM(transactions.amount) as volume')
                                             ->where('transactions.created_at', '>=', now()->subMonths(12))
                                             ->where('transactions.status', 'completed')
                                             ->groupBy('month')
                                             ->orderBy('month')
                                             ->pluck('volume', 'month'),
        ];

        return view('admin.virtual-cards.analytics', compact('analytics'));
    }

    /**
     * Generate a new card number.
     */
    private function generateCardNumber()
    {
        do {
            $cardNumber = '5399' . str_pad(mt_rand(0, 999999999999), 12, '0', STR_PAD_LEFT);
        } while (VirtualCard::where('card_number', $cardNumber)->exists());

        return $cardNumber;
    }

    /**
     * Generate a new CVV.
     */
    private function generateCvv()
    {
        return str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
    }
}